//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Dec 8, 2007
//---------------------

package org.cleversafe.control;

import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;

import org.apache.log4j.Logger;

public class FixedPunishmentSelector implements ParameterSelector
{
   private static Logger _logger = Logger.getLogger(FixedPunishmentSelector.class);

   private volatile boolean initialized = false;

   // Adaptive state selector
   private ParameterState parameterState;

   private int minimalSingleAttemptCount = 20;
   private double errorPanishmentCoef = 1.;

   private ParameterRuntime state;
   private int successAttemptCount;

   public static class ParameterRuntime
   {
      private int threshold; // threshold to overcome to advance state     
      private int cumulativeErrorNumber; // # of errors encountered in a single state

      public ParameterRuntime(int threshold)
      {
         this.cumulativeErrorNumber = 0;
         this.threshold = threshold;
      }

      /**
       * @return the errors
       */
      public int getCumulativeErrors()
      {
         return cumulativeErrorNumber;
      }

      /**
       * @return the threshold
       */
      public int getThreshold()
      {
         return threshold;
      }

      /* (non-Javadoc)
       * @see java.lang.Object#toString()
       */
      @Override
      public String toString()
      {
         return "[" + this.threshold + "," + this.cumulativeErrorNumber + "]";
      }

   }

   Map<ParameterState, ParameterRuntime> parameterStates = new TreeMap<ParameterState, ParameterRuntime>();

   public FixedPunishmentSelector()
   {
   }

   public FixedPunishmentSelector(
         ParameterState initialState,
         int minimalSingleAttemptCount,
         double errorPanishment)
   {
      this.minimalSingleAttemptCount = minimalSingleAttemptCount;
      this.errorPanishmentCoef = errorPanishment;
      this.parameterState = initialState;
      initialize();
   }

   protected synchronized void initialize()
   {
      if (this.initialized == true)
      {
         throw new IllegalStateException("Second attempt to initialize selector");
      }
      if (parameterState == null)
      {
         throw new IllegalStateException("An initial parameter value was not set for selector");
      }
      this.successAttemptCount = 0;

      this.state = new ParameterRuntime(minimalSingleAttemptCount);
      parameterStates.put(this.parameterState, state);

      this.initialized = true;
   }

   /**
    * Returns current parameter
    */
   public synchronized ParameterState getState()
   {
      return this.parameterState;
   }

   /**
    * This method is needed by framework, can be only called before object was full initialized
    * @param state
    */
   public synchronized void setState(ParameterState state)
   {
      if (this.initialized == true)
      {
         throw new IllegalStateException("Can't change selectors state after initialization");
      }
      this.parameterState = state;
   }

   /**
    * After success a parameter advanced after threshold number of successes is observed
    * The successful state threshold is set to minimal value (this.minimalSingleAttemptCount)
    * The threshold of a new state is  <b>increased</b> by minimal value (this.minimalSingleAttemptCount)
    * in other words a parameter that produces errors sets a higher threshold thus making
    * the transition to a more error prone parameter less likely
    * 
    * @see org.cleversafe.control.ParameterSelector#onSuccess
    */
   public synchronized void onSuccess()
   {
      this.state.threshold--;
      assert (state.threshold >= 0);
      // Over threshold?
      if (state.threshold == 0)
      {
         if (_logger.isDebugEnabled())
            _logger.trace("Advancing from state " + this.parameterState);

         this.parameterState = this.parameterState.advance();
         ParameterRuntime newState = parameterStates.get(this.parameterState);
         if (newState == null)
         {
            newState = new ParameterRuntime(this.minimalSingleAttemptCount);
            parameterStates.put(this.parameterState, newState);
         }
         else
         {
            newState.threshold += this.minimalSingleAttemptCount;
         }
         if (_logger.isDebugEnabled())
            _logger.trace("Threshold for a new state " + this.parameterState + " is "
                  + newState.threshold);

         this.state = newState;
         this.successAttemptCount = 0;
      }
      else
      // Stay in the same state and keep success attempts counting
      {
         this.successAttemptCount++;
      }
   }

   /**
    * An immediate transition to a less "dangerous" state is made.
    * The previous state threshold is set as a product of a given coefficient (this.errorPanishmentCoef)
    * and a measure of how "well the current state behaved" calculated in as a ratio
    * of total remaining to finish versus successful. If threshold was set to 100 and error was detected
    * after 50 attempts the ration will be (100-50)/50 --> 1
    * @see org.cleversafe.control.ParameterSelector#onFailure
    */
   public synchronized void onFailure()
   {

      if (_logger.isDebugEnabled())
         _logger.debug("Failure detected, reversing from state " + this.parameterState);
      this.state.cumulativeErrorNumber++;

      // Punish yourself for an error
      // The more successful attempts the lesser punishment
      double previousStateCoef = (double) this.state.threshold / (this.successAttemptCount + 1);
      this.state.threshold +=
            this.minimalSingleAttemptCount * (this.errorPanishmentCoef * previousStateCoef +1);

      if (_logger.isDebugEnabled())
         _logger.trace("Next threshold set to " + this.state.threshold + " based on ("
               + previousStateCoef + " * " + this.errorPanishmentCoef + " * "
               + minimalSingleAttemptCount + ")");

      // Goto the previous state
      this.parameterState = this.parameterState.reverse();
      ParameterRuntime newState = parameterStates.get(this.parameterState);
      if (newState == null)
      {
         newState = new ParameterRuntime(this.state.threshold);
         parameterStates.put(this.parameterState, newState);
      }
      else
      {
         newState.threshold += this.state.threshold;
      }
      this.state = newState;
      this.successAttemptCount = 0;
   }

   /**
    * Should not change values
    * Don't want to create a deep copy, too expensive
    * @return ummodifiable map
    */

   public Map<ParameterState, ParameterRuntime> getCurrentResults()
   {
      return Collections.unmodifiableMap(parameterStates);
   }

   /**
    * @return the errorPanishmentCoef
    */
   public synchronized double getErrorPunishment()
   {
      return errorPanishmentCoef;
   }

   /**
    * @param errorPanishmentCoef the errorPanishmentCoef to set
    */
   public synchronized void setErrorPanishment(double errorPanishmentCoef)
   {
      this.errorPanishmentCoef = errorPanishmentCoef;
   }

   /**
    * @return the minimalSingleAttemptCount
    */
   public synchronized int getMinimalSingleAttemptCount()
   {
      return minimalSingleAttemptCount;
   }

   /**
    * @param minimalSingleAttemptCount the minimalSingleAttemptCount to set
    */
   public synchronized void setMinimalSingleAttemptCount(int minimalSingleAttemptCount)
   {
      this.minimalSingleAttemptCount = minimalSingleAttemptCount;
   }

   /**
    * 
    */
   public String toString()
   {
      return "Parameter:" + this.parameterState + " punishmentCoef=" + this.errorPanishmentCoef
            + this.parameterStates;

   }
}
