//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 31, 2007
//---------------------

package org.cleversafe.config.evaluator;

import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;

import org.apache.log4j.Logger;
import org.cleversafe.config.BindingsProvider;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.config.exceptions.ObjectValidationException;
import org.cleversafe.util.NamingHelper;
import org.cleversafe.util.TypeUtils;
import org.cleversafe.util.Version;

// TODO: Describe class or interface
public class XMLValidator
{
   private static Logger _logger = Logger.getLogger(XMLValidator.class);

   private static BindingsProvider bindingsProvider = null;

   static
   {
      // Load configuration
      try
      {
         bindingsProvider =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);
      }
      catch (final ConfigurationLoadException cle)
      {
         throw new RuntimeException("Could not load system bindings configuration", cle);
      }
      assert XMLValidator.bindingsProvider != null;
   }

   // Annotation for a setter method that must be called for full object initialization.
   @Retention(RUNTIME)
   public @interface RequiredForInitialization {
   }

   private static Class<?> getLiteralClass(Literal literal)
   {
      SupportedTypes.Enum literalType = literal.getType();

      Class<?> retVal = null;
      switch (literalType.intValue())
      {
         case SupportedTypes.Enum.INT_BOOLEAN :
            retVal = boolean.class;
            break;
         case SupportedTypes.Enum.INT_BYTE :
            retVal = byte.class;
            break;
         case SupportedTypes.Enum.INT_DOUBLE :
            retVal = double.class;
            break;
         case SupportedTypes.Enum.INT_FLOAT :
            retVal = float.class;
            break;
         case SupportedTypes.Enum.INT_INT :
            retVal = int.class;
            break;
         case SupportedTypes.Enum.INT_LONG :
            retVal = long.class;
            break;
         case SupportedTypes.Enum.INT_SHORT :
            retVal = short.class;
            break;
         case SupportedTypes.Enum.INT_STRING :
            retVal = String.class;
            break;
      }

      return retVal;
   }

   /**
    * Validates that the array of parameters passed are sufficient to successfully initialize an
    * object of the given class.
    * 
    * @param clazz
    * @param params
    * @throws ObjectValidationException
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    */
   public static void validateClass(Class<?> clazz, Parameter[] params, ExecutionContext ctx)
         throws ObjectValidationException, ConfigurationItemNotDefinedException,
         IllegalConfigurationContentException
   {
      Set<Method> methods = new HashSet<Method>();
      Set<String> ignoreList = new HashSet<String>();

      for (Parameter param : params)
      {
         String methodName = NamingHelper.convertNameIntoJavaSetter(param.getName());

         Class<?> parameterType = null;
         if (param.isSetLiteral())
         {
            parameterType = getLiteralClass(param.getLiteral());
         }
         else if (param.isSetReference())
         {
            Reference reference = param.getReference();
            if (reference.isSetInterface())
            {
               try
               {
                  if (reference.isSetReferral())
                  {
                     parameterType =
                           bindingsProvider.getImplementationClass(
                                 Class.forName(reference.getInterface()), reference.getReferral());
                  }
                  else
                  {
                     parameterType =
                           bindingsProvider.getDefaultImplementationClass(Class.forName(reference.getInterface()));
                  }
               }
               catch (ClassNotFoundException cnfe)
               {
                  throw new IllegalConfigurationContentException("Could not find interface class "
                        + reference.getInterface(), cnfe);
               }
            }
            else if (reference.isSetKind())
            {
               if (reference.isSetReferral())
               {
                  parameterType =
                        bindingsProvider.getImplementationClass(reference.getKind().split("\\."),
                              reference.getReferral());
               }
               else
               {
                  parameterType =
                        bindingsProvider.getDefaultImplementationClass(reference.getKind().split(
                              "\\."));
               }
            }
            else if (reference.isSetObject())
            {
               if (ctx == null)
               {
                  ignoreList.add(methodName);
               }
               else if (!ctx.contains(reference.getObject()))
               {
                  throw new IllegalConfigurationContentException(
                        "Could not find object in ExecutionContext: " + reference.getObject());
               }
               else
               {
                  parameterType = ctx.get(reference.getObject()).getClass();
               }
            }
         }
         else if (param.isSetMethod())
         {
            if (ctx == null)
            {
               ignoreList.add(methodName);
            }
            else if (!ctx.contains(param.getMethod().getObject()))
            {
               throw new IllegalConfigurationContentException(
                     "Could not find object in ExecutionContext: " + param.getMethod().getObject());
            }
            else
            {
               // Assume method without arguments
               Method m =
                     TypeUtils.getMatchingMethod(ctx.get(param.getMethod().getObject()).getClass(),
                           param.getMethod().getName(), new Class<?>[]{}, false);

               if (m != null)
               {
                  parameterType = m.getReturnType();
               }
               else
               {
                  // Method possibly requires argument(s) or was not found for some other reason.
                  // Recursive validation may be required. Put
                  // method name in ignore list.
                  ignoreList.add(methodName);
               }
            }
         }

         if (parameterType != null)
         {
            Method method = TypeUtils.getMatchingMethod(clazz, methodName, new Class<?>[]{
               parameterType
            }, false);

            methods.add(method);
         }
      }

      Method[] classMethods = clazz.getMethods();
      for (Method method : classMethods)
      {
         if (method.isAnnotationPresent(RequiredForInitialization.class))
         {
            if (!methods.contains(method) && !ignoreList.contains(method.getName()))
            {
               throw new ObjectValidationException("Parameter for method " + method.getName()
                     + "() in class " + clazz.getName() + " is not declared.");
            }
         }
      }
   }

   /**
    * Validates that the array of parameters passed are sufficient to successfully initialize an
    * object of the default implementation of the given interface.
    * 
    * @param interfaceClassName
    * @param isReferralDefined
    * @param referral
    * @param params
    * @throws ConfigurationItemNotDefinedException
    * @throws IllegalConfigurationContentException
    * @throws ObjectValidationException
    */
   public static void validateInterface(
         String interfaceClassName,
         boolean isReferralDefined,
         String referral,
         Parameter[] params,
         ExecutionContext ctx) throws ConfigurationItemNotDefinedException,
         IllegalConfigurationContentException, ObjectValidationException
   {
      Class<?> implementationClass = null;

      try
      {
         if (isReferralDefined)
         {
            implementationClass =
                  bindingsProvider.getImplementationClass(Class.forName(interfaceClassName),
                        referral);
         }
         else
         {
            implementationClass =
                  bindingsProvider.getDefaultImplementationClass(Class.forName(interfaceClassName));

         }
      }
      catch (ClassNotFoundException e)
      {
         throw new IllegalConfigurationContentException("Class " + interfaceClassName
               + " was not found!", e);
      }

      validateClass(implementationClass, params, ctx);
   }

   /**
    * Validates that the array of parameters passed are sufficient to successfully initialize an
    * object of the given kind and referral.
    * 
    * @param kind
    * @param isReferralDefined
    * @param referral
    * @param params
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    * @throws IllegalConfigurationContentException
    * @throws ClassNotFoundException
    */
   public static void validateKind(
         String[] kind,
         boolean isReferralDefined,
         String referral,
         Parameter[] params,
         ExecutionContext ctx) throws ConfigurationItemNotDefinedException,
         ObjectValidationException, IllegalConfigurationContentException
   {
      Class<?> kindClass = null;

      if (isReferralDefined)
         kindClass = bindingsProvider.getImplementationClass(kind, referral);
      else
         kindClass = bindingsProvider.getDefaultImplementationClass(kind);
      validateClass(kindClass, params, ctx);
   }

   /**
    * 
    * @param referenceArray
    * @param ctx
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    */
   public static void validateReferenceArray(Reference[] referenceArray, ExecutionContext ctx)
         throws IllegalConfigurationContentException, ConfigurationItemNotDefinedException,
         ObjectValidationException
   {
      for (Reference reference : referenceArray)
      {
         validateReference(reference, ctx);
      }
   }

   /**
    * Validates a given reference to ensure that all necessary parameters are set.
    * 
    * @param reference
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    */
   public static void validateReference(Reference reference, ExecutionContext ctx)
         throws IllegalConfigurationContentException, ConfigurationItemNotDefinedException,
         ObjectValidationException
   {
      if (_logger.isTraceEnabled())
         _logger.trace("Validating object reference " + reference.xmlText());

      if (reference.isSetVersion())
      {
         if (!reference.isSetKind() && !reference.isSetInterface())
         {
            throw new ObjectValidationException(
                  "Version can only be set for a kind or an interface reference");
         }

         try
         {
            Version.fromString(reference.getVersion());
         }
         catch (IllegalArgumentException e)
         {
            throw new ObjectValidationException("Illegal version format: " + reference.getVersion());
         }
      }

      if (reference.isSetInterface())
      {
         if (reference.isSetKind() || reference.isSetObject())
         {
            throw new ObjectValidationException(
                  "Reference node cannot be of more than one type (choose one of: interface, kind or object)");
         }
         validateInterface(reference.getInterface(), reference.isSetReferral(),
               reference.getReferral(), reference.getParamArray(), ctx);
      }
      else if (reference.isSetKind())
      {
         if (reference.isSetInterface() || reference.isSetObject())
         {
            throw new ObjectValidationException(
                  "Reference node cannot be of more than one type (choose one of: interface, kind or object)");
         }

         validateKind(reference.getKind().split("\\."), reference.isSetReferral(),
               reference.getReferral(), reference.getParamArray(), ctx);
      }
      else if (reference.isSetObject())
      {
         if (reference.isSetInterface() || reference.isSetKind())
         {
            throw new ObjectValidationException(
                  "Reference node cannot be of more than one type (choose one of: interface, kind or object)");
         }

         if (ctx != null)
         {
            if (ctx.get(reference.getObject()) == null)
            {
               throw new IllegalConfigurationContentException(
                     "Could not find object in ExecutionContext: " + reference.getObject());
            }

            validateClass(ctx.get(reference.getObject()).getClass(), reference.getParamArray(), ctx);
         }
      }
      else
      {
         throw new ObjectValidationException(
               "Reference node must be one of: interface, kind or objects");
      }
   }
}
