//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: Jun 18, 2007
//---------------------

package org.cleversafe.config.evaluator;

import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.reflect.Method;
import java.util.List;

import org.cleversafe.config.exceptions.ObjectInitializationException;

import org.cleversafe.util.NamingHelper;
import org.cleversafe.util.TypeUtils;

// Evaluator that represents a reference to an object
public class ReferenceEvaluator implements Evaluatable
{
   Object referenced = null;
   List<NamedEvaluator> params = null;
   private static String DEFAULT_INITIALIZE_METHOD = "initialize";

   private boolean initialized = false;

   @Retention(RUNTIME)
   public @interface GridLoadInitMethod {
   }

   /**
    * @param referenced
    */
   public ReferenceEvaluator(Object referenced)
   {
      super();
      this.referenced = referenced;
   }

   /**
    * 
    * @param referenced
    * @param params
    */
   public ReferenceEvaluator(Object referenced, List<NamedEvaluator> params)
   {
      super();
      this.referenced = referenced;
      this.params = params;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.evaluator.Evaluatable#evaluate()
    */
   public Object evaluate() throws ObjectInitializationException
   {
      if (!this.initialized)
      {
         if (this.params != null)
         {
            setParameters();
         }

         callInitializeMethod();
         this.initialized = true;
      }
      return this.referenced;
   }

   public Object evaluateWithoutInitialization() throws ObjectInitializationException
   {
      if (this.params != null)
      {
         setParameters();
      }
      return this.referenced;
   }

   private void callInitializeMethod() throws ObjectInitializationException
   {
      // If an object has a method initialize with no arguments, call it in
      // order to complete object instantiation process
      Method initMethod = null;
      String initializeMethod = DEFAULT_INITIALIZE_METHOD;
      try
      {
         // Check if a class has a Annotation for a method to be initializer

         for (final Method method : this.referenced.getClass().getMethods())
         {
            if (method.isAnnotationPresent(GridLoadInitMethod.class))
            {
               if (method.getParameterTypes().length == 0)
               { // Should not have arguments
                  initializeMethod = method.getName();
                  break;
               }
               else
               {
                  throw new ObjectInitializationException("Method "
                        + this.referenced.getClass().getName() + "." + method.getName()
                        + "() should not have any arguments to be used for initialization");
               }
            }
         }

         try
         {
            initMethod = this.referenced.getClass().getMethod(initializeMethod);
         }
         catch (NoSuchMethodException nsme)
         {
            // Ignore
         }

         // Init method would be null if no annotation is found on any method
         // and the default initialize method is also not found. In this case,
         // just skip.
         if (initMethod != null)
         {
            initMethod.invoke(this.referenced);
         }
      }
      catch (Exception e)
      {
         throw new ObjectInitializationException("Failed to initialize an object for "
               + this.referenced.getClass().getName(), e);
      }
   }

   private void setParameters() throws ObjectInitializationException
   {
      for (NamedEvaluator param : this.params)
      {
         if (param.getName().length() == 0)
         {
            throw new IllegalArgumentException("Empty parameter name for "
                  + this.referenced.getClass().getName());
         }

         // Evaluate the parameter and set it on the referenced object
         String methodName = "unknown";
         Object paramObj = null;
         try
         {
            methodName = NamingHelper.convertNameIntoJavaSetter(param.getName());

            paramObj = param.evaluate();

            if (paramObj == null)
            {
               throw new ObjectInitializationException("Parameter " + param.getName()
                     + " evaulated to null!");
            }

            Class<?> paramType = paramObj.getClass();

            Method setupMethod =
                  TypeUtils.getMatchingMethod(this.referenced.getClass(), methodName, new Class[]{
                     paramType
                  }, false);

            if (setupMethod == null)
            {
               throw new ObjectInitializationException("Method " + methodName + "("
                     + paramType.getName() + ") not found");
            }

            setupMethod.invoke(this.referenced, paramObj);
         }

         catch (final Exception e)
         {
            throw new ObjectInitializationException(
                  "Failed to set parameters for an object of type "
                        + this.referenced.getClass().getName() + " invoking method " + methodName
                        + "(" + (paramObj != null ? paramObj : "undefined") + ")", e);
         }
      }
   }
}
