//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Feb 6, 2008
//---------------------

package org.cleversafe.config;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlError;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;
import org.cleversafe.config.bindings.Binding;
import org.cleversafe.config.bindings.BindingsDocument;
import org.cleversafe.config.bindings.Implementation;
import org.cleversafe.config.evaluator.SimpleParameter;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.config.exceptions.IllegalConfigurationFormatException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.util.Version;

// TODO: Describe class or interface
public class XMLBindingsProvider extends BindingsProviderBase
{
   private static Logger _logger = Logger.getLogger(XMLBindingsProvider.class);

   private final Map<List<String>, org.cleversafe.config.Binding> bindingsMap =
         new HashMap<List<String>, org.cleversafe.config.Binding>();

   /**
    * Default constructor.
    */
   public XMLBindingsProvider()
   {
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#loadBindings(java.io.InputStream)
    */
   public void loadBindings(final InputStream xmlInputStream) throws XmlException, IOException,
         IllegalConfigurationContentException, IllegalConfigurationFormatException
   {
      final BindingsDocument bindingsDocument = BindingsDocument.Factory.parse(xmlInputStream);

      final XmlOptions validateOptions = new XmlOptions();
      final ArrayList<XmlError> errorList = new ArrayList<XmlError>();

      validateOptions.setErrorListener(errorList);

      if (!bindingsDocument.validate(validateOptions))
      {
         final StringBuffer errorBuffer = new StringBuffer();
         errorBuffer.append("Number of errors: " + errorList.size() + ". ");
         int i = 1;
         for (final XmlError xmlError : errorList)
         {
            errorBuffer.append("XML Error " + i++ + " - " + xmlError.getMessage() + " at location "
                  + xmlError.getCursorLocation().xmlText() + ";");
         }

         _logger.error("Could not validate configuration schema: " + errorBuffer.toString());
         throw new IllegalConfigurationFormatException("Could not validate configuration schema: "
               + errorBuffer.toString());
      }

      populateBindings(bindingsDocument);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#loadBindings(java.lang.String)
    */
   public void loadBindings(final String xmlFileName) throws IllegalConfigurationContentException,
         IllegalConfigurationFormatException, FileNotFoundException, XmlException, IOException
   {
      loadBindings(new FileInputStream(xmlFileName));
   }

   private void populateBindings(final BindingsDocument bindingsDocument)
         throws IllegalConfigurationContentException, IllegalConfigurationFormatException
   {
      _logger.debug("Loading bindings file");
      for (final Binding binding : bindingsDocument.getBindings().getBindingArray())
      {
         addBinding(new ArrayList<String>(), new ArrayList<Class<?>>(), binding, true);
      }
   }

   private void addBinding(
         final List<String> parentQualifiedBindingName,
         final List<Class<?>> parentInterfaceClasses,
         final Binding xmlBinding,
         boolean isParentAKind) throws IllegalConfigurationContentException,
         IllegalConfigurationFormatException
   {
      if (!xmlBinding.isSetKind() && !xmlBinding.isSetRequiresInterface())
      {
         _logger.error("All bindings must have either 'kind' or 'requiresInterface' attribute set");
         throw new IllegalConfigurationFormatException(
               "All bindings must have either 'kind' or 'requiresInterface' attribute set");
      }

      // Create the binding name
      final List<String> qualifiedBindingName = new ArrayList<String>(parentQualifiedBindingName);

      // Kind
      if (xmlBinding.isSetKind())
      {
         if (!isParentAKind)
         {
            _logger.error("Heirarchical sub-kinds can only be defined for bindings that have the 'kind' attribute defined");
            throw new IllegalConfigurationFormatException(
                  "Heirarchical sub-kinds can only be defined for bindings that have the 'kind' attribute defined");
         }
         qualifiedBindingName.add(xmlBinding.getKind());
      }

      // Interface
      else
      {
         // For debugging and exception outputs
         qualifiedBindingName.add(xmlBinding.getRequiresInterface());
      }

      // Now that we have the full qualifiedBinding name, load the Binding object...
      org.cleversafe.config.Binding binding = this.bindingsMap.get(qualifiedBindingName);

      // ...or create a new one if it doesn't exist.
      if (binding == null)
      {
         binding = new org.cleversafe.config.Binding(qualifiedBindingName);
         this.bindingsMap.put(qualifiedBindingName, binding);
      }

      // Get the required interface class, if any
      List<Class<?>> interfaceClasses = parentInterfaceClasses;
      if (xmlBinding.isSetRequiresInterface())
      {
         String interfaceClassName = xmlBinding.getRequiresInterface();
         Class<?> interfaceClass = null;
         try
         {
            interfaceClass = Class.forName(interfaceClassName);
         }
         catch (final ClassNotFoundException cnfe)
         {
            _logger.error("Binding (" + qualifiedBindingName + ") has a required interface"
                  + " which was not found: " + interfaceClassName, cnfe);
            throw new IllegalConfigurationContentException("Binding (" + qualifiedBindingName
                  + ") has a required interface which was not found: " + interfaceClassName, cnfe);
         }
         assert interfaceClass != null;

         // Check if the parent class is assignable from the required interface class.
         if (parentInterfaceClasses.size() > 0)
         {
            final Class<?> parentClass =
                  parentInterfaceClasses.get(parentInterfaceClasses.size() - 1);
            if (!parentClass.isAssignableFrom(interfaceClass))
            {
               _logger.error("Binding (" + qualifiedBindingName + ") has a required interface ("
                     + interfaceClassName + ") that is not assignable to its parent: "
                     + parentClass.getName());
               throw new IllegalConfigurationContentException("Binding (" + qualifiedBindingName
                     + ") has a required interface (" + interfaceClassName
                     + ") that is not assignable to its parent: " + parentClass.getName());
            }
         }

         // Create a new list of required interface classes for the purpose
         // of reading sub-kinds (this method called recursively).
         interfaceClasses = new ArrayList<Class<?>>(parentInterfaceClasses);
         interfaceClasses.add(interfaceClass);

         binding.setRequiredInterface(interfaceClass);

         // Make the binding mappable from the interface string as well
         this.bindingsMap.put(Arrays.asList(new String[]{
            interfaceClassName
         }), binding);
      }

      // For each implementation of this binding
      for (final Implementation implementation : xmlBinding.getImplementationArray())
      {
         addImplementation(binding, xmlBinding, implementation, interfaceClasses);
      }

      for (final Binding subBinding : xmlBinding.getBindingArray())
      {
         if (subBinding.isSetKind())
         {
            binding.addSubKind(subBinding.getKind());
         }

         // recursively add bindings
         addBinding(qualifiedBindingName, interfaceClasses, subBinding, xmlBinding.isSetKind());
      }
   }

   private void addImplementation(
         org.cleversafe.config.Binding binding,
         Binding xmlBinding,
         Implementation implementation,
         List<Class<?>> interfaceClasses) throws IllegalConfigurationContentException,
         IllegalConfigurationFormatException
   {
      // NOTE: ".getClass1" is created by XMLBeans since ".getClass()" is reserved.
      final String implementationClassName = implementation.getClass1();

      // Get the parameters
      final SimpleParameter[] paramArray = implementation.getParamArray();

      // Create the implementation factory
      ImplementationFactoryBase iFactory = null;

      try
      {
         if (interfaceClasses.size() > 0)
         {
            iFactory =
                  new InterfaceImplementationFactory(interfaceClasses.get(
                        interfaceClasses.size() - 1).getName(), implementationClassName,
                        Arrays.asList(paramArray));
         }
         else
         {
            iFactory =
                  new ImplementationFactoryBase(implementationClassName, Arrays.asList(paramArray));
         }
      }
      catch (final ClassNotFoundException cnfe)
      {
         _logger.error("Binding (" + binding.getQualifiedBindingName()
               + ") has a class defined, which was not found: " + cnfe.getMessage(), cnfe);
         throw new IllegalConfigurationContentException("Binding ("
               + binding.getQualifiedBindingName() + ") has a class defined, which was not found: "
               + cnfe.getMessage(), cnfe);
      }
      catch (final ObjectInstantiationException oie)
      {
         _logger.error("One or more of the parameters defined for implementation "
               + implementationClassName + " of binding " + binding.getQualifiedBindingName()
               + " could not be instantiated correctly. Is the parameter type correct?", oie);
         throw new IllegalConfigurationContentException(
               "One or more of the parameters defined for implementation "
                     + implementationClassName + " of binding " + binding.getQualifiedBindingName()
                     + " could not be instantiated correctly. Is the parameter type correct?", oie);
      }

      // If the referral is set
      if (implementation.isSetReferral())
      {
         // Add the implementation factory as a referral
         final String referral = implementation.getReferral();

         if (implementation.isSetMinVersion() || implementation.isSetMaxVersion())
         {
            Version minVersion = org.cleversafe.config.Binding.MIN_VERSION;
            Version maxVersion = org.cleversafe.config.Binding.MAX_VERSION;

            try
            {
               if (implementation.isSetMinVersion())
                  minVersion = Version.fromString(implementation.getMinVersion());
            }
            catch (IllegalArgumentException e)
            {
               _logger.error("Error in binding " + binding.getQualifiedBindingName() + ": "
                     + e.getMessage(), e);
               throw new IllegalConfigurationContentException("Error in binding "
                     + binding.getQualifiedBindingName() + ": " + e.getMessage(), e);
            }

            try
            {
               if (implementation.isSetMaxVersion())
                  maxVersion = Version.fromString(implementation.getMaxVersion());
            }
            catch (IllegalArgumentException e)
            {
               _logger.error("Error in binding " + binding.getQualifiedBindingName() + ": "
                     + e.getMessage(), e);
               throw new IllegalConfigurationContentException("Error in binding "
                     + binding.getQualifiedBindingName() + ": " + e.getMessage(), e);
            }

            binding.addImplementationFactory(referral, iFactory, minVersion, maxVersion);
         }
         else
         {
            binding.addImplementationFactory(referral, iFactory);
         }
      }

      // If this implementation is the default implementation for a given binding
      if (implementation.isSetDefault() && implementation.getDefault())
      {
         binding.setDefaultImplementationFactory(iFactory);
      }

   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getDefaultImplementation(java.lang.Class)
    */
   @SuppressWarnings("unchecked")
   public <T> T getDefaultImplementation(Class<T> interfaceClass)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      List<String> bindingName = Arrays.asList(new String[]{
         interfaceClass.getName()
      });
      if (!this.bindingsMap.containsKey(bindingName))
      {
         throw new ConfigurationItemNotDefinedException(
               "No default implementation defined for interface class " + interfaceClass.getName());
      }

      InterfaceImplementationFactory iFactory =
            (InterfaceImplementationFactory) this.bindingsMap.get(bindingName).getDefaultImplementationFactory();

      return (T) iFactory.getImplementationObject();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getDefaultImplementation(java.lang.String[])
    */
   public Object getDefaultImplementation(String[] kind)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      // Kind cannot be null.
      assert kind != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = Arrays.asList(kind);

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException(
               "No default implementation defined for kind " + kindName);
      }

      ImplementationFactoryBase iFactory =
            this.bindingsMap.get(kindName).getDefaultImplementationFactory();
      return iFactory.getImplementationObject();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getDefaultImplementationClass(java.lang.Class)
    */
   public Class<?> getDefaultImplementationClass(Class<?> interfaceClass)
         throws ConfigurationItemNotDefinedException
   {
      List<String> bindingName = Arrays.asList(new String[]{
         interfaceClass.getName()
      });
      if (!this.bindingsMap.containsKey(bindingName))
      {
         throw new ConfigurationItemNotDefinedException(
               "No default implementation defined for interface class " + interfaceClass.getName());
      }

      InterfaceImplementationFactory iFactory =
            (InterfaceImplementationFactory) this.bindingsMap.get(bindingName).getDefaultImplementationFactory();

      return iFactory.getImplementationClass();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getDefaultImplementationClass(java.lang.String[])
    */
   public Class<?> getDefaultImplementationClass(String[] kind)
         throws ConfigurationItemNotDefinedException
   {
      // Kind cannot be null.
      assert kind != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = Arrays.asList(kind);

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException(
               "No default implementation defined for kind " + kindName);
      }

      ImplementationFactoryBase iFactory =
            this.bindingsMap.get(kindName).getDefaultImplementationFactory();
      return iFactory.getImplementationClass();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementation(java.lang.String[],
    *      java.lang.String)
    */
   public Object getImplementation(String[] kind, String referral)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      // Kind and referral cannot be null.
      assert kind != null;
      assert referral != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = Arrays.asList(kind);

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for kind " + kindName);
      }

      ImplementationFactoryBase iFactory =
            this.bindingsMap.get(kindName).getImplementationFactory(referral);

      return iFactory.getImplementationObject();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementation(java.lang.Class,
    *      java.lang.String)
    */
   @SuppressWarnings("unchecked")
   public <T> T getImplementation(Class<T> interfaceClass, String referral)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      List<String> bindingName = Arrays.asList(new String[]{
         interfaceClass.getName()
      });
      if (!this.bindingsMap.containsKey(bindingName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for interface "
               + interfaceClass.getName());
      }

      InterfaceImplementationFactory iFactory =
            (InterfaceImplementationFactory) this.bindingsMap.get(bindingName).getImplementationFactory(
                  referral);

      return (T) iFactory.getImplementationObject();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementationClass(java.lang.String[],
    *      java.lang.String)
    */
   public Class<?> getImplementationClass(String[] kind, String referral)
         throws ConfigurationItemNotDefinedException
   {
      // Kind and referral cannot be null.
      assert kind != null;
      assert referral != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = Arrays.asList(kind);

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for kind " + kindName);
      }

      ImplementationFactoryBase iFactory =
            this.bindingsMap.get(kindName).getImplementationFactory(referral);

      return iFactory.getImplementationClass();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementationClass(java.lang.Class,
    *      java.lang.String)
    */
   public Class<?> getImplementationClass(Class<?> interfaceClass, String referral)
         throws ConfigurationItemNotDefinedException
   {
      List<String> bindingName = Arrays.asList(new String[]{
         interfaceClass.getName()
      });
      if (!this.bindingsMap.containsKey(bindingName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for interface "
               + interfaceClass.getName());
      }

      InterfaceImplementationFactory iFactory =
            (InterfaceImplementationFactory) this.bindingsMap.get(bindingName).getImplementationFactory(
                  referral);

      return iFactory.getImplementationClass();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementationClassSet(java.lang.String[])
    */
   public Set<Class<?>> getImplementationClassSet(String[] kind)
         throws ConfigurationItemNotDefinedException
   {
      // Kind cannot be null.
      assert kind != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = Arrays.asList(kind);

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for kind " + kindName);
      }

      return this.bindingsMap.get(kindName).getImplementationClassSet();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getRequiredInterface(java.lang.String[])
    */
   public Class<?> getRequiredInterface(String[] kind) throws ConfigurationItemNotDefinedException
   {
      // Kind cannot be null.
      assert kind != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = new ArrayList<String>(Arrays.asList(kind));

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for kind " + kindName);
      }

      Class<?> requiredInterface = null;
      while (requiredInterface == null && !kindName.isEmpty())
      {
         requiredInterface = this.bindingsMap.get(kindName).getRequiredInterface();

         kindName.remove(kindName.size() - 1);
      }

      return requiredInterface;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getSubKindsList(java.lang.String[])
    */
   public Set<String> getSubKindsList(String[] kind) throws ConfigurationItemNotDefinedException
   {
      // Kind cannot be null.
      assert kind != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = Arrays.asList(kind);

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for kind " + kindName);
      }

      return this.bindingsMap.get(kindName).getSubKinds();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementation(java.lang.Class,
    *      java.lang.String, org.cleversafe.util.Version)
    */
   @SuppressWarnings("unchecked")
   public <T> T getImplementation(Class<T> interfaceClass, String referral, Version version)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      List<String> bindingName = Arrays.asList(new String[]{
         interfaceClass.getName()
      });
      if (!this.bindingsMap.containsKey(bindingName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for interface "
               + interfaceClass.getName());
      }

      InterfaceImplementationFactory iFactory =
            (InterfaceImplementationFactory) this.bindingsMap.get(bindingName).getImplementationFactory(
                  referral, version);

      return (T) iFactory.getImplementationObject();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementation(java.lang.String[],
    *      java.lang.String, org.cleversafe.util.Version)
    */
   public Object getImplementation(String[] kind, String referral, Version version)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      // Kind and referral cannot be null.
      assert kind != null;
      assert referral != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = Arrays.asList(kind);

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for kind " + kindName);
      }

      ImplementationFactoryBase iFactory =
            this.bindingsMap.get(kindName).getImplementationFactory(referral, version);

      return iFactory.getImplementationObject();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementationClass(java.lang.Class,
    *      java.lang.String, org.cleversafe.util.Version)
    */
   public Class<?> getImplementationClass(Class<?> interfaceClass, String referral, Version version)
         throws ConfigurationItemNotDefinedException
   {
      List<String> bindingName = Arrays.asList(new String[]{
         interfaceClass.getName()
      });
      if (!this.bindingsMap.containsKey(bindingName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for interface "
               + interfaceClass.getName());
      }

      InterfaceImplementationFactory iFactory =
            (InterfaceImplementationFactory) this.bindingsMap.get(bindingName).getImplementationFactory(
                  referral, version);

      return iFactory.getImplementationClass();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.BindingsProvider#getImplementationClass(java.lang.String[],
    *      java.lang.String, org.cleversafe.util.Version)
    */
   public Class<?> getImplementationClass(String[] kind, String referral, Version version)
         throws ConfigurationItemNotDefinedException
   {
      // Kind and referral cannot be null.
      assert kind != null;
      assert referral != null;

      // Create the qualified kind name from the string array.
      final List<String> kindName = Arrays.asList(kind);

      if (!this.bindingsMap.containsKey(kindName))
      {
         throw new ConfigurationItemNotDefinedException("No binding defined for kind " + kindName);
      }

      ImplementationFactoryBase iFactory =
            this.bindingsMap.get(kindName).getImplementationFactory(referral, version);

      return iFactory.getImplementationClass();
   }
}
