//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER


package org.cleversafe.config;

import java.util.List;

import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;

/**
 * This interface extends the generic {@link PropertyInterface} and allows typed values to be
 * accessed by the application using qualified names. A qualified name corresponds to a path that is
 * split into an array of strings.
 * 
 * <p/> For example, a property's qualified name, <b>org.cleversafe.network.concurrency</b> will be
 * represented as follows in order to be accessed via the PropertiesProvider:<br/>
 * 
 * <b>new String()[]{ "org", "cleversafe", "network", "concurrency" };</b>
 * 
 * @author Manish Motwani
 */
public interface PropertiesProvider extends PropertyInterface
{
   /**
    * Returns the string value of the property specified by the input qualified name, if it exists.
    * If the string type property does not exist, an exception is thrown. Use the generic getValue()
    * method to access any type property value converted to a string.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The string value for the input property.
    * @throws ConfigurationItemNotDefinedException
    */
   public String getStringValue(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the string values list of the property specified by the input qualified name, if any
    * exist. If the string type property does not exist, an exception is thrown. Use the generic
    * getValuesList() method to access any type property values converted to a string list.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The list of string values for the input property.
    * @throws ConfigurationItemNotDefinedException
    */
   public List<String> getStringValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException;

   /**
    * Sets the default string value of the property specified by the input qualified name, if no
    * values exist in the configuration. If one or more values exist for this property in the
    * configuration, this method is ignored.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @param value
    *           The default string value to set for the specified property.
    */
   public void setDefaultStringValue(final String[] qualifiedName, final String value);

   /**
    * Sets the default string values list of the property specified by the input qualified name, if
    * no values exist in the configuration. If one or more values exist for this property in the
    * configuration, this method is ignored.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @param values
    *           The default string values to set for the specified property.
    */
   public void setDefaultStringValuesList(final String[] qualifiedName, final String[] values);

   /**
    * Gets the number of string values that exist in the configuration for the property specified by
    * the input qualified name.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The number of string values for the input property.
    */
   public int getNumStringValuesSet(final String[] qualifiedName);

   /**
    * Returns the int value of the property specified by the input qualified name, if it exists. If
    * the int type property does not exist, an exception is thrown.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The int value for the input property.
    * @throws ConfigurationItemNotDefinedException
    */
   public int getIntValue(final String[] qualifiedName) throws ConfigurationItemNotDefinedException;

   /**
    * Returns the int values list of the property specified by the input qualified name, if any
    * exist. If the int type property does not exist, an exception is thrown.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The list of int values for the input property.
    * @throws ConfigurationItemNotDefinedException
    */
   public List<Integer> getIntValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException;

   /**
    * Sets the default int value of the property specified by the input qualified name, if no values
    * exist in the configuration. If one or more values exist for this property in the
    * configuration, this method is ignored.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @param value
    *           The default int value to set for the specified property.
    */
   public void setDefaultIntValue(final String[] qualifiedName, final int value);

   /**
    * Sets the default int values list of the property specified by the input qualified name, if no
    * values exist in the configuration. If one or more values exist for this property in the
    * configuration, this method is ignored.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @param values
    *           The default int values to set for the specified property.
    */
   public void setDefaultIntValuesList(final String[] qualifiedName, final int[] values);

   /**
    * Gets the number of int values that exist in the configuration for the property specified by
    * the input qualified name.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The number of int values for the input property.
    */
   public int getNumIntValuesSet(final String[] qualifiedName);

   /**
    * Returns the boolean value of the property specified by the input qualified name, if it exists.
    * If the boolean type property does not exist, an exception is thrown.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The boolean value for the input property.
    * @throws ConfigurationItemNotDefinedException
    */
   public boolean getBooleanValue(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the boolean values list of the property specified by the input qualified name, if any
    * exist. If the boolean type property does not exist, an exception is thrown.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The list of boolean values for the input property.
    * @throws ConfigurationItemNotDefinedException
    */
   public List<Boolean> getBooleanValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException;

   /**
    * Sets the default boolean value of the property specified by the input qualified name, if no
    * values exist in the configuration. If one or more values exist for this property in the
    * configuration, this method is ignored.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @param value
    *           The default boolean value to set for the specified property.
    */
   public void setDefaultBooleanValue(final String[] qualifiedName, final boolean value);

   /**
    * Sets the default boolean values list of the property specified by the input qualified name, if
    * no values exist in the configuration. If one or more values exist for this property in the
    * configuration, this method is ignored.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @param values
    *           The default boolean values to set for the specified property.
    */
   public void setDefaultBooleanValuesList(final String[] qualifiedName, final boolean[] values);

   /**
    * Gets the number of boolean values that exist in the configuration for the property specified
    * by the input qualified name.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The number of boolean values for the input property.
    */
   public int getNumBooleanValuesSet(final String[] qualifiedName);

   /**
    * Returns the double value of the property specified by the input qualified name, if it exists.
    * If the double type property does not exist, an exception is thrown.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The double value for the input property.
    * @throws ConfigurationItemNotDefinedException
    */
   public double getDoubleValue(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the double values list of the property specified by the input qualified name, if any
    * exist. If the double type property does not exist, an exception is thrown.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The list of double values for the input property.
    * @throws ConfigurationException
    */
   public List<Double> getDoubleValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException;

   /**
    * Sets the default double value of the property specified by the input qualified name, if no
    * values exist in the configuration. If one or more values exist for this property in the
    * configuration, this method is ignored.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @param value
    *           The default double value to set for the specified property.
    */
   public void setDefaultDoubleValue(final String[] qualifiedName, final double value);

   /**
    * Sets the default double values list of the property specified by the input qualified name, if
    * no values exist in the configuration. If one or more values exist for this property in the
    * configuration, this method is ignored.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @param values
    *           The default double values to set for the specified property.
    */
   public void setDefaultDoubleValuesList(final String[] qualifiedName, final double[] values);

   /**
    * Gets the number of double values that exist in the configuration for the property specified by
    * the input qualified name.
    * 
    * @param qualifiedName
    *           The qualified name string array that represents the property.
    * @return The number of double values for the input property.
    */
   public int getNumDoubleValuesSet(final String[] qualifiedName);
}
