//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: May 16, 2007
//---------------------

package org.cleversafe.config;

import java.util.ArrayList;
import java.util.List;

import org.cleversafe.config.evaluator.Literal;
import org.cleversafe.config.evaluator.LiteralEvaluatorFactory;
import org.cleversafe.config.evaluator.NamedEvaluator;
import org.cleversafe.config.evaluator.ReferenceEvaluator;
import org.cleversafe.config.evaluator.SimpleParameter;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;

// TODO: Describe class or interface
public class ImplementationFactoryBase
{
   private Class<?> implementationClass;
   private List<NamedEvaluator> parameters;

   /**
    * 
    * @param implementationName
    * @param parameters
    * @throws ObjectInstantiationException
    * @throws ConfigurationException
    */
   public ImplementationFactoryBase(
         final String implementationName,
         final List<SimpleParameter> parameters) throws ClassNotFoundException,
         ObjectInstantiationException
   {
      this.implementationClass = Class.forName(implementationName);
      this.parameters = getNamedEvaluators(parameters);
   }

   /**
    * 
    * @param implementationClass
    * @param parameters
    * @throws ObjectInstantiationException
    * @throws ConfigurationException
    */
   public ImplementationFactoryBase(
         final Class<?> implementationClass,
         final List<SimpleParameter> parameters) throws ObjectInstantiationException
   {
      this.implementationClass = implementationClass;
      this.parameters = getNamedEvaluators(parameters);
   }

   /**
    * Creates an object based on Class and initialization parameters
    * 
    * @return
    * @throws ConfigurationException
    */
   public Object getImplementationObject() throws ObjectInstantiationException,
         ObjectInitializationException
   {
      Object classInstance;

      try
      {
         classInstance = this.implementationClass.newInstance();
      }
      catch (final Exception e)
      {
         throw new ObjectInstantiationException("Can't instatiate an object of type '"
               + this.implementationClass.getName() + "'", e);
      }

      final ReferenceEvaluator evaluator = new ReferenceEvaluator(classInstance, this.parameters);
      return evaluator.evaluateWithoutInitialization();
   }

   private List<NamedEvaluator> getNamedEvaluators(final List<SimpleParameter> parameters)
         throws ObjectInstantiationException
   {
      if (parameters == null)
      {
         return new ArrayList<NamedEvaluator>();
      }
      else
      {
         final List<NamedEvaluator> evaluators = new ArrayList<NamedEvaluator>(parameters.size());
         for (final SimpleParameter param : parameters)
         {
            final Literal literal = param.getLiteral();
            final NamedEvaluator evaluator =
                  new NamedEvaluator(param.getName(), LiteralEvaluatorFactory.getEvaluator(
                        literal.getType().toString(), literal.getValue()));
            evaluators.add(evaluator);
         }
         return evaluators;
      }
   }

   /**
    * @return the implementationClass
    */
   public Class<?> getImplementationClass()
   {
      return this.implementationClass;
   }

   /**
    * @param implementationClass
    *           the implementationClass to set
    */
   public void setImplementationClass(final Class<?> implementationClass)
   {
      this.implementationClass = implementationClass;
   }

   /**
    * @param parameters
    *           the parameters to set
    * @throws ObjectInstantiationException
    */
   public void setParameters(final List<SimpleParameter> parameters)
         throws ObjectInstantiationException
   {
      this.parameters = getNamedEvaluators(parameters);
   }

}
