//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: May 17, 2007
//---------------------

package org.cleversafe.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.cleversafe.config.exceptions.ConfigurationLoadException;

// This class is used to instantiate configuration of different types
// Currently XML configuration is supported
// Configuration implementation should have a constructor with InputStream
// It is by default is configured with a well defined property 

public class ConfigurationFactory
{
   private static Logger _logger = Logger.getLogger(ConfigurationFactory.class);

   static Map<String, Class<?>> bindingTypes = new HashMap<String, Class<?>>();
   static Map<String, String> bindingsInit = new HashMap<String, String>();

   static Map<String, Class<?>> propertyTypes = new HashMap<String, Class<?>>();
   static Map<String, String> propertyInit = new HashMap<String, String>();

   static Map<String, BindingsProvider> bindingsInstance = new HashMap<String, BindingsProvider>();
   static Map<String, PropertiesProvider> propertiesInstance =
         new HashMap<String, PropertiesProvider>();

   public static final String XML_PROPERTIES_CONFIG_PROPERTY = "org.cleversafe.xml.properties";
   public static final String XML_BINDINGS_CONFIG_PROPERTY = "org.cleversafe.xml.configuration";
   public static final String XML_CONFIG_TYPE = "XML";

   static
   {
      bindingTypes.put(XML_CONFIG_TYPE, XMLBindingsProvider.class);
      bindingsInit.put(XML_CONFIG_TYPE, XML_BINDINGS_CONFIG_PROPERTY);

      propertyTypes.put(XML_CONFIG_TYPE, XMLPropertiesProvider.class);
      propertyInit.put(XML_CONFIG_TYPE, XML_PROPERTIES_CONFIG_PROPERTY);
   }

   public static void registerBinding(
         final String type,
         final String property,
         final Class<?> implementation)
   {
      bindingTypes.put(type, implementation);
      bindingsInit.put(type, property);
   }

   public static void registerProperties(
         final String type,
         final String property,
         final Class<?> implementation)
   {
      propertyTypes.put(type, implementation);
      propertyInit.put(type, property);
   }

   public static PropertiesProvider getPropertiesProvider(final String type)
         throws ConfigurationLoadException
   {
      if (!propertiesInstance.containsKey(type))
      {
         synchronized (ConfigurationFactory.class)
         {
            // build a new one
            final String propertyName = propertyInit.get(type);
            if (propertyName == null)
            {
               throw new ConfigurationLoadException("Unknown configuration type:" + type);
            }
            final String propertyValue = System.getProperty(propertyName);
            if (propertyValue == null)
            {
               throw new ConfigurationLoadException("Property " + propertyName
                     + " for properties configuration type:" + type + " is not defined");
            }

            InputStream if1 = null;
            try
            {
               if1 = new FileInputStream(new File(propertyValue));
            }
            catch (final FileNotFoundException e)
            {
               if1 = ConfigurationFactory.class.getClassLoader().getResourceAsStream(propertyValue);
            }

            if (if1 == null)
            {
               throw new ConfigurationLoadException("Can't load resource '" + propertyValue);
            }
            else
            {
               _logger.info("Loading " + type + " properties configuration from resource: "
                     + propertyValue);
            }

            try
            {
               final Constructor<?> ctr = propertyTypes.get(type).getConstructor(InputStream.class);
               final PropertiesProvider propertiesProvider =
                     (PropertiesProvider) ctr.newInstance(if1);
               propertiesInstance.put(type, propertiesProvider);
            }
            catch (final Exception e)
            {
               throw new ConfigurationLoadException("Could not instantiate PropertiesProvider", e);
            }
            finally
            {
               if (if1 != null)
               {
                  try
                  {
                     if1.close();
                  }
                  catch (final IOException ignore)
                  {
                  }
               }
            }
         }
      }

      return propertiesInstance.get(type);
   }

   public static BindingsProvider getBindingsProvider(final String type)
         throws ConfigurationLoadException
   {
      if (!bindingsInstance.containsKey(type))
      {
         synchronized (ConfigurationFactory.class)
         {
            // build a new one
            final String propertyName = bindingsInit.get(type);
            if (propertyName == null)
            {
               throw new ConfigurationLoadException("Unknown configuration type:" + type);
            }
            final String propertyValue = System.getProperty(propertyName);
            if (propertyValue == null)
            {
               throw new ConfigurationLoadException("Property " + propertyName
                     + " for bindings configuration type:" + type + " is not defined");
            }

            // Multiple binding resources separated by semicolons
            final String[] propertyList = propertyValue.split("\\;");
            BindingsProvider bindingsProvider = null;

            for (final String singleProperty : propertyList)
            {
               InputStream if1 = null;
               try
               {
                  if1 = new FileInputStream(new File(singleProperty));
               }
               catch (final FileNotFoundException e)
               {
                  if1 =
                        ConfigurationFactory.class.getClassLoader().getResourceAsStream(
                              singleProperty);
               }

               if (if1 == null)
               {
                  throw new ConfigurationLoadException("Can't load resource '" + singleProperty);
               }
               else
               {
                  _logger.info("Loading " + type + " bindings configuration from resource: "
                        + singleProperty);
               }

               try
               {
                  if (bindingsProvider == null)
                  {
                     final Class<?> cls = bindingTypes.get(type);
                     bindingsProvider = (BindingsProvider) cls.newInstance();
                     bindingsInstance.put(type, bindingsProvider);
                  }
                  bindingsProvider.loadBindings(if1);
               }
               catch (final Exception e)
               {
                  throw new ConfigurationLoadException("Could not instantiate BindingsProvider", e);
               }
               finally
               {
                  if (if1 != null)
                  {
                     try
                     {
                        if1.close();
                     }
                     catch (final IOException ignore)
                     {
                     }
                  }
               }
            }
         }
      }

      return bindingsInstance.get(type);
   }

   /**
    * Unloads a configuration instance if one exists. Allows run-time reloading If any
    * synchronization is required, it should be performed by client
    * 
    * @param type
    * @return
    */
   public static BindingsProvider unloadBindings(final String type)
   {
      return bindingsInstance.remove(type);
   }
}
