//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Nov 29, 2007
//---------------------

package org.cleversafe.config;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;

import org.apache.xmlbeans.XmlException;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.config.exceptions.IllegalConfigurationFormatException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.util.Version;

// TODO: Describe class or interface
public interface BindingsProvider
{
   /**
    * Returns the typed object of the implementation of the input interface class. Throws an
    * exception if an implementation is not found in configuration for the given interface.
    * 
    * @param interfaceClass
    *           The interface class to look for in configuration.
    * @return
    * @throws ConfigurationItemNotDefinedException,
    *            ObjectInstantiationException, ObjectInitializationException
    */
   public <T> T getDefaultImplementation(Class<T> interfaceClass)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException;

   /**
    * Returns the default implementation for a given qualified kind name. An exception is thrown if
    * no such kind is found. If the kind is found and there's no default implementation, null is
    * returned. In this case, a referral must be used to acquire the implementing object.
    * 
    * @param kind
    * @return
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public Object getDefaultImplementation(final String[] kind)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException;

   /**
    * Returns the default implementation class for a given interface class. Throws an exception if
    * an implementation is not found in configuration for the given interface.
    * 
    * @param interfaceClass
    * @return
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getDefaultImplementationClass(Class<?> interfaceClass)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the default implementation class for a given kind. Throws an exception if an
    * implementation is not found in configuration for the kind.
    * 
    * @param kind
    * @return
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getDefaultImplementationClass(String[] kind)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the implementation from an interface class + referral combination.
    * 
    * @param <T>
    * @param interfaceClass
    * @param referral
    * @return
    */
   public <T> T getImplementation(final Class<T> interfaceClass, final String referral)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException;

   /**
    * Returns the implementation from an interface class + referral + version combination.
    * 
    * @param <T>
    * @param interfaceClass
    * @param referral
    * @param version
    * @return
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public <T> T getImplementation(
         final Class<T> interfaceClass,
         final String referral,
         final Version version) throws ConfigurationItemNotDefinedException,
         ObjectInstantiationException, ObjectInitializationException;

   /**
    * Convinience interface for most typical cases (1 kind) A Shortcut getImplementation(new
    * String[]{kind1}, referral);
    * 
    * @param kind1
    * @param referral
    * @return
    * @throws ConfigurationItemNotDefinedException,
    *            ObjectInstantiationException, ObjectInitializationException
    */
   public Object getImplementation(final String kind1, final String referral)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException;

   /**
    * Convenience interface for most typical cases (2 kinds) A shortcut getImplementation(new
    * String[]{kind1, kind2}, referral);
    * 
    * @param kind1
    * @param referral
    * @return
    * @throws ConfigurationItemNotDefinedException,
    *            ObjectInstantiationException, ObjectInitializationException
    */
   public Object getImplementation(final String kind1, final String kind2, final String referral)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException;

   /**
    * Instantiates and returns an object of the given kind if the referral name exists in one of the
    * kind's implementations. If the kind does not exist, or the referral for the kind does not
    * exist, an exception is thrown.
    * 
    * @param kind
    *           The string array that represents the path to a kind.
    * @param referral
    * @return Object of the given kind instantiated with the input referral implementation.
    * @throws ConfigurationItemNotDefinedException,
    *            ObjectInstantiationException, ObjectInitializationException
    */
   public Object getImplementation(final String[] kind, final String referral)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException;

   /**
    * Instantiates and returns an object of the given kind with a matching referral name and
    * version. If the kind does not exist, or the referral for the kind does not exist, or the
    * version does not match, an exception is thrown.
    * 
    * @param kind
    * @param referral
    * @param version
    * @return Object of the given kind instantiated with the input referral and version.
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public Object getImplementation(final String[] kind, final String referral, final Version version)
         throws ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException;

   /**
    * Returns the implementing class of the given interface class if the referral name exists in one
    * of the kind's implementations. If the interface is not defined, or the referral for the
    * interface does not exist, an exception is thrown.
    * 
    * @param interfaceClass
    * @param referral
    * @return
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getImplementationClass(Class<?> interfaceClass, final String referral)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the implementing class of the given interface class if a matching referral name and
    * version exists in one of the kind's implementations. If the interface is not defined, or the
    * matching referral and version does not exist, an exception is thrown.
    * 
    * @param interfaceClass
    * @param referral
    * @param version
    * @return
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getImplementationClass(
         Class<?> interfaceClass,
         final String referral,
         final Version version) throws ConfigurationItemNotDefinedException;

   /**
    * Convenience interface for most typical cases (1 kind). A Shortcut getImplementationClass(new
    * String[]{kind1}, referral);
    * 
    * @param kind1
    * @param referral
    * @return
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getImplementationClass(final String kind1, final String referral)
         throws ConfigurationItemNotDefinedException;

   /**
    * Convenience interface for most typical cases (2 kinds). A Shortcut getImplementationClass(new
    * String[]{kind1, kind2}, referral);
    * 
    * @param kind1
    * @param kind2
    * @param referral
    * @return
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getImplementationClass(
         final String kind1,
         final String kind2,
         final String referral) throws ConfigurationItemNotDefinedException;

   /**
    * Returns the class of the given kind if the referral name exists in one of the kind's
    * implementations. If the kind does not exist, or the referral for the kind does not exist, an
    * exception is thrown.
    * 
    * @param kind
    * @param referral
    * @return
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getImplementationClass(final String[] kind, final String referral)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the class of the given kind if a matching referral name and version exists in one of
    * the kind's implementations. If the kind does not exist, or there's not match for the given
    * referral and version, an exception is thrown.
    * 
    * @param kind
    * @param referral
    * @param version
    * @return
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getImplementationClass(
         final String[] kind,
         final String referral,
         final Version version) throws ConfigurationItemNotDefinedException;

   /**
    * Returns the implementation classes for a given kind. If there are no implementations defined
    * for the given kind, an empty list is returned. If the given kind does not exist, an exception
    * is thrown.
    * 
    * @param kind
    *           The string array that represents the path to a kind.
    * @return The array of all classes of the given kind.
    * @throws ConfigurationItemNotDefinedException
    */
   public Set<Class<?>> getImplementationClassSet(final String[] kind)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the interface class that the input kind requires. If no interface is required for the
    * given kind, null is returned. If the given kind does not exist, an exception is thrown.
    * 
    * @param kind
    *           The string array that represents the path to a kind.
    * @return The required interface class for the kind.
    * @throws ConfigurationItemNotDefinedException
    */
   public Class<?> getRequiredInterface(final String[] kind)
         throws ConfigurationItemNotDefinedException;

   /**
    * Returns the sub kinds for a given kind. If the given kind does not exist, an exception is
    * thrown.
    * 
    * @param kind
    *           The string array that represents the path to a kind.
    * @return The string array of kind names.
    * @throws ConfigurationItemNotDefinedException
    */
   public Set<String> getSubKindsList(final String[] kind)
         throws ConfigurationItemNotDefinedException;

   /**
    * Load bindings from an XML input stream.
    * 
    * @param xmlInputStream
    * @throws XmlException
    * @throws IOException
    * @throws IllegalConfigurationContentException
    * @throws IllegalConfigurationFormatException
    */
   public void loadBindings(final InputStream xmlInputStream) throws XmlException, IOException,
         IllegalConfigurationContentException, IllegalConfigurationFormatException;

   /**
    * Load bindings from an XML file.
    * 
    * @param xmlFileName
    * @throws IllegalConfigurationContentException
    * @throws IllegalConfigurationFormatException
    * @throws FileNotFoundException
    * @throws XmlException
    * @throws IOException
    */
   public void loadBindings(final String xmlFileName) throws IllegalConfigurationContentException,
         IllegalConfigurationFormatException, FileNotFoundException, XmlException, IOException;
}
