//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Feb 6, 2008
//---------------------

package org.cleversafe.config;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.util.Version;

// TODO: Describe class or interface
public class Binding
{
   private static Logger _logger = Logger.getLogger(Binding.class);

   public static final Version MIN_VERSION = Version.fromString("0");
   public static final Version MAX_VERSION =
         Version.fromString(Short.MAX_VALUE + "." + Short.MAX_VALUE + "." + Short.MAX_VALUE);

   public static class Implementation implements Comparable<Implementation>
   {
      private final ImplementationFactoryBase implementationFactory;

      public Implementation(ImplementationFactoryBase implementationFactory)
      {
         this.implementationFactory = implementationFactory;
      }

      public ImplementationFactoryBase getImplementationFactory()
      {
         return this.implementationFactory;
      }

      public int compareTo(Implementation o)
      {
         // If the other class is not versioned, they're equal (0)
         if (o.getClass() == this.getClass())
            return 0;

         // otherwise 'this' is smaller (-1) than the versioned implementation.
         else
            return -1;
      }

   }

   public static class VersionedImplementation extends Implementation
   {
      private final Version minVersion;
      private final Version maxVersion;

      public VersionedImplementation(
            ImplementationFactoryBase implementationFactory,
            Version minVersion,
            Version maxVersion)
      {
         super(implementationFactory);
         this.minVersion = minVersion;
         this.maxVersion = maxVersion;
      }

      public String getVersionRange()
      {
         return this.minVersion.toString() + ":" + this.maxVersion.toString();
      }

      @Override
      public int compareTo(Implementation o)
      {
         // If the other class is also versioned, compare the maxVersions
         if (o instanceof VersionedImplementation)
         {
            return this.maxVersion.compareTo(((VersionedImplementation) o).maxVersion);
         }

         // else 'this' is greater than the non-versioned implementation.
         else
         {
            return 1;
         }
      }

      public boolean supports(Version version)
      {
         // If the minVersion and maxVersions are same
         if (this.minVersion.compareTo(this.maxVersion) == 0)
            return version.compareTo(this.minVersion) == 0;

         // If the version is within range, INclusive of minVersion, EXclusive of maxVersion 
         else if (version.compareTo(this.minVersion) >= 0 && version.compareTo(this.maxVersion) < 0)
            return true;

         // Return false otherwise
         else
            return false;
      }
   }

   // Binding name used for logging and error reporting purposes
   private final List<String> qualifiedBindingName;

   // Maps referral to an implementation
   private final Map<String, List<Implementation>> referralImplementations =
         new HashMap<String, List<Implementation>>();

   // Will be set if this binding requires an interface
   private Class<?> requiredInterface = null;

   private final Set<String> subKinds = new HashSet<String>();

   private ImplementationFactoryBase defaultImplementation = null;

   public Binding(List<String> qualifiedBindingName)
   {
      this.qualifiedBindingName = qualifiedBindingName;
   }

   private void checkRequiredInterface(ImplementationFactoryBase implementationFactory)
         throws IllegalConfigurationContentException
   {
      if (this.requiredInterface != null)
      {
         if (!(implementationFactory instanceof InterfaceImplementationFactory))
         {
            _logger.error("Implementation class " + implementationFactory.getImplementationClass()
                  + " in binding " + this.qualifiedBindingName
                  + " must implement the required interface " + this.requiredInterface);
            throw new IllegalConfigurationContentException("Implementation class "
                  + implementationFactory.getImplementationClass() + " in binding "
                  + this.qualifiedBindingName + " must implement the required interface "
                  + this.requiredInterface);
         }
         else
         {
            InterfaceImplementationFactory interfaceImplementationFactory =
                  (InterfaceImplementationFactory) implementationFactory;

            if (!this.requiredInterface.isAssignableFrom(interfaceImplementationFactory.getInterfaceClass()))
            {
               _logger.error("Implementation class "
                     + implementationFactory.getImplementationClass() + " in binding "
                     + this.qualifiedBindingName + " must implement the required interface "
                     + this.requiredInterface);
               throw new IllegalConfigurationContentException("Implementation class "
                     + implementationFactory.getImplementationClass() + " in binding "
                     + this.qualifiedBindingName + " must implement the required interface "
                     + this.requiredInterface);
            }
         }
      }
   }

   public List<String> getQualifiedBindingName()
   {
      return this.qualifiedBindingName;
   }

   public void setDefaultImplementationFactory(ImplementationFactoryBase implementationFactory)
         throws IllegalConfigurationContentException
   {
      if (this.defaultImplementation != null)
      {
         _logger.error("Multiple default implementations defined for binding "
               + this.qualifiedBindingName);
         throw new IllegalConfigurationContentException(
               "Multiple default implementations defined for binding " + this.qualifiedBindingName);
      }

      checkRequiredInterface(implementationFactory);
      this.defaultImplementation = implementationFactory;
   }

   public ImplementationFactoryBase getDefaultImplementationFactory()
         throws ConfigurationItemNotDefinedException
   {
      if (this.defaultImplementation == null)
      {
         _logger.error("No default implementation defined for binding " + this.qualifiedBindingName);
         throw new ConfigurationItemNotDefinedException(
               "No default implementation defined for binding " + this.qualifiedBindingName);
      }
      return this.defaultImplementation;
   }

   public void setRequiredInterface(Class<?> requiredInterface)
   {
      this.requiredInterface = requiredInterface;
   }

   public Class<?> getRequiredInterface()
   {
      return this.requiredInterface;
   }

   public void addImplementationFactory(
         String referral,
         ImplementationFactoryBase implementationFactory)
         throws IllegalConfigurationContentException
   {
      checkRequiredInterface(implementationFactory);

      List<Implementation> implementations = this.referralImplementations.get(referral);

      if (implementations == null)
      {
         implementations = new ArrayList<Implementation>();
         this.referralImplementations.put(referral, implementations);
      }

      implementations.add(new Implementation(implementationFactory));
   }

   public void addSubKind(String subBinding)
   {
      this.subKinds.add(subBinding);
   }

   public Set<String> getSubKinds()
   {
      return this.subKinds;
   }

   public void addImplementationFactory(
         String referral,
         ImplementationFactoryBase implementationFactory,
         Version minVersion,
         Version maxVersion) throws IllegalConfigurationContentException
   {
      checkRequiredInterface(implementationFactory);

      List<Implementation> implementations = this.referralImplementations.get(referral);

      if (implementations == null)
      {
         implementations = new ArrayList<Implementation>();
         this.referralImplementations.put(referral, implementations);
      }

      implementations.add(new VersionedImplementation(implementationFactory, minVersion, maxVersion));
   }

   public ImplementationFactoryBase getImplementationFactory(String referral)
         throws ConfigurationItemNotDefinedException
   {
      List<Implementation> implementations = this.referralImplementations.get(referral);
      if (implementations == null)
      {
         _logger.error("No implementation is defined for binding " + this.qualifiedBindingName
               + " and referral " + referral);
         throw new ConfigurationItemNotDefinedException("No implementation is defined for binding "
               + this.qualifiedBindingName + " and referral " + referral);
      }

      // Create new list so the original is not modified during sorting
      implementations = new ArrayList<Implementation>(implementations);

      // If there's only one implementation version defined
      if (implementations.size() == 1)
      {
         // return it
         return implementations.get(0).getImplementationFactory();
      }
      // If there are multiple implementation versions defined for this referral
      else if (implementations.size() > 1)
      {
         Collections.sort(implementations);

         // Return the largest version
         return implementations.get(implementations.size() - 1).getImplementationFactory();
      }

      // If no implementation is defined for the referral
      else
      {
         _logger.error("No implementation is defined for binding " + this.qualifiedBindingName
               + " and referral " + referral);
         throw new ConfigurationItemNotDefinedException("No implementation is defined for binding "
               + this.qualifiedBindingName + " and referral " + referral);
      }
   }

   public ImplementationFactoryBase getImplementationFactory(String referral, Version version)
         throws ConfigurationItemNotDefinedException
   {
      List<Implementation> implementations = this.referralImplementations.get(referral);
      if (implementations == null)
      {
         _logger.error("No implementation is defined for binding " + this.qualifiedBindingName
               + " and referral " + referral);
         throw new ConfigurationItemNotDefinedException("No implementation is defined for binding "
               + this.qualifiedBindingName + " and referral " + referral);
      }

      // Create new list so the original is not modified during sorting/reversing
      implementations = new ArrayList<Implementation>(implementations);

      Collections.sort(implementations); // puts the versioned ones at the back
      Collections.reverse(implementations); // put the versioned ones at the front with highest versions first

      for (Implementation implementation : implementations)
      {
         if (implementation instanceof VersionedImplementation)
         {
            if (((VersionedImplementation) implementation).supports(version))
               return implementation.getImplementationFactory();
            else
               continue;
         }
         else
            break;
      }

      _logger.error("No implementation is defined for binding " + this.qualifiedBindingName
            + " and referral " + referral + " of version " + version.toString());
      throw new ConfigurationItemNotDefinedException("No implementation is defined for binding "
            + this.qualifiedBindingName + " and referral " + referral + " of version "
            + version.toString());
   }

   public Set<Class<?>> getImplementationClassSet()
   {
      Set<Class<?>> classSet = new HashSet<Class<?>>();

      Collection<List<Implementation>> c = this.referralImplementations.values();

      for (List<Implementation> list : c)
      {
         for (Implementation impl : list)
         {
            classSet.add(impl.getImplementationFactory().getImplementationClass());
         }
      }

      if (this.defaultImplementation != null
            && !classSet.contains(this.defaultImplementation.getImplementationClass()))
      {
         classSet.add(this.defaultImplementation.getImplementationClass());
      }

      return classSet;
   }
}
