//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Oct 10, 2007
//---------------------

package org.cleversafe.codec.misc;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import org.cleversafe.codec.Decoder;
import org.cleversafe.codec.exceptions.CodecDecodeException;
import org.cleversafe.layer.grid.SourceName;

// TODO: Describe class or interface
class TranspositionDecoder implements Decoder
{
   private final int threshold;
   private final int blockSize;
   private byte[] data;
   private int position;
   
   private ByteArrayOutputStream buffer = new ByteArrayOutputStream();

   public TranspositionDecoder(int threshold)
   {
      this.position = 0;
      this.threshold = threshold;
      this.blockSize = (threshold * threshold);
      this.data = new byte[blockSize];
   }

   private int calculateInverseTranspositionIndex(int arrayLength)
   {
      int product = this.position * (arrayLength / this.threshold);
      return ((product % arrayLength) + (product / arrayLength));
   }

   public byte[] finish() throws CodecDecodeException
   {
      if (buffer.size() % this.threshold != 0)
      {
         throw new CodecDecodeException("Expected data length to be multiple of threshold");
      }
      
      if (buffer.size() == 0)
      {
         throw new CodecDecodeException("Expected padding");
      }
      
      assert (this.position == 0) : "expected position to be zero";
      
      // Flush remaining data from the buffer
      byte[] processData = this.buffer.toByteArray();
      
      // Untranspose
      byte[] dataPlusPadding = new byte[processData.length];
      
      int i = 0;
      while (i < dataPlusPadding.length)
      {         
         int invTranspositionIndex = calculateInverseTranspositionIndex(dataPlusPadding.length);
         if (invTranspositionIndex < dataPlusPadding.length)
         {
            dataPlusPadding[invTranspositionIndex] = processData[i];
         }

         this.position++;
         i++;  
      }

      // Strip Padding
      byte paddingByte = dataPlusPadding[dataPlusPadding.length - 1];
      int paddingLength = 0xFF & paddingByte;
      if ((paddingLength <= 0) || (paddingLength > this.threshold))
      {
         throw new CodecDecodeException("Padding byte length marker is invalid");
      }
      
      int dataLength = (dataPlusPadding.length - paddingLength);
      ByteArrayOutputStream byteOut = new ByteArrayOutputStream(dataLength);
      
      byteOut.write(dataPlusPadding, 0, dataLength);
      
      this.buffer.reset();
      this.position = 0;
      
      return byteOut.toByteArray();
   }

   public byte[] finish(byte[] data) throws CodecDecodeException
   {
      try
      {
         ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
         byteOut.write(process(data));
         byteOut.write(finish());

         try
         {
            byteOut.close();
         }
         catch (IOException ignore)
         {
         }

         return byteOut.toByteArray();
      }
      catch (IOException e)
      {
         throw new CodecDecodeException("Unknown IO exception while finishing data", e);
      }
   }

   public void reset(SourceName sourceName, long transactionId)
   {
      this.buffer.reset();
      this.position = 0;
   }

   public boolean isInitialized()
   {
      return true;
   }

   public byte[] process(byte[] dataIn) throws CodecDecodeException
   {
      // Save data in buffer until we have more than a (chunkSize) worth of data.
      // Only then can we be sure we are not returning padding
      this.buffer.write(dataIn, 0, dataIn.length);

      
      // Only when we get MORE THAN a complete chunk do we attempt to decode.
      if (this.buffer.size() > this.blockSize)
      {
         byte[] processData = this.buffer.toByteArray();
         
         int processLength = ((processData.length - 1) / this.blockSize) * this.blockSize;
         
         int remainderLength = processData.length - processLength;
         
         this.buffer.reset();
         this.buffer.write(processData, processLength, remainderLength);

         
         
         
         // Create OutputStream for writing out transposed data   
         ByteArrayOutputStream byteOut = new ByteArrayOutputStream(processLength);
   
         int i = 0;
         while (i < processLength)
         {
   
            int invTranspositionIndex = calculateInverseTranspositionIndex(this.blockSize);
            if (invTranspositionIndex < this.blockSize)
            {
               this.data[invTranspositionIndex] = processData[i];
            }
   
            this.position++;
            i++;
   
            if (this.position == this.blockSize)
            {
               byteOut.write(this.data, 0, this.blockSize);
               this.position = 0;
            }
         }
   
         return byteOut.toByteArray();
      }
      else
      {
         return new byte[0];
      }
   }
}