//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: jquigley
//
// Date: Jun 12, 2007
//---------------------

package org.cleversafe.codec.integrity;

import java.io.*;
import java.util.zip.CRC32;

import org.cleversafe.codec.Decoder;
import org.cleversafe.codec.exceptions.CodecDecodeException;
import org.cleversafe.codec.exceptions.CodecInvalidDataFormatException;
import org.cleversafe.codec.exceptions.CodecInvalidParametersException;
import org.cleversafe.codec.exceptions.CodecNotInitializedException;
import org.cleversafe.layer.grid.SourceName;

public class CRCIntegrityDecoder implements Decoder
{
   public static final int CRC_LENGTH = 4;

   private CRC32 crc32;
   private ByteArrayOutputStream incommingData;

   public void reset(SourceName sourceName, long transactionId)
   throws CodecInvalidParametersException
   {
      this.crc32 = new CRC32();
      this.incommingData = new ByteArrayOutputStream();
   }

   public boolean isInitialized()
   {
      return this.crc32 != null;
   }

   public byte[] process(byte[] data) throws CodecDecodeException, CodecInvalidDataFormatException,
         CodecNotInitializedException
   {
      try
      {
         incommingData.write(data);
      }
      catch (IOException e)
      {
         throw new RuntimeException("I/O error", e);
      }

      if (incommingData.size() > CRC_LENGTH)
      {
         data = incommingData.toByteArray();

         byte processedData[] = new byte[data.length - CRC_LENGTH];
         System.arraycopy(data, 0, processedData, 0, processedData.length);

         crc32.update(processedData);

         incommingData.reset();
         incommingData.write(data, data.length - CRC_LENGTH, CRC_LENGTH);

         return processedData;
      }

      return new byte[0];
   }

   public byte[] finish() throws CodecDecodeException, CodecInvalidDataFormatException,
         CodecNotInitializedException
   {
      if (incommingData.size() != CRC_LENGTH)
      {
         throw new CodecInvalidDataFormatException("Invalid data length");
      }

      DataInputStream in =
            new DataInputStream(new ByteArrayInputStream(incommingData.toByteArray()));

      int actualCRC;

      try
      {
         actualCRC = in.readInt();
      }
      catch (IOException e)
      {
         throw new RuntimeException("I/O error", e);
      }

      int expectedCRC = (int) crc32.getValue();

      if (actualCRC != expectedCRC)
      {
         throw new CodecInvalidDataFormatException("CRC integrity verification failed");
      }

      return new byte[0];
   }

   public byte[] finish(byte[] data) throws CodecDecodeException, CodecInvalidDataFormatException,
         CodecNotInitializedException
   {
      byte processedData[] = process(data);

      finish();

      return processedData;
   }

}
