//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 20, 2007
//---------------------

package org.cleversafe.authentication.cli;

import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;
import org.cleversafe.authentication.PropertiesFileCredentialsManager;
import org.cleversafe.authentication.exceptions.CredentialsException;
import org.cleversafe.authentication.exceptions.CredentialsIOException;
import org.cleversafe.authentication.exceptions.CredentialsKeyException;
import org.cleversafe.util.JSAPCommandLineParser;

import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;

/**
 * Command line interface tool to create grid account credentials and generate account keypairs.
 * 
 */
public class CreateAccount
{

   protected static final String COMMAND_NAME = "dsgrid-create-account";

   protected static final String ACCOUNT_PATH_PROPERTY =
         "org.cleversafe.authentication.default.path";

   protected static final String ACCOUNT_PATH_OPTION = "ACCOUNT_PATH";
   protected static final String USERNAME_OPTION = "USERNAME";
   protected static final String PASSWORD_OPTION = "PASSWORD";
   protected static final String KEYALG_OPTION = "KEYALG";
   protected static final String KEYSIZE_OPTION = "KEYSIZE";
   protected static final String SIGHASH_OPTION = "SIGHASH";

   private static Logger _logger = Logger.getLogger(CreateAccount.class);

   public static final int OPTION_ERROR = 255;
   public static final int IO_ERROR = 10;
   public static final int UNKNOWN_ERROR = 1;

   protected static UUID create(JSAPResult options)
   {
      String username = options.getString(USERNAME_OPTION);
      String password = options.getString(PASSWORD_OPTION);
      String keyAlgorithm = options.getString(KEYALG_OPTION);
      int keySize = options.getInt(KEYSIZE_OPTION);
      String signatureHash = options.getString(SIGHASH_OPTION);

      PropertiesFileCredentialsManager manager = null;
      try
      {
         manager =
               PropertiesFileCredentialsManager.create(username, password, keyAlgorithm, keySize,
                     signatureHash + "with" + keyAlgorithm);

         return manager.getAccountIdentifier(username);
      }
      catch (CredentialsIOException e)
      {
         System.err.println("error: could not write credentials to file: " + e.getMessage());
         System.exit(IO_ERROR);
      }
      catch (CredentialsKeyException e)
      {
         System.err.println("error: could not create account key pair: " + e.getMessage());
         System.exit(IO_ERROR);
      }
      catch (CredentialsException e)
      {
         System.err.println("error: could not generate grid account identifier");
         System.exit(UNKNOWN_ERROR);
      }
      return null;
   }

   /**
    * @param args
    */
   public static void main(String[] args)
   {
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));

      JSAPCommandLineParser.logEnvironment(args);

      // Log environment variables
      Map<String, String> variables = System.getenv();
      Set<String> variableNames = variables.keySet();
      Iterator<String> nameIterator = variableNames.iterator();

      for (int index = 0; index < variableNames.size(); index++)
      {
         String name = (String) nameIterator.next();
         String value = (String) variables.get(name);
         _logger.debug("Environment variable: " + name + " = " + value);
      }

      JSAPResult result = null;
      try
      {
         JSAPCommandLineParser parser =
               new JSAPCommandLineParser(CreateAccount.class, "create_account_options.jsap",
                     COMMAND_NAME + " [OPTION]... -u <USERNAME> -p <PASSWORD>",
                     "Creates a new grid account.");

         result = parser.parse(args);
      }
      catch (IOException e)
      {
         System.err.println("error: could not parse command line arguments: " + e.getMessage());
         System.exit(IO_ERROR);
      }
      catch (JSAPException e)
      {
         System.err.println("error: could not parse command line arguments: " + e.getMessage());
         System.exit(OPTION_ERROR);
      }

      if (result == null)
      {
         // Parser handled argument exception
         System.exit(OPTION_ERROR);
      }

      // TODO: This is a temporary hack put in place because apparently the Java
      // command line randomly unsets system properties. Over time this should be
      // phased out!
      if (result.contains(ACCOUNT_PATH_OPTION))
      {
         System.setProperty("org.cleversafe.authentication.default.path",
               result.getFile(ACCOUNT_PATH_OPTION).getPath());
      }

      UUID account = create(result);

      System.out.println("Created new account.");
      System.out.println("Grid Account Username:   " + result.getString(USERNAME_OPTION));
      System.out.println("Grid Account Identifier: " + account);

      _logger.debug("Exiting application");
   }

}
