//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: dhendrickson
//
// Date: Jun 6, 2007
//---------------------

package org.cleversafe.authentication.authenticators;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.Authenticator;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.authentication.exceptions.AuthenticationException;
import org.cleversafe.authentication.exceptions.RemoteAuthenticationException;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.exceptions.CommunicationException;
import org.cleversafe.layer.protocol.PasswordAuthenticationRequest;
import org.cleversafe.layer.protocol.PasswordAuthenticationResponse;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.server.handlers.authentication.PasswordAuthenticationHandler;

/**
 * An authenticator that uses a username and password to authenticate with a ServerApplication using
 * a PasswordAuthenticationHandler.
 * 
 * @see PasswordAuthenticationHandler
 */
public class PasswordAuthenticator implements Authenticator
{
   private static Logger _logger = Logger.getLogger(PasswordAuthenticator.class);

   private Connector conn;
   private PasswordAuthenticationRequest authenticationRequest;
   private PasswordCredentials credentials;

   /**
    * Constructs the authenticator using the given ProtocolEngine and connection ID to define the
    * connection to authenticate.
    * 
    * @param protocolEngine
    *           The engine used to send and receive network requests.
    * @param connectionID
    *           The ID of the connection to authenticate.
    */
   public PasswordAuthenticator(Connector conn)
   {
      this.conn = conn;
      this.authenticationRequest = new PasswordAuthenticationRequest();
      this.credentials = null;
   }

   /**
    * Sets the credentials to use.
    * 
    * @param credentials
    *           A UsernamePasswordCredentials object containing a valid username, password and grid
    *           account ID.
    * 
    * @throws ClassCastException
    *            An invalid Credential implementation is used.
    * 
    * @see PasswordCredentials
    */
   public void setCredentials(Credentials credentials)
   {
      this.credentials = (PasswordCredentials) credentials;

      authenticationRequest.setUsername(this.credentials.getUsername());
      authenticationRequest.setPassword(this.credentials.getPassword());
      authenticationRequest.setGridAccountID(this.credentials.getAccountIdentifier());
   }

   /**
    * Returns the credentials used to authenticate.
    * 
    * @returns The credentials used to authenticate.
    * 
    * @see PasswordCredentials
    */
   public Credentials getCredentials()
   {
      return this.credentials;
   }

   /**
    * Attempts to authenticate by sending a PasswordAuthenticationRequest to a ServerApplication.
    * 
    * @return True if authentication succeeded, otherwise false.
    */
   public boolean login() throws AuthenticationException
   {
      _logger.debug("Attempting to authenticate");

      boolean authenticated = false;

      try
      {
         Response response;
         response = conn.exchange(authenticationRequest);

         if (response.getExceptionFlag())
         {
            authenticated = false;
            _logger.warn(
                  "Authentication failed against server " + conn.getRemoteAddress() +
                  ": " + response.getException().getMessage() );
            throw new RemoteAuthenticationException(
                  "Authentication to slice server " + conn.getRemoteAddress() + 
                  " failed for user " + authenticationRequest.getUsername() + ": " +
                  response.getException().getMessage(), response.getException() );
         }
         else
         {
            authenticated = ((PasswordAuthenticationResponse) response).getAuthenticated();   
         }
      }
      catch (CommunicationException ex)
      {
         _logger.error("Communication error while authenticating: " + ex.getMessage());
         authenticated = false;
      }
      
      if(authenticated)
      {
         _logger.info("Authentication succeeded for user " + authenticationRequest.getUsername());
      }
      else
      {
         _logger.info("Authentication failed for user " + authenticationRequest.getUsername());
      }

      return authenticated;
   }
}
