//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Zach
//
// Date: Jun 18, 2007
//---------------------

package org.cleversafe.authentication;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.security.KeyException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.Calendar;
import java.util.Properties;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.bouncycastle.asn1.x509.X509Name;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.x509.X509V3CertificateGenerator;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.authentication.exceptions.CredentialsException;
import org.cleversafe.authentication.exceptions.CredentialsIOException;
import org.cleversafe.authentication.exceptions.CredentialsKeyException;
import org.cleversafe.util.FileResolver;
import org.cleversafe.util.Resolver;
import org.cleversafe.util.UUIDGen;
import org.cleversafe.util.exceptions.ResolutionException;



/**
 * Manages creating and loading {@link PasswordCredentials} saved in a local properties file.
 * Also manages the grid account encryption key pair when saved in a local Java Key Store file
 * (JKS).
 * <p>
 * JKS files require a password both for writing the key store file itself and for writing and
 * reading specific key store entries. The current implementation saves the encryption key pair
 * using an empty string. The grid account password is not used. Future implementations may use
 * the grid account password but, given that this could potentially result in mismatches between
 * the credentials property file and the key store while at the same time the password is stored
 * in plaintext within the properties file, the security gains are minimal.
 */
public class PropertiesFileCredentialsManager implements CredentialsManager
{
   private static Logger _logger = Logger.getLogger(PropertiesFileCredentialsManager.class);
   
   public final static String SYSTEM_PROPERTY_CREDENTIALS_PATH = 
      "org.cleversafe.authentication.credentials.path";
   public final static String SYSTEM_PROPERTY_KEYSTORE_PATH =
      "org.cleversafe.authentication.keystore.path";
   
   public final static String SYSTEM_PROPERTY_DEFAULT_PATH =
      "org.cleversafe.authentication.default.path";
   
   public final static String CREDENTIALS_EXTENSION = ".credentials";
   public final static String KEYSTORE_EXTENSION = ".jks";
   
   public final static String USERNAME_PROPERTY = "username";
   public final static String PASSWORD_PROPERTY = "password";
   public final static String KEYSTORE_PROPERTY = "keystore";
   
   public static String DEFAULT_CREDENTIALS_DIR = "accounts";
   public static String DEFAULT_CREDENTIALS_PATH = "/etc/dsgrid/" + DEFAULT_CREDENTIALS_DIR;
   
   public final static String KEY_STORE_PROVIDER = "JKS";
   
   private final static int CERTIFICATE_DURATION_YEARS = 15;
   
   private Properties credentialsProperties; 
   
   // These are set to null if location is determined by system configuration (RESOURCE_PATH)
   private File credentialsFile;
   private File keystoreFile;
   
   
   static
   {
      Security.addProvider( new BouncyCastleProvider() );
      
      DEFAULT_CREDENTIALS_PATH =
         System.getProperty( SYSTEM_PROPERTY_DEFAULT_PATH, DEFAULT_CREDENTIALS_PATH );
   }
   
   // Used by unit test
   public static void setDefaultCredentialsPath( String path )
   {
      DEFAULT_CREDENTIALS_PATH = path;
   }
   
   // Used by unit test
   public static String getDefaultCredentialsPath()
   {
      return DEFAULT_CREDENTIALS_PATH;
   }
   
   
   /**
    * Creates a new credentials manager which loads credentials from a base resource path. The
    * location of the resource path is loaded from a java system property. If not set the default
    * is a subdirectory of the user's home (e.g., on Linux <code>${HOME}/.grid/accounts</code>).
    *
    * @see PropertiesFileCredentialsManager#RESOURCE_PATH
    */
   public PropertiesFileCredentialsManager()
   {
     // Set the default location to get/put credentials
//     if( System.getProperty(SYSTEM_PROPERTY_CREDENTIALS) == null )
//     {
//        System.setProperty( SYSTEM_PROPERTY_CREDENTIALS, DEFAULT_CREDENTIALS_PATH );
//     }
     this.credentialsProperties = null;
     this.credentialsFile = null;
     this.keystoreFile = null;
   }
   
   /**
    * Creates a new credentials manager which loads credentials from the indicated file paths.
    * The location of either or both of the credentials or keystore files may be set. For both,
    * if the indicated path is set to null the default will be used.
    * <p>
    * The default path for both files is a subdirectory of the user's home. When a new credentials
    * property file is created the path of the keystore may be stored in it. When loading existing
    * credentials by default the keystore will be loaded from this stored path if it is available.
    * 
    * @param credentialsFilePath A custom path where the credentials property file is located;
    *                            Null to use the default path.
    * @param keystoreFilePath The custom path where the keystore file is located; Null to use
    *                            the previously stored path.
    */
   public PropertiesFileCredentialsManager(File credentialsFile, File keystoreFile)
   {
      this.credentialsProperties = null;
      this.credentialsFile = credentialsFile;
      this.keystoreFile = keystoreFile;
   }


   /**
    * Creates new credentials and saves them in the default location.
    * <p>
    * In addition to saving the username and password to file the grid account encryption keypair
    * will be generated and saved to a java keysore (JKS).
    * 
    * @param username The grid account username.
    * @param password The grid account password.
    * @param keyAlgorithm The key algorithm to use for the grid account encryption keypair.
    * @param keySize The key size to use for the grid account encryption keypair.
    * @param signatureAlgorithm The signature algorithm to use when creating the grid account
 *                           encryption certificate (used to store the public key in the key store).
    * @return A credentials manager using the newly created credentials files.
    * @throws CredentialsKeyException 
    * @throws CredentialsIOException 
    * @throws IOException If any credentials file could not be generated.
    * @throws KeyException If the keys could not be generated or stored in the key store.
    */
   public static PropertiesFileCredentialsManager create( 
         String username,
         String password,
         String keyAlgorithm,
         int keySize,
         String signatureAlgorithm ) throws CredentialsIOException, CredentialsKeyException
            
   {
      return create(
            null,
            null,
            username,
            password,
            keyAlgorithm,
            keySize,
            signatureAlgorithm );
   }
   
   
   /**
    * Creates new credentials and saves them in the given files.
    * <p>
    * In addition to saving the username and password to file the grid account encryption keypair
    * will be generated and saved to a java keysore (JKS).
    * 
    * @param credentialsPropertiesFile The file in which credentials information will be written.
    *    If null the credentials will be written to the default path.
    * @param keystoreFile The file where the key store will be saved. If null the key store
    *    will be written to the default path.
    * @param username The grid account username.
    * @param password The grid account password.
    * @param keyAlgorithm The key algorithm to use for the grid account encryption keypair.
    * @param keySize The key size to use for the grid account encryption keypair.
    * @param signatureAlgorithm The signature algorithm to use when creating the grid account
 *                           encryption certificate (used to store the public key in the key store).
    * @return A credentials manager using the newly created credentials files.
    * @throws IOException If any credentials file could not be generated.
    * @throws KeyException If the keys could not be generated or stored in the key store.
    */
   public static PropertiesFileCredentialsManager create(
         File credentialsPropertiesFile,
         File keystoreFile,
         String username,
         String password,
         String keyAlgorithm,
         int keySize,
         String signatureAlgorithm )
            throws CredentialsIOException, CredentialsKeyException
   {
      UUID account = UUIDGen.getUUIDFromUsername(username);
      String accountID = account.toString();
      
      
      // Create credentials properties file
      
      FileResolver resolver;
      try
      {
         resolver = new FileResolver(
               SYSTEM_PROPERTY_CREDENTIALS_PATH,
               new File(DEFAULT_CREDENTIALS_PATH + File.separator + accountID + CREDENTIALS_EXTENSION),
               credentialsPropertiesFile );
         // Make sure the parent directory is created
         resolver.mkdirs();
         
      }
      catch (ResolutionException e1)
      {
         throw new CredentialsIOException(e1);
      }
      
    
      OutputStream out;
      try
      {
         out = resolver.getOutputStream();
      }
      catch (ResolutionException e2)
      {
         throw new CredentialsIOException(e2);
      }
      
      Properties credentials = new Properties();
      credentials.setProperty(USERNAME_PROPERTY, username);
      credentials.setProperty(PASSWORD_PROPERTY, password);
      try
      {
         credentials.store( out, "" );
         out.close();
      }
      catch (IOException e1)
      {
         throw new CredentialsIOException(e1);
      }

      
      _logger.debug(
            "Grid account username and password written to credentials file: " +
            resolver.getFile().toString() );
      
      
      try
      {
         // Generate account key pair.
         
         KeyPairGenerator keygen = KeyPairGenerator.getInstance(keyAlgorithm);
         keygen.initialize(keySize);
         KeyPair accountKeyPair = keygen.generateKeyPair();
         
         _logger.debug("Grid account encryption keypair generated");

         
         Certificate certificate;
         try
         {
            certificate = generateCertificate( accountID, accountKeyPair, signatureAlgorithm );
         }
         catch (Exception e)
         {
            throw new CredentialsKeyException("Could not generate grid account encryption certificate");
         }
         
         // Create key store.
                
         resolver = new FileResolver(
            SYSTEM_PROPERTY_KEYSTORE_PATH,
            new File(DEFAULT_CREDENTIALS_PATH + File.separator + accountID + KEYSTORE_EXTENSION),
            keystoreFile );
         out = resolver.getOutputStream();
            
         KeyStore ks = KeyStore.getInstance(KEY_STORE_PROVIDER);
         ks.load(null, null);
         
         // The passphrase for both the keystore and keystore entries is hardcoded to "".
         // This implies "psuedo-plaintext" storage of keys
         
         ks.setKeyEntry(
               accountID.toString(),
               accountKeyPair.getPrivate(),
               "".toCharArray(),
               new Certificate[] { certificate } );
         
         ks.store( out, "".toCharArray() );
         out.close();
         
         _logger.debug( "Grid account encryption keypair stored in new key store" );
      }
      catch (ResolutionException e)
      {
         throw new CredentialsIOException(e);
      }
      catch (KeyStoreException e)
      {
         throw new CredentialsKeyException("Error creating new keystore: " + e.getMessage(), e);
      }
      catch (NoSuchAlgorithmException e)
      {
         throw new CredentialsKeyException("Error creating new keystore: " + e.getMessage(), e);
      }
      catch (CertificateException e)
      {
         throw new CredentialsKeyException("Error creating new keystore: " + e.getMessage(), e);
      }
      catch (IOException e)
      {
         throw new CredentialsIOException(e);
      }
      
      return new PropertiesFileCredentialsManager( credentialsPropertiesFile, keystoreFile );
   }
   
   public Credentials getCredentials(UUID accountUUID) throws CredentialsIOException
   { 
      try
      {  
         loadProperties(accountUUID);
      }
      catch(ResolutionException e)
      {
         throw new CredentialsIOException("Could not load credentials for account " + accountUUID, e);
      }
      PasswordCredentials cred = new PasswordCredentials();
      cred.setUsername(credentialsProperties.getProperty(USERNAME_PROPERTY));
      cred.setPassword(credentialsProperties.getProperty(PASSWORD_PROPERTY));
      return cred;
   }

   
   private Resolver getCredentialsResolver( UUID accountID ) throws ResolutionException
   {
      return new FileResolver(
            SYSTEM_PROPERTY_CREDENTIALS_PATH,
            new File(DEFAULT_CREDENTIALS_PATH + File.separator + accountID + CREDENTIALS_EXTENSION),
            this.credentialsFile );
   }
   
   private Resolver getKeystoreResolver( UUID accountID ) throws ResolutionException
   {
      return new FileResolver(
            SYSTEM_PROPERTY_KEYSTORE_PATH,
            new File(DEFAULT_CREDENTIALS_PATH + File.separator + accountID + KEYSTORE_EXTENSION),
            this.keystoreFile );
   }
   
   
   
   public KeyPair getAccountKeyPair(UUID accountID) throws CredentialsException
   {  
      String account = accountID.toString();
      try
      {  
         loadProperties( accountID );
         
         Resolver resolver = getKeystoreResolver( accountID );
         InputStream in = resolver.getInputStream();
         
         KeyStore keystore = KeyStore.getInstance("JKS");
         keystore.load( in, "".toCharArray() );
         in.close();
         
         if ( ! keystore.isKeyEntry(account) )
         {
            throw new CredentialsKeyException(
                  "Keystore does not contain keypair for account: " + account );
         }
         
         Certificate cert = keystore.getCertificate(account);
         PrivateKey privateKey;
         try
         {
            privateKey = (PrivateKey) keystore.getKey(account, "".toCharArray());
         }
         catch (ClassCastException e)
         {
            throw new CredentialsKeyException(
                  "Keystore does not contain proper key type (expected private key) for account" +
                  account );
         }
         catch (UnrecoverableKeyException e)
         {
            throw new CredentialsKeyException(
                  "Could not load private key from keystore for account " + account );
         }
         
         return new KeyPair( cert.getPublicKey(), privateKey );
      }
      catch(ResolutionException e)
      {
         throw new CredentialsIOException("Could not load keystore file for account " + account);
      }
      catch(IOException e)
      {
         _logger.error("Could not load keystore: " + e.getMessage());
         throw new CredentialsKeyException("Could not load keystore file for account " + account);
      }
      catch(KeyStoreException e)
      {
         _logger.error("Could not load keystore: " + e.getMessage());
         throw new CredentialsKeyException("Could not load keystore file for account " + account);
      }
      catch(CertificateException e)
      {
         _logger.error("Certificate error while loading keystore file: " + e.getMessage());
         throw new CredentialsKeyException("Could not load keystore file for account " + account);
      }
      catch(NoSuchAlgorithmException e)
      {
         _logger.error("Loading keystore requires unsupported algorithm: " + e.getMessage());
         throw new CredentialsKeyException("Could not load keystore file for account " + account);
      }
   }
   
   private void loadProperties(UUID accountID) throws ResolutionException, CredentialsIOException
   {
      _logger.debug("Loading credentials file");
      
      // If a custom path is set for credentials file, we will use that. Key store file path
      // is loaded from credentials file.
      if ( this.credentialsProperties == null )
      {
         Resolver resolver = getCredentialsResolver( accountID );
         InputStream propertiesFile = resolver.getInputStream();
         
         this.credentialsProperties = new Properties();
         try
         {
            this.credentialsProperties.load(propertiesFile);
            propertiesFile.close();
         }
         catch(IOException e)
         {
            throw new CredentialsIOException(e);
         }
      }
   }
   
   protected static Certificate generateCertificate( String accountID, KeyPair keypair, String sigAlg )
         throws Exception
   {
      Calendar date = Calendar.getInstance();
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( new X509Name("UID=" + accountID) );
      certgen.setPublicKey( keypair.getPublic() );
      certgen.setNotBefore( date.getTime() );
      date.add( Calendar.YEAR, CERTIFICATE_DURATION_YEARS );
      certgen.setNotAfter( date.getTime() );
      certgen.setSerialNumber( BigInteger.valueOf(1) );
      certgen.setIssuerDN( new X509Name("UID=" + accountID) );
      certgen.setSignatureAlgorithm( sigAlg );
      return certgen.generate(keypair.getPrivate(), "BC" );
   }

   public UUID getAccountIdentifier(String username) throws CredentialsException
   {
      return UUID.nameUUIDFromBytes(username.getBytes());
   }   
   
   
   
   
}


