//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: May 24, 2007
//---------------------

package org.cleversafe.layer.slicestore.remote;

import java.io.File;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.authenticators.PasswordAuthenticator;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.authentication.exceptions.AuthenticationException;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.network.mina.MinaAcceptor;
import org.cleversafe.layer.communication.network.mina.MinaConnector;
import org.cleversafe.layer.communication.policy.SimpleConnectorManager;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreTestBase;
import org.cleversafe.layer.slicestore.block.BlockFileSliceStoreInfo;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.serialization.ProtocolMessageFactory;
import org.cleversafe.serialization.asn1.ASN1GridProtocolMessageFactory;
import org.cleversafe.serialization.raw.RawGridProtocolMessageFactory;
import org.cleversafe.storage.ss.SliceServerConfiguration;
import org.cleversafe.storage.ss.SliceServerDaemon;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;

import static org.junit.Assert.*;

/**
 * Used to test the Remote Slice Store implementation. A SliceServer will be started during the
 * course of the test.
 * 
 * @see SliceStoreTestBase
 */
public class RemoteSliceStoreIntegrationTest extends SliceStoreTestBase
{
   private static Logger logger = Logger.getLogger(RemoteSliceStoreTest.class);

   private static int port = 7632;

   private static String CONFIGURATION_PATH =
         "test-data/org/cleversafe/layer/slicestore/RemoteSliceStoreTest/slice-server.xml";

   private static final String OUTPUT_PATH = "test-data/dsgrid/slice-server";

   private static SliceServerDaemon sliceServer = null;

   private static boolean wasStarted = false;

   public synchronized SliceStore getSliceStore() throws SliceStoreIOException
   {
      // Start server
      if (wasStarted == false)
      {
         wasStarted = true;
         oneTimeSetUp();

         // Ensure directory is empty before running test
         File dbFile = new File(OUTPUT_PATH); // Must be a directory
         deleteDirectory(dbFile);

      }

      // Create connection
      Connector conn =
            new MinaConnector("127.0.0.1", port, new SimpleConnectorManager(),
                  new ASN1GridProtocolMessageFactory());

      try
      {
         conn.ensureConnected();
      }
      catch (CommunicationIOException e)
      {
         throw new SliceStoreIOException("Couldn't establish network connection", e);
      }
      catch (final CommunicationConnectionException e)
      {
         throw new SliceStoreIOException("Couldn't establish network connection", e);
      }

      // Create credentials
      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("username");
      credentials.setPassword("password");

      // Authenticate connection
      PasswordAuthenticator authenticator = new PasswordAuthenticator(conn);
      authenticator.setCredentials(credentials);

      try
      {
         authenticator.login();
      }
      catch (AuthenticationException e)
      {
         fail("Authentication failed");
      }

      BlockFileSliceStoreInfo blockFileSliceStoreInfo = new BlockFileSliceStoreInfo();
      blockFileSliceStoreInfo.setSliceStoreType("block-file");
      blockFileSliceStoreInfo.setSync(false);

      UUID vaultID = UUID.nameUUIDFromBytes("RemoteSliceStore".getBytes());

      RemoteSliceStore remoteSliceStore =
            new RemoteSliceStore(vaultID, conn, blockFileSliceStoreInfo);
      remoteSliceStore.setCredentials(credentials);

      return remoteSliceStore;
   }

   @Before
   public void setUp() throws Exception
   {
      super.setUp();
   }

   @After
   public void tearDown() throws Exception
   {
      super.tearDown();
   }

   @BeforeClass
   public static void oneTimeSetUp()
   {
      System.setProperty("org.cleversafe.storage.ss.xml.configuration", CONFIGURATION_PATH);
      // Start remote store 
      logger.info("Starting slice servers " + "(baseport: " + port + ")");

      try
      {
         SliceServerConfiguration config = new SliceServerConfiguration();

         MinaAcceptor acceptor = new MinaAcceptor();
         acceptor.setPort(port);
         acceptor.setHost("127.0.0.1");
         
         ProtocolMessageFactory messageFactory = new RawGridProtocolMessageFactory();
         
         config.setAcceptor(acceptor);
         config.setProtocolMessageFactory(messageFactory);
         
         sliceServer.run();
         // Wait until the grid comes up.... 
         Thread.sleep(5000);
      }
      catch (Exception ex)
      {
         logger.warn("Unable to start slice server!", ex);
      }
   }

   @AfterClass
   public static void oneTimeTearDown()
   {
      try
      {
         sliceServer.shutdown();
      }
      catch (Exception ex)
      {
         logger.warn("Unable to stop sliceserver!", ex);
      }
   }
   protected void createStore()
   {
      try
      {
         this.sliceStore.createStore("block", 8192, 100*1024*1024, // 100MB
               vaultACL, null);
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }
   }
   private static boolean deleteDirectory(File dir)
   {
      if (dir.isDirectory())
      {
         String[] children = dir.list();
         for (int i = 0; i < children.length; i++)
         {
            boolean success = deleteDirectory(new File(dir, children[i]));

            if (!success)
            {
               return false;
            }
         }
      }

      // The directory is now empty so delete it
      return dir.delete();
   }

}
