//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 7, 2007
//---------------------

package org.cleversafe.layer.access;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.net.URI;
import java.util.Set;
import java.util.UUID;

import org.cleversafe.authentication.CredentialsManager;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceNotFoundException;
import org.cleversafe.layer.access.test.TestService;
import org.cleversafe.test.BaseTest;
import org.cleversafe.test.TestException;
import org.cleversafe.util.FileSystemUtils;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultManager;
import org.cleversafe.vault.VaultManagerTest;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public abstract class GridAccessManagerTest extends BaseTest
{
   private final static String SERVICE_TYPE_1 = "test";
   
   private final static String TEST_SERVICE_INTERFACE_HOST = "0.0.0.0";
   private final static int TEST_SERVICE_INTERFACE_PORT = 3260;
   
   private final static String SERVICE_NAME_A = "serviceA";
   private final static String SERVICE_NAME_B = "serviceB";
   
   private final static String VAULT_NAME_A = "vaultA";
   private final static String VAULT_NAME_B = "vaultB";
   private final static String VAULT_NAME_C = "vaultC";
   
   
   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
   }

   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
   }

   @Before
   public void setUp() throws Exception
   {
   }

   @After
   public void tearDown() throws Exception
   {
   }
   
   public File getInputDirectory(String testName) throws TestException
   {
      return new File(super.getInputDirectory() + File.separator + testName);
   }

   public File getOutputDirectory(String testName) throws TestException
   {
      return new File(super.getOutputDirectory() + File.separator + testName);
   }
   
   public String getVaultDescriptor(String vaultName, String testName) throws TestException
   {
      String sliceOutputPath = this.getOutputDirectory(testName).toURI().getPath() + "/slice-data";
      return VaultManagerTest.VAULT_DESCRIPTOR_STRING
         .replaceAll("00000000-0000-0000-0000-000000000000", vaultName.replaceAll(" ", ""))
         .replaceAll("output/slice-data", sliceOutputPath);
   }
   
   /**
    * Creates a vault with the given vault manager using
    * {@link VaultManagerTest#VAULT_DESCRIPTOR_STRING}.
    * 
    * @param manager A vault manager instance.
    * @param vaultName The vault name.
    * @return The vault identifier.
    * @throws Exception
    */
   private UUID createVault(URI directory, VaultManager manager, String vaultName, String testName)
         throws Exception
   {
      Tuple2<UUID, CredentialsManager> credentials = this.getCredentials(directory);
      return manager.createVault(
            vaultName, 
            this.getVaultDescriptor(vaultName, testName),
            credentials.getFirst(),
            credentials.getSecond());
   }
   
   private Vault loadVault(URI directory, VaultManager manager, UUID vaultIdentifier)
         throws Exception
   {
      Tuple2<UUID, CredentialsManager> credentials = this.getCredentials(directory);
      return manager.loadVault(
            vaultIdentifier,
            credentials.getFirst(),
            credentials.getSecond());
   }
   
   public abstract GridAccessManager getGridAccessManagerInstance();
   
   public abstract Tuple2<UUID, CredentialsManager> getCredentials(URI directory) throws Exception;


   @SuppressWarnings("unchecked")
   @Test
   public void testGetVaultIdentifiers() throws Exception
   {
      File output = this.getOutputDirectory("testGetVaultIdentifiers");
      FileSystemUtils.deleteDir(output);
      URI directory = output.toURI();
      
      System.out.println("Testing get vault identifiers at: " + output);
      
      GridAccessManager manager = this.getGridAccessManagerInstance().load(directory);
      
      // Create vaults A, B, and C
      
      UUID vaultA = this.createVault(
            directory, manager.getVaultManager(), VAULT_NAME_A, "testGetVaultIdentifiers");
      UUID vaultB = this.createVault(
            directory, manager.getVaultManager(), VAULT_NAME_B, "testGetVaultIdentifiers");
      UUID vaultC = this.createVault(
            directory, manager.getVaultManager(), VAULT_NAME_C, "testGetVaultIdentifiers");
      
      // Create a service interface
      
      ServiceInterface<TestService> serviceInterface =
         (ServiceInterface<TestService>)manager.createServiceInterface(
               SERVICE_TYPE_1,
               TEST_SERVICE_INTERFACE_HOST,
               TEST_SERVICE_INTERFACE_PORT,
               true );
      
      assertTrue("service interface did not start automatically on creation",
            serviceInterface.isRunning());
      
      // Create services for vaults A and B and add them to service interface
      
      Vault vault = this.loadVault(directory, manager.getVaultManager(), vaultA);
      Service service = new TestService().load(SERVICE_NAME_A, vault);
      serviceInterface.add((TestService)service);
      
      vault = this.loadVault(directory, manager.getVaultManager(), vaultB);
      service = new TestService().load(SERVICE_NAME_B, vault);
      serviceInterface.add((TestService)service);
      
      // Query loaded manager
      
      Set<UUID> ids = manager.getVaultIdentifiers(false);
      assertTrue("missing vault from returned list", ids.contains(vaultA));
      assertTrue("missing vault from returned list", ids.contains(vaultB));
      assertTrue("missing vault from returned list", ids.contains(vaultC));
      
      ids = manager.getVaultIdentifiers(true);
      assertTrue("missing vault from returned list", ids.contains(vaultA));
      assertTrue("missing vault from returned list", ids.contains(vaultB));
      
      manager.shutdown();
      
      // load new manager and check 
      
      manager = this.getGridAccessManagerInstance().load(directory);
      
      serviceInterface = 
         (ServiceInterface<TestService>) manager.getServiceInterface(SERVICE_TYPE_1);

      assertTrue("service interface did not start automatically on creation",
            serviceInterface.isRunning());
      
      // Query new manager
      
      ids = manager.getVaultIdentifiers(false);
      assertTrue("missing vault from returned list", ids.contains(vaultA));
      assertTrue("missing vault from returned list", ids.contains(vaultB));
      assertTrue("missing vault from returned list", ids.contains(vaultC));
      
      ids = manager.getVaultIdentifiers(true);
      assertTrue("missing vault from returned list", ids.contains(vaultA));
      assertTrue("missing vault from returned list", ids.contains(vaultB));
   }
   

   @SuppressWarnings("unchecked")
   @Test
   public void testCreateServiceInterface() throws Exception
   {
      File output = this.getOutputDirectory("testCreateServiceInterface");
      FileSystemUtils.deleteDir(output);
      URI directory = output.toURI();
      
      System.out.println("Testing create service interface at: " + output);
      
      GridAccessManager manager = this.getGridAccessManagerInstance();
      manager.load(directory);
      
      // Create a vault to use later
      
      UUID vaultIdentifier = this.createVault(
            directory, manager.getVaultManager(), VAULT_NAME_A, "testCreateServiceInterface");
      
      // Create service interface
      manager.createServiceInterface(
            SERVICE_TYPE_1,
            TEST_SERVICE_INTERFACE_HOST,
            TEST_SERVICE_INTERFACE_PORT,
            false);
      
      // Get service interface from manager
      ServiceInterface<TestService> serviceInterface = 
         (ServiceInterface<TestService>)manager.getServiceInterface(SERVICE_TYPE_1);
      
      assertEquals("service interface started incorrectly", false, serviceInterface.isRunning());
      assertEquals("incorrect field", TEST_SERVICE_INTERFACE_HOST, serviceInterface.getHost());
      assertEquals("incorrect field", TEST_SERVICE_INTERFACE_PORT, serviceInterface.getPort());
      assertEquals("incorrect field", SERVICE_TYPE_1, serviceInterface.getType());
      assertEquals("unexpected services", 0, serviceInterface.getServices().size());

      // Create a service and add it to service interface
      
      Vault vault = this.loadVault(directory, manager.getVaultManager(), vaultIdentifier);
      Service service = new TestService().load(SERVICE_NAME_A, vault);
      
      serviceInterface.add((TestService)service);
      
      // Reload grid access manager and check that service and service mapping still exists
      
      manager.shutdown();
      manager = this.getGridAccessManagerInstance().load(directory);
      serviceInterface = (ServiceInterface<TestService>)manager.getServiceInterface(SERVICE_TYPE_1);
      assertEquals("service interface started incorrectly", false, serviceInterface.isRunning());
      assertEquals("incorrect field", TEST_SERVICE_INTERFACE_HOST, serviceInterface.getHost());
      assertEquals("incorrect field", TEST_SERVICE_INTERFACE_PORT, serviceInterface.getPort());
      assertEquals("incorrect field", SERVICE_TYPE_1, serviceInterface.getType());
      assertEquals("unexpected services", 1, serviceInterface.getServices().size());
      
      service = serviceInterface.getService(SERVICE_NAME_A);
      // check that service is the SAME INSTANCE as the one contained by the service interface
      assertTrue("service instances not the same", 
            service == manager.getService(SERVICE_TYPE_1, SERVICE_NAME_A));
      
      TestService test = (TestService)service;
      assertEquals("incorrect field", SERVICE_TYPE_1, test.getType());
      assertEquals("incorrect field", SERVICE_NAME_A, test.getName());
   }

   @SuppressWarnings("unchecked")
   @Test
   public void testDeleteServiceInterface() throws Exception
   {
      File output = this.getOutputDirectory("testDeleteServiceInterface");
      FileSystemUtils.deleteDir(output);
      URI directory = output.toURI();
      
      System.out.println("Testing delete service interface at: " + output);
      
      GridAccessManager manager = this.getGridAccessManagerInstance();
      manager.load(directory);
      
      // Create a vault to use later
      
      UUID vaultIdentifier = this.createVault(
            directory, manager.getVaultManager(), VAULT_NAME_A, "testDeleteServiceInterface");
      
      // Create service interface
      manager.createServiceInterface(
            SERVICE_TYPE_1,
            TEST_SERVICE_INTERFACE_HOST,
            TEST_SERVICE_INTERFACE_PORT,
            false);
      
      // Get service interface from manager
      ServiceInterface<TestService> serviceInterface = 
         (ServiceInterface<TestService>)manager.getServiceInterface(SERVICE_TYPE_1);
      
      
      // Create a service and add it to service interface
      
      Vault vault = this.loadVault(directory, manager.getVaultManager(), vaultIdentifier);
      Service service = new TestService().load(SERVICE_NAME_A, vault);
      
      serviceInterface.add((TestService)service);
      
      // Reload grid access manager and try to delete the service interface without deleting
      // the service itself
      
      manager.shutdown();
      manager = this.getGridAccessManagerInstance().load(directory);
      manager.deleteServiceInterface(SERVICE_TYPE_1);
      
      try
      {
         manager.getServiceInterface(SERVICE_TYPE_1);
         fail("unexpectedly returned service interface object");
      }
      catch (ServiceInterfaceNotFoundException e) {}
      
      
      // Reload grid access manager and make sure that the service interface and service no longer
      // exist.
      
      manager.shutdown();
      manager = this.getGridAccessManagerInstance().load(directory);
      
      try
      {
         manager.getServiceInterface(SERVICE_TYPE_1);
         fail("service interface query succeeded unexpectedly");
      }
      catch (ServiceInterfaceNotFoundException e) {}
      
      try
      {
         manager.getService(SERVICE_TYPE_1, SERVICE_NAME_A);
         fail("service query succeeded unexpectedly");
      }
      catch (ServiceNotFoundException e) {}
   }

}


