package org.cleversafe.layer.block;


import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.List;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;
import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.layer.block.BlockDeviceVault;
import org.cleversafe.storage.ss.SliceServerDaemon;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.vault.FullAccountCreator;
import org.cleversafe.vault.Vault;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class BlockDevicePerformanceTest {

	   private static Logger _logger = Logger.getLogger(BlockDeviceUDCTest.class);
	   // Base port for locally-started slice servers
	   private static final int BASE_SERVER_PORT = 5001;
	   // How long the test should be run in minutes
	   private static int testDuration;
	   // I/O Request Size in KB
	   private static int requestSize;
	   // Number of concurrent requests
	   private static int numConcurrentRequests;
	   
	   // running total IOPs
	   int totalIOPS = 0;
	   // running total latency
	   long totalLatency = 0;
	   
	   // SliceServer object
	   private static SliceServerDaemon ss = null;

       public byte[] data;
       
	   Vault vault;
	   FullAccountCreator accountCreator;
	   BlockDeviceController bdc;
	   Random rng;

	   public synchronized void updateStats(long incr)
	   {
		   totalLatency +=   incr;
		   totalIOPS++;
	   }
	   
	   static SliceServerDaemon sliceServerGroup = null;

	   @BeforeClass
	   public static void runBeforeTests() throws Exception
	   {
	      // Start remote stores if necessary
	      String storeString = System.getProperty("org.cleversafe.ss.test.servercount");
	      int stores = storeString != null ? Integer.parseInt(storeString) : 0;

	      testDuration = Integer.getInteger("test.duration", 5); // 5 minutes
	      numConcurrentRequests = Integer.getInteger("test.concurrency", 1); //
	      requestSize = Integer.getInteger("test.requestsize", 4); // 4K

	      if (stores > 0)
	      {
	         _logger.debug("Starting " + stores + " slice servers " + "(base port: " + BASE_SERVER_PORT
	               + ")");

	         String[] args = new String[2];
	         args[0] = "--host=127.0.0.1";
	         args[1] = "--port=" + BASE_SERVER_PORT;

	         // cleanup earlier store
	         // File f = new File("test-data/dsgrid/*");
	         //BlockDevicePerformanceTest.ss = new SliceServerDaemon(args);
	         //ss.startSliceServers();
	         
	         // Wait until the grid comes up....
	         Thread.sleep(3000);
	      }
	   }

	   @AfterClass
	   public static void runAfterTests() throws Exception
	   {
	      if (BlockDevicePerformanceTest.ss != null)
	      {
	    	  BlockDevicePerformanceTest.ss.shutdown();
	      }
	   }
	   
	   @Before
	   public void setUp()
	   {
	      this.rng = new Random(); // Intentional repeatability

	      try
	      {
	         String vaultDescriptorPath =
	               System.getProperty("org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor");
	         assertNotNull(
	               "Vault descriptor must be provided (org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor)",
	               vaultDescriptorPath);

	         this.accountCreator = new FullAccountCreator(vaultDescriptorPath);
	         this.vault = this.accountCreator.getVault();

	         // Create controller
	         assertTrue(this.vault instanceof BlockDeviceVault);
	         this.bdc = new BlockDeviceController((BlockDeviceVault) this.vault);
	         this.bdc.startup();
	         
	         _logger.debug("Block device controller was started");
	         assertNotNull(this.bdc);
	         this.data = new byte[requestSize * 1024];
	      }
	      catch (Exception e)
	      {
	         _logger.error("Initialization error", e);
	         fail(e.getMessage());
	      }
	   }

	   @After
	   public void tearDown()
	   {
	      try
	      {
	         this.bdc.shutdown();

	         _logger.debug("Block device controller was shutdown");

	         this.accountCreator.cleanup();
	      }
	      catch (final Exception e)
	      {
	         _logger.warn("Exception on tearDown", e);
	      }

	      this.bdc = null;
	      this.rng = null;
	      this.accountCreator = null;
	   }
	   /**
	    * Measure average performance by repeatedly sending IO requests of specified
	    * size for the specified duration. Does not check correctness of data.
	    * 
	    * Outputs IOPS, Data rate (MB/s), latency (ms) 
	    * 
	    * @throws ExecutionException
	    * @throws InterruptedException
	    */
	   @Test
	   public void writePerformanceTest() throws InterruptedException, ExecutionException, Exception
	   {
	      // Make configurable # of blocks per request
	      final int blocksPerRequest = requestSize / bdc.getBlockSize();
	      final long bytesSent = requestSize * 1024;
	      ExecutorService executor = new BoundedThreadPoolExecutor("UDC Test", numConcurrentRequests);

	      /**
	       * Time read and write phases, results contain the time stamp of task.
	       */
	      List<Future<Long>> results = null;
	      long beginTime, duration;

	      /**
	       * Write requests
	       */

	      long endTime;
 

          BlockDevicePerformanceTest bdct = BlockDevicePerformanceTest.this;
          int numBlocks = (requestSize * 1024)/bdct.bdc.getBlockSize();
          System.out.println(numBlocks + " **** " + bdct.bdc.getBlockSize());
	      long startTime = System.currentTimeMillis();
	      long testEndTime = startTime + testDuration * 1000;
	      int seqNum = 1;
	      while( startTime < testEndTime)
	      {
	    	  totalIOPS++;
	    	  _logger.debug("Beginning write request: " + totalIOPS);

	    	  try
	    	  {
	    		  bdct.bdc.writeBlocks(seqNum-1, numBlocks, data);
	          }
	    	  catch (Exception e)
	    	  {
	    		  throw e;
	    	  }

	          _logger.debug("Write request complete: " + totalIOPS);
	          endTime = System.currentTimeMillis();
	          totalLatency += endTime - startTime;
	          startTime = endTime;
	      }

	      System.out.println(String.format("Write(IOPS) %.2f : Rate(MB/s) %.3f : Latency(ms) %.3f",
	    		  (totalIOPS * 1.0)/(testDuration ), (totalIOPS * requestSize * 1000.0)/(1024*totalLatency), (totalLatency * 1.0)/totalIOPS));
	      
		  _logger.debug("Begining Read operations.");
		  startTime =  System.currentTimeMillis();
		  int overflowCount = 0;
		  int maxBlocksWritten = totalIOPS;

		  testEndTime = startTime + testDuration * 1000;
	      while( startTime < testEndTime)
	      {
	    	  totalIOPS++;
	    	  _logger.debug("Beginning read request: " + totalIOPS);

	    	  try
	    	  {
	    		  if (totalIOPS >= maxBlocksWritten){
	    			  overflowCount++;
	    			  totalIOPS = 1;
	    		  }
	    		  data = bdct.bdc.readBlocks(totalIOPS-1, numBlocks);
	          }
	    	  catch (Exception e)
	    	  {
	    		  throw e;
	    	  }

	          _logger.debug("Read request complete: " + totalIOPS);
	          endTime = System.currentTimeMillis();
	          totalLatency += endTime - startTime;
	          startTime = endTime;
	      }

	      totalIOPS += overflowCount * maxBlocksWritten;
	      System.out.println(String.format("Read(IOPS) %.2f : Rate(MB/s) %.3f : Latency(ms) %.3f",
	    		  (totalIOPS*1.0)/(testDuration), ( (totalIOPS + overflowCount * maxBlocksWritten) * requestSize * 1000.0)/(1024*totalLatency), (totalLatency * 1.0)/totalIOPS));
	      
	      
	      
	   }
	
	   //@Test
	   public void writePerformanceTest_n() throws InterruptedException, ExecutionException, Exception
	   {
		      // Make configurable # of blocks per request
		      final int blocksPerRequest = requestSize / bdc.getBlockSize();
		      final long bytesSent = requestSize * 1024;
		      //byte[] data = new byte[requestSize * 1024];
		        
		      // ExecutorService executor = new BoundedThreadPoolExecutor("UDC Test", numConcurrentRequests);
		      ExecutorService executor = Executors.newFixedThreadPool(numConcurrentRequests);
		      abstract class Request implements Runnable
		      {
		         protected long requestNum;
		         protected long firstBlock;

		         public Request(long requestNum, long firstBlock)
		         {
		            this.requestNum = requestNum;
		            this.firstBlock = firstBlock;
		         }
		      }

		      class WriteRequest extends Request
		      {
		         public WriteRequest(long requestNum, long firstBlock)
		         {
		            super(requestNum, firstBlock);
		         }

		         public void run()
		         {
		            _logger.info("Beginning write request: " + this.requestNum);
		            BlockDevicePerformanceTest bdct = BlockDevicePerformanceTest.this;

		            try
		            {
		               long beginTime = System.currentTimeMillis();
		               bdct.bdc.writeBlocks(this.firstBlock, blocksPerRequest, bdct.data);
		               long latency = System.currentTimeMillis() - beginTime;
		               
		            	bdct.updateStats(latency);
		            	
		            }
		            catch (final Exception e)
		            {
		               //e.printStackTrace();
		               //System.out.println("INTERRUPTED");
		            }

		            _logger.info("Write request complete: " + this.requestNum);
		           
		         }
		      }

		      class ReadRequest extends Request
		      {
		         public ReadRequest(long requestNum, long firstBlock)
		         {
		            super(requestNum, firstBlock);
		         }

		         public void run()
		         {
		            _logger.debug("Beginning read request: " + this.requestNum);
		            BlockDevicePerformanceTest bdct = BlockDevicePerformanceTest.this;

		            try
		            {
		               bdct.bdc.readBlocks(this.firstBlock, blocksPerRequest);
		            }
		            catch (final Exception e)
		            {
		               e.printStackTrace();
		            }

		            _logger.debug("Read request complete: " + this.requestNum);
		            
		         }
		      }

		      /**
		       * Time read and write phases, results contain the time stamp of task.
		       */
		      /*
		      List<Future<Long>> results = null;
		      long beginTime, duration;
            */

		      /**
		       * Write requests
		       */
		     
		      _logger.info("Write Test Begin.");
		      long currentTime = System.currentTimeMillis();
		      long testEndTime =  currentTime + testDuration * 1000;
		      int seqNum = 0;
		      
	          BlockDevicePerformanceTest bdct = BlockDevicePerformanceTest.this;
	          int numBlocks = (requestSize * 1024)/bdct.bdc.getBlockSize();
	          long firstBlock = 1;
		      while( currentTime < testEndTime)
		      {
		    	  WriteRequest workItem = new WriteRequest( seqNum++, firstBlock);
		    	  executor.execute( workItem );
		    	  //workItem.run();
		    	  firstBlock += numBlocks;
		    	  currentTime = System.currentTimeMillis();
		    	  //System.out.println(totalIOPS + " CurrentTime=" + currentTime + " testEndTime=" + testEndTime);
			  }
		      _logger.info("Completed executor whileloop");
		      executor.shutdownNow();
		  
		      _logger.info("Write Test completed. completed Requests = " + totalIOPS + ", submitted =" + seqNum);
		      
		      System.out.println(String.format("Write(IOPS) %.2f : Rate(MB/s) %.3f : Latency(ms) %.3f",
		    		  (totalIOPS * 1000.0 * numConcurrentRequests)/(totalLatency), (totalIOPS * requestSize * 1000.0)/(1024*totalLatency), (totalLatency * 1.0)/totalIOPS));
		      
			  /*  
		      long endTime;
	          byte[] data = new byte[requestSize * 1024];
	          BlockDevicePerformanceTest bdct = BlockDevicePerformanceTest.this;
	          int numBlocks = (requestSize * 1024)/bdct.bdc.getBlockSize();
	          System.out.println(numBlocks + " **** " + bdct.bdc.getBlockSize());
		      long startTime = System.currentTimeMillis();
		      long testEndTime = startTime + testDuration * 60 * 1000;
		    
		      while( startTime < testEndTime)
		      {
		    	  totalIOPS++;
		    	  _logger.debug("Beginning write request: " + totalIOPS);

		    	  try
		    	  {
		    		  bdct.bdc.writeBlocks(totalIOPS-1, numBlocks, data);
		          }
		    	  catch (Exception e)
		    	  {
		    		  throw e;
		    	  }

		          _logger.debug("Write request complete: " + totalIOPS);
		          endTime = System.currentTimeMillis();
		          totalLatency += endTime - startTime;
		          startTime = endTime;
		      }

		      System.out.println(String.format("Write(IOPS) %.2f : Rate(MB/s) %.3f : Latency(ms) %.3f",
		    		  (totalIOPS * 1.0)/(testDuration), (totalIOPS * requestSize * 1000.0)/(1024*totalLatency), (totalLatency * 1.0)/totalIOPS));
    */
		   }
	}
