
package org.cleversafe.util.performance;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.extras.DOMConfigurator;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.codec.Codec;
import org.cleversafe.codec.Decoder;
import org.cleversafe.codec.Encoder;
import org.cleversafe.codec.exceptions.CodecDecodeException;
import org.cleversafe.codec.exceptions.CodecEncodeException;
import org.cleversafe.codec.exceptions.CodecInvalidDataFormatException;
import org.cleversafe.codec.exceptions.CodecInvalidParametersException;
import org.cleversafe.codec.exceptions.CodecNotInitializedException;
import org.cleversafe.codec.exceptions.CodecUnknownSizeChangeException;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.ida.InformationDispersalDecoder;
import org.cleversafe.ida.InformationDispersalEncoder;
import org.cleversafe.ida.exceptions.IDADecodeException;
import org.cleversafe.ida.exceptions.IDAEncodeException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;
import org.cleversafe.layer.block.BlockDeviceVault;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.exceptions.NotConnectedException;
import org.cleversafe.layer.communication.network.mina.MinaConnector;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.remote.RemoteSliceStore;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.util.Log4jReloader;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.XMLVaultLoader;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultKeyGenerationException;
import org.cleversafe.vault.storage.VaultKeyInfo;

public class PerformanceEstimator
{
   // Constants
   public static final int KB = 1024;
   public static final int MB = 1024 * KB;

   // iSCSI properties
   //public static int ISCSI_QUEUE_SIZE = 32;     // Number of outstanding requests allowed by ISCSI
   public static int BLOCKS_PER_REQUEST = 32; // Number of blocks written or read per request

   // Bandwidth properties (May be obtained using iperf and ping)
   //public static int THROUGHPUT_INITIATOR_TO_ACCESSER = 100*MB / 8;  // In Bytes/Second
   public static double THROUGHPUT_ACCESSER_TO_SLICESTOR = 1000 * MB / 8; // In Bytes/Second
   //public static double LATENCY_INITIATOR_TO_ACCESSER = 1;            // In Milliseconds
   public static double LATENCY_ACCESSER_TO_SLICESTOR = 1; // In Milliseconds

   // SliceStor properties
   public static int HARD_DRIVE_WRITE_SPEED = 50 * MB; // In Bytes/Second
   public static double HARD_DRIVE_SEEK_TIME = 5; // In Milliseconds

   public static final int NUM_CPUS = Runtime.getRuntime().availableProcessors();

   // Store encoded result for decoding tests
   private static List<byte[]> encodedSlices = new ArrayList<byte[]>();

   private static ExecutionContext createExecutionContext() throws VaultKeyGenerationException,
         VaultException
   {
      // Generate Key
      KeyPairGenerator keygen;
      try
      {
         keygen = KeyPairGenerator.getInstance("RSA");
      }
      catch (NoSuchAlgorithmException e)
      {
         throw new RuntimeException("Expected algorithm RSA");
      }
      keygen.initialize(512);
      KeyPair keyPair = keygen.generateKeyPair();

      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("account_creator_");
      credentials.setPassword("password");

      // Creating the Vault ACL
      VaultACLFactory fact = new VaultACLFactory();
      ByteArrayOutputStream out = new ByteArrayOutputStream();
      fact.create(out, UUID.randomUUID(), credentials, keyPair.getPublic(),
            new ArrayList<VaultKeyInfo>());
      ByteArrayInputStream in = new ByteArrayInputStream(out.toByteArray());
      VaultACL acl = fact.getInstance(in);

      // Clear memory
      out.reset();
      in.reset();

      // Create the execution context for vault loading
      ExecutionContext ctx = new ExecutionContext();
      ctx.add(VaultDescriptor.ACL_CTX_STRING, acl);
      ctx.add(VaultDescriptor.ACL_ENTRY_CTX_STRING, acl.getEntry(acl.getOwner(),
            keyPair.getPrivate()));
      ctx.add(VaultDescriptor.CREDENTIALS_CTX_STRING, new PasswordCredentials() /* this.credentials */);
      return ctx;
   }

   private static Vault setupVault()
   {
      try
      {
         String vaultDescriptorFileName =
               System.getProperty("org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor");
         String storageMap =
               System.getProperty("org.cleversafe.block.BlockDeviceUDCTest.storageMap");
         assertNotNull(
               "Vault descriptor must be provided (org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor)",
               vaultDescriptorFileName);

         ExecutionContext ctx = createExecutionContext();

         // Load the vault

         Vault vault =
               new XMLVaultLoader().loadVaultDescriptor(
                     new FileInputStream(vaultDescriptorFileName), ctx).createVaultObject();

         // FIXME: Avoid using this code, it will create a vault on the slicestors, we do not
         // want this test to create any files that would need to be cleaned up later.  The
         // following is only needed to test vaults that require encryption keys because the above
         // code for creating a vault does not create keys needed for the vault.
         /*
          * FullAccountCreator accountCreator = new FullAccountCreator(vaultDescriptorFileName);
          * Vault vault = accountCreator.getVault();
          */
         return vault;
      }
      catch (Exception e)
      {
         e.printStackTrace();
         System.err.println("Unable to create vault.");
         System.exit(-1);
         return null;
      }
   }

   /**
    * Sends small message to each server, finds the one with the largest RTT and returns it
    * 
    * @param sliceStores
    * @return
    * @throws CommunicationIOException
    * @throws CommunicationConnectionException
    * @throws NotConnectedException
    * @throws CommunicationInterruptedException
    * @throws CommunicationResponseException
    * @throws CommunicationTransmissionException
    */
   private static double benchmarkLatency(List<SliceStore> sliceStores)
         throws CommunicationIOException, CommunicationConnectionException, NotConnectedException,
         CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException
   {
      final int numRequests = 30;
      long maxLatency = 0;

      for (SliceStore store : sliceStores)
      {
         if (store instanceof RemoteSliceStore)
         {
            RemoteSliceStore remoteStore = (RemoteSliceStore) store;
            Connector conn = remoteStore.getConnection();

            conn.connect();
            conn.ensureConnected();

            NoopRequest req = new NoopRequest();

            long beginTime = System.currentTimeMillis();
            for (int i = 0; i < numRequests; i++)
            {
               conn.exchange(req);
            }
            long duration = (System.currentTimeMillis() - beginTime) / numRequests;

            if (duration > maxLatency)
            {
               maxLatency = duration;
            }

            conn.disconnect();
         }
      }

      return maxLatency;
   }

   /**
    * Determine the amount of throughput that exists between the accesser and slicestors by sending
    * large messages in many threads.
    * 
    * @param sliceStores
    * @return
    * @throws CommunicationIOException
    * @throws CommunicationConnectionException
    * @throws NotConnectedException
    * @throws CommunicationInterruptedException
    * @throws CommunicationResponseException
    * @throws CommunicationTransmissionException
    * @throws InterruptedException
    * @throws ExecutionException
    */
   private static double benchmarkAccesserThroughput(List<SliceStore> sliceStores)
         throws CommunicationIOException, CommunicationConnectionException, NotConnectedException,
         CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException, InterruptedException, ExecutionException
   {
      final int messageSize = 4 * KB * BLOCKS_PER_REQUEST;
      final int messagesSent = 1000;
      final AtomicInteger messagesLeft = new AtomicInteger(messagesSent);
      final AtomicInteger actualMessagesSent = new AtomicInteger(0);
      final int numThreads = 200;

      byte[] payload = new byte[messageSize];

      // Create set of tasks, each one represents a slicestor to which we will send as much data as possible
      List<Callable<Void>> tasks = new ArrayList<Callable<Void>>(sliceStores.size());

      for (SliceStore store : sliceStores)
      {
         if (store instanceof RemoteSliceStore)
         {
            RemoteSliceStore remoteStore = (RemoteSliceStore) store;
            Connector conn = remoteStore.getConnection();

            conn.connect();
            conn.ensureConnected();

            class MultiplexedTransmission implements Callable<Void>
            {
               private Request _req = null;
               private MinaConnector _conn = null;
               private AtomicInteger _messagesLeft = null;
               private AtomicInteger _actualSent = null;

               public MultiplexedTransmission(
                     Request req,
                     MinaConnector conn,
                     AtomicInteger messagesLeft,
                     AtomicInteger actualSent)
               {
                  this._req = req;
                  this._conn = conn;
                  this._messagesLeft = messagesLeft;
                  this._actualSent = actualSent;
               }

               public Void call() throws CommunicationIOException, NotConnectedException,
                     CommunicationInterruptedException, CommunicationResponseException,
                     CommunicationTransmissionException
               {
                  // As long as there are message to be sent, keep sending them
                  while (this._messagesLeft.decrementAndGet() > 0)
                  {
                     this._conn.exchange(this._req);
                     this._actualSent.incrementAndGet();
                  }
                  return null;
               }
            }

            // Add task
            for (int t = 0; t<numThreads; t++)
            {
               NoopRequest req = new NoopRequest();
               req.setPayload(payload);
               tasks.add(t, new MultiplexedTransmission((Request) req, (MinaConnector) conn, messagesLeft, actualMessagesSent));
            }
         }
      }

      // Begin test sending messages to all slicestors simultaneously
      ExecutorService executor =
            new BoundedThreadPoolExecutor("accesser-throughput-test", numThreads);
      List<Future<Void>> results = null;

      long beginTime = System.currentTimeMillis();

      // FIXME: We need to add a non-echoing NoOp Request, perhaps
      // have a separate EchoRequest and NoOpRequest, otherwise we
      // may get inaccurate results here

      try
      {
         results = executor.invokeAll(tasks);
      }
      catch (final InterruptedException e)
      {
         fail(e.getMessage());
      }

      // When invokeAll returns, all tasks have been executed
      long duration = System.currentTimeMillis() - beginTime;

      for (Future<Void> result : results)
      {
         result.get();
      }

      // Disconnect connections
      for (SliceStore store : sliceStores)
      {
         if (store instanceof RemoteSliceStore)
         {
            RemoteSliceStore remoteStore = (RemoteSliceStore) store;
            Connector conn = remoteStore.getConnection();
            conn.disconnect();
         }
      }

      executor.shutdown();

      // Reform transmission time to a double of bytes / seconds
      double bytesSent = messageSize * actualMessagesSent.get();
      return (bytesSent / (duration / 1000.0));
   }

   /**
    * Determine the amount of throughput that exists between the accesser and slicestors by sending
    * large messages in many threads.
    * 
    * @param sliceStores
    * @return
    * @throws CommunicationIOException
    * @throws CommunicationConnectionException
    * @throws NotConnectedException
    * @throws CommunicationInterruptedException
    * @throws CommunicationResponseException
    * @throws CommunicationTransmissionException
    * @throws InterruptedException
    * @throws ExecutionException
    */
   private static List<Double> benchmarkSlicestorThroughput(List<SliceStore> sliceStores)
         throws CommunicationIOException, CommunicationConnectionException, NotConnectedException,
         CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException, InterruptedException, ExecutionException
   {
      final int messageSize = 4 * KB * BLOCKS_PER_REQUEST;
      final int numRequests = 100;
      final int numThreads = 200;

      byte[] payload = new byte[messageSize];

      List<Double> serverThroughputs = new ArrayList<Double>(sliceStores.size());

      for (SliceStore store : sliceStores)
      {
         if (store instanceof RemoteSliceStore)
         {
            RemoteSliceStore remoteStore = (RemoteSliceStore) store;
            Connector conn = remoteStore.getConnection();

            conn.connect();
            conn.ensureConnected();

            ExecutorService executor =
                  new BoundedThreadPoolExecutor("slicestor-throughput-test", numThreads);

            class Transmission implements Callable<Void>
            {
               private Request _req = null;
               private MinaConnector _conn = null;

               public Transmission(Request req, MinaConnector conn)
               {
                  this._req = req;
                  this._conn = conn;
               }

               public Void call() throws CommunicationIOException, NotConnectedException,
                     CommunicationInterruptedException, CommunicationResponseException,
                     CommunicationTransmissionException
               {
                  this._conn.exchange(this._req);
                  return null;
               }
            }

            List<Callable<Void>> tasks = new ArrayList<Callable<Void>>(numRequests);
            for (int i = 0; i < numRequests; i++)
            {
               NoopRequest req = new NoopRequest();
               req.setPayload(payload);
               tasks.add(i, new Transmission((Request) req, (MinaConnector) conn));
            }

            long beginTime = System.currentTimeMillis();

            // FIXME: We need to add a non-echoing NoOp Request, perhaps
            // have a separate EchoRequest and NoOpRequest, otherwise we
            // may get inaccurate results here
            List<Future<Void>> results = null;
            try
            {
               results = executor.invokeAll(tasks);
            }
            catch (final InterruptedException e)
            {
               fail(e.getMessage());
            }

            // When invokeAll returns, all tasks have been executed
            long duration = System.currentTimeMillis() - beginTime;

            for (Future<Void> result : results)
            {
               result.get();
            }

            conn.disconnect();

            executor.shutdown();

            // Save this server's throughput in a list
            double bytesSent = messageSize * numRequests;
            double throughput = (bytesSent / (duration / 1000.0));
            serverThroughputs.add(throughput);
         }
      }

      // Reform transmission time to a double of bytes / seconds
      return serverThroughputs;
      
   }

   /**
    * Returns performance in bytes/second of a stack of encoders
    * 
    * @param encoders
    * @return
    * @throws CodecInvalidParametersException
    * @throws CodecNotInitializedException
    * @throws CodecInvalidDataFormatException
    * @throws CodecEncodeException
    * @throws IDANotInitializedException
    * @throws IDAEncodeException
    */
   private static double benchmarkEncoding(
         List<Encoder> dataEncoders,
         InformationDispersalEncoder idaEncoder,
         List<Encoder> sliceEncoders,
         int blockSize) throws CodecInvalidParametersException, CodecEncodeException,
         CodecInvalidDataFormatException, CodecNotInitializedException, IDAEncodeException,
         IDANotInitializedException
   {
      // Create block
      final byte[] block = new byte[blockSize];
      Random rnd = new Random();
      rnd.nextBytes(block);

      // Test approximately 10 MB
      int numBlocks = (10 * MB) / blockSize;
      int dataSize = numBlocks * blockSize;

      // Make up source name and transaction Id
      SourceName sourceName = new SourceName("1");
      int transactionId = 0;

      // Save time
      long startTime = System.currentTimeMillis();

      // Simulate coding stack on multiple blocks
      for (int cnt = 0; cnt < numBlocks; cnt++)
      {

         byte[] data = block;

         // Encode with data source codecs
         for (Encoder encoder : dataEncoders)
         {
            encoder.reset(sourceName, transactionId);
            data = encoder.finish(data);
         }

         // Encode with IDA
         List<byte[]> slices = idaEncoder.finish(data);

         // Encode slices with slice codecs
         for (int sliceIndex = 0; sliceIndex < slices.size(); sliceIndex++)
         {
            byte[] sliceData = slices.get(sliceIndex);

            for (Encoder encoder : sliceEncoders)
            {
               encoder.reset(sourceName, transactionId);
               sliceData = encoder.finish(sliceData);
            }

            if (cnt == 0)
            {
               encodedSlices.add(sliceData);
            }
         }
      }

      long endTime = System.currentTimeMillis();

      long testTime = endTime - startTime;

      double bytesPerSecond = ((float) (dataSize)) / (((float) testTime) / 1000.0);

      return bytesPerSecond * NUM_CPUS; // Assumes zero overhead of multithreading
   }

   /**
    * 
    * @param dataDecoders
    * @param idaDecoder
    * @param sliceDecoders
    * @param slices
    * @param blockSize
    * @return
    * @throws CodecInvalidParametersException
    * @throws CodecEncodeException
    * @throws CodecInvalidDataFormatException
    * @throws CodecNotInitializedException
    * @throws IDAEncodeException
    * @throws IDANotInitializedException
    * @throws IDADecodeException
    * @throws CodecDecodeException
    */
   private static double benchmarkDecoding(
         List<Decoder> dataDecoders,
         InformationDispersalDecoder idaDecoder,
         List<Decoder> sliceDecoders,
         List<byte[]> slices,
         int blockSize) throws CodecInvalidParametersException, CodecEncodeException,
         CodecInvalidDataFormatException, CodecNotInitializedException, IDAEncodeException,
         IDANotInitializedException, IDADecodeException, CodecDecodeException
   {
      int numSlices = slices.size();
      List<byte[]> decodedSlices = new ArrayList<byte[]>(numSlices);

      // Test approximately 10 MB
      int numBlocks = (10 * MB) / blockSize;
      int dataSize = numBlocks * blockSize;

      // Make up source name and transaction Id
      SourceName sourceName = new SourceName("1");
      int transactionId = 0;

      // Save time
      long startTime = System.currentTimeMillis();

      // Simulate coding stack on multiple blocks
      for (int cnt = 0; cnt < numBlocks; cnt++)
      {
         // Decode slices with slice codecs
         decodedSlices.clear();
         for (int sliceIndex = 0; sliceIndex < slices.size(); sliceIndex++)
         {
            byte[] sliceData = slices.get(sliceIndex);

            for (Decoder decoder : sliceDecoders)
            {
               decoder.reset(sourceName, transactionId);
               sliceData = decoder.finish(sliceData);
            }

            decodedSlices.add(sliceData);
         }

         // Decode with IDA
         byte[] encodedBlock = idaDecoder.finish(decodedSlices);

         // Decode with data source codecs
         for (Decoder decoder : dataDecoders)
         {
            decoder.reset(sourceName, transactionId);
            encodedBlock = decoder.finish(encodedBlock);
         }

      }

      long endTime = System.currentTimeMillis();

      long testTime = endTime - startTime;

      double bytesPerSecond = ((float) (dataSize)) / (((float) testTime) / 1000.0);

      return bytesPerSecond * NUM_CPUS; // Assumes zero overhead of multithreading
   }

   private static double calculateBlowUp(Vault vault, int blockSize, int numSlices)
         throws CodecUnknownSizeChangeException
   {
      long dispersedSize = blockSize;

      // Get data source size after codecs
      for (Codec codec : vault.getDatasourceCodecs())
      {
         dispersedSize = codec.getEncodedSize(dispersedSize);
      }

      dispersedSize = vault.getInformationDispersalCodec().getDispersedSize(dispersedSize);

      dispersedSize /= numSlices;

      for (Codec codec : vault.getSliceCodecs())
      {
         dispersedSize = codec.getEncodedSize(dispersedSize);
      }

      dispersedSize *= numSlices;

      return ((double) dispersedSize) / ((double) blockSize);
   }

   /**
    * Calculates the number of requests that can be satisfied per second and the amount of data
    * written in each request. Returns the bytes per second that can be written.
    * 
    * @param blowUp
    * @param blockSize
    * @param randomAccess
    * @return
    */
   private static double estimateFullStackThroughput(
         double blowUp,
         int blockSize,
         boolean randomAccess,
         int numSlicestors)
   {
      double networkLatencyInSeconds = ((double) LATENCY_ACCESSER_TO_SLICESTOR) / 1000.0;
      double hardDriveSeekTimeInSeconds = ((double) HARD_DRIVE_SEEK_TIME) / 1000.0;

      double beginTransactionTime = networkLatencyInSeconds;

      double rawDataLength = BLOCKS_PER_REQUEST * blockSize; // Useful data written
      double writeLength = BLOCKS_PER_REQUEST * blockSize * blowUp; // Amount written after expansion

      double writeSendTime = writeLength / ((double) THROUGHPUT_ACCESSER_TO_SLICESTOR);
      double writeRequestTime = networkLatencyInSeconds + writeSendTime;

      double writtenPerSliceStore = writeLength / numSlicestors;
      double seekTime =
            randomAccess
                  ? (BLOCKS_PER_REQUEST * hardDriveSeekTimeInSeconds)
                  : hardDriveSeekTimeInSeconds;
      double hardDriveWriteTime = writtenPerSliceStore / ((double) HARD_DRIVE_WRITE_SPEED);
      double commitTransactionTime = networkLatencyInSeconds + seekTime + hardDriveWriteTime;

      double requestTime = beginTransactionTime + writeRequestTime + commitTransactionTime;

      return (rawDataLength / requestTime);
   }

   /**
    * Calculates the amount of data that can be written to all the slicestor hard drives given their
    * write speed and the number of seeks that must be done. Includes blow up and the number of
    * slicestors to see how much raw data can be written. Returns the number of raw bytes per second
    * that can be written.
    * 
    * @param blowUp
    * @param blockSize
    * @param randomAccess
    * @return
    */
   private static double estimateHarddriveThroughput(
         double blowUp,
         int blockSize,
         boolean randomAccess,
         int numSlicestors)
   {
      double hardDriveSeekTimeInSeconds = ((double) HARD_DRIVE_SEEK_TIME) / 1000.0;

      double rawDataLength = BLOCKS_PER_REQUEST * blockSize; // Useful data written
      double writeLength = rawDataLength * blowUp; // Amount written after expansion
      double writtenPerSliceStore = writeLength / numSlicestors; // Amount of data each slicestor writes

      double seekTime =
            randomAccess
                  ? (BLOCKS_PER_REQUEST * hardDriveSeekTimeInSeconds)
                  : hardDriveSeekTimeInSeconds;
      double hardDriveWriteTime = writtenPerSliceStore / ((double) HARD_DRIVE_WRITE_SPEED);
      double totalWriteTime = seekTime + hardDriveWriteTime;

      return (rawDataLength / totalWriteTime);
   }

   /**
    * Calculates the amount of data that can be written over the network. Includes blow up in the
    * calculation. Returns the number of raw bytes per second that can be written.
    * 
    * @param blowUp
    * @return
    */
   private static double estimateNetworkThroughput(double blowUp)
   {
      return (THROUGHPUT_ACCESSER_TO_SLICESTOR / blowUp);
   }

   private static void loadProperties()
   {
      // Get accesser network throughput
      //String throughputAccesserToSlicestor = 
      //   System.getProperty("org.cleversafe.performanceestimator.networkthroughput", Integer.toString(THROUGHPUT_ACCESSER_TO_SLICESTOR));   
      //THROUGHPUT_ACCESSER_TO_SLICESTOR = Integer.parseInt(throughputAccesserToSlicestor);

      // Get accesser network latency
      //String latencyAccesserToSlicestor = 
      //   System.getProperty("org.cleversafe.performanceestimator.networklatency", Double.toString(LATENCY_ACCESSER_TO_SLICESTOR));   
      //LATENCY_ACCESSER_TO_SLICESTOR = Double.parseDouble(latencyAccesserToSlicestor);

      // Get harddrive write throughput
      String throughputSlicestorHarddrive =
            System.getProperty("org.cleversafe.performanceestimator.harddrivethroughput",
                  Integer.toString(HARD_DRIVE_WRITE_SPEED));
      HARD_DRIVE_WRITE_SPEED = Integer.parseInt(throughputSlicestorHarddrive);

      // Get harddrive seek time
      String latencySlicestorHarddrive =
            System.getProperty("org.cleversafe.performanceestimator.harddriveseektime",
                  Double.toString(HARD_DRIVE_SEEK_TIME));
      HARD_DRIVE_SEEK_TIME = Double.parseDouble(latencySlicestorHarddrive);

   }

   /**
    * Program will take a vault descriptor path as an argument
    * 
    * @param args
    */
   public static void main(String[] args) throws Exception
   {
      // Initialize logging
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));
      Log4jReloader.launch();

      // Load properties
      loadProperties();

      // Create vault object from descriptor
      Vault vault = setupVault();
      System.out.println("Calculating theoretical maximum performance for vault:");
      System.out.println(vault);
      System.out.println("");

      int vaultWidth = vault.getInformationDispersalCodec().getNumSlices();
      int blockSize = ((BlockDeviceVault) vault).getBlockSize();

      //////////////////////////////////////////////////////////////////////////////////////////////
      // Benchmark Network
      LATENCY_ACCESSER_TO_SLICESTOR = benchmarkLatency(vault.getSliceStores());
      System.out.println("Maximum Server Latency:    " + LATENCY_ACCESSER_TO_SLICESTOR + " ms");
      THROUGHPUT_ACCESSER_TO_SLICESTOR = benchmarkAccesserThroughput(vault.getSliceStores());
      System.out.println("Accesser to Server Throughput: "
            + (8 * THROUGHPUT_ACCESSER_TO_SLICESTOR / MB) + " Mbps");
      
      List<Double> serverThroughputs = benchmarkSlicestorThroughput(vault.getSliceStores());
      double minServerThroughput = Double.MAX_VALUE;
      for (int c = 0; c < serverThroughputs.size(); c++)
      {
         double throughput = serverThroughputs.get(c);
         if (throughput < minServerThroughput)
            minServerThroughput = throughput;
         System.out.println("Slicestor #" + c + "'s throughput: " + (8 * throughput / MB) + " Mbps");
      }
      
      System.out.println("Minimum Slicestor Throughput: " + (8 * minServerThroughput / MB) + " Mbps");
      
      double parallelSlicestorThroughput = (vaultWidth * minServerThroughput);
      System.out.println("Parallelized Slicestor Throughput: " + (8 * parallelSlicestorThroughput / MB) + " Mbps");
      
      if (THROUGHPUT_ACCESSER_TO_SLICESTOR < parallelSlicestorThroughput)
      {
         System.out.println("Network bottleneck: Accessor Link");
      }
      else
      {
         System.out.println("Network bottleneck: Parallel Slicestor Writes");
         THROUGHPUT_ACCESSER_TO_SLICESTOR = parallelSlicestorThroughput; // Use the minimum value for furuture calculations
      }
      
      System.out.println("");

      //////////////////////////////////////////////////////////////////////////////////////////////
      // Display ideal blowup
      System.out.println("Ideal IDA expansion:                  " + ((float) vaultWidth)
            / (float) vault.getThreshold());

      // Calculate and display data expansion factor
      double blowUp = calculateBlowUp(vault, blockSize, vaultWidth);
      System.out.println("Blowup after encoding:                " + blowUp);

      // Calculate blowup after storage on slicestors
      final int SLICE_FILE_HEADER_LENGTH = 16;
      double sliceStoreBlowup =
            ((blockSize * blowUp) + (vaultWidth * SLICE_FILE_HEADER_LENGTH)) / blockSize;
      System.out.println("Blowup after storage on slicestors:   " + sliceStoreBlowup);
      long GB = MB * KB;
      System.out.println("Storage requirement for 1 GB of data: "
            + ((int) (GB * sliceStoreBlowup) / MB) + " MB");

      System.out.println("");
      //////////////////////////////////////////////////////////////////////////////////////////////

      // Store set of data encoders to use
      List<Encoder> dataEncoders = new ArrayList<Encoder>();
      for (Codec codec : vault.getDatasourceCodecs())
      {
         dataEncoders.add(codec.getEncoder());
      }

      // Store set of slice encoders to use
      List<Encoder> sliceEncoders = new ArrayList<Encoder>();
      for (Codec codec : vault.getSliceCodecs())
      {
         sliceEncoders.add(codec.getEncoder());
      }

      // Reverse Data Codec list and get decoders
      List<Decoder> dataDecoders = new ArrayList<Decoder>();
      List<Codec> reverseDataCodecs = new ArrayList<Codec>(vault.getDatasourceCodecs());
      Collections.reverse(reverseDataCodecs);
      for (Codec codec : reverseDataCodecs)
      {
         dataDecoders.add(codec.getDecoder());
      }

      // Reverse Slice Codec List and get decoders
      List<Decoder> sliceDecoders = new ArrayList<Decoder>();
      List<Codec> reverseSliceCodecs = new ArrayList<Codec>(vault.getSliceCodecs());
      Collections.reverse(reverseSliceCodecs);
      for (Codec codec : reverseSliceCodecs)
      {
         sliceDecoders.add(codec.getDecoder());
      }

      // Get IDA's encoder and decoder
      InformationDispersalEncoder idaEncoder = vault.getInformationDispersalCodec().getEncoder();
      InformationDispersalDecoder idaDecoder = vault.getInformationDispersalCodec().getDecoder();

      //////////////////////////////////////////////////////////////////////////////////////////////
      // Output performance of the complete encoding stack
      double encodeStackPerformance =
            benchmarkEncoding(dataEncoders, idaEncoder, sliceEncoders, blockSize);
      System.out.println("Encoding stack performance (Mbps): "
            + (8.0 * encodeStackPerformance / MB));

      // Output performance of the complete decoding stack
      double decodeStackPerformance =
            benchmarkDecoding(dataDecoders, idaDecoder, sliceDecoders, encodedSlices, blockSize);
      System.out.println("Decoding stack performance (Mbps): "
            + (8.0 * decodeStackPerformance / MB));

      System.out.println("");

      // Calculate throughput of slicestor drives
      double harddriveRandom =
            estimateHarddriveThroughput(sliceStoreBlowup, blockSize, true, vaultWidth);
      System.out.println("Slicestor Harddrive Limit (random access) Mbps:     " + 8.0
            * harddriveRandom / MB);
      double harddriveSequential =
            estimateHarddriveThroughput(sliceStoreBlowup, blockSize, false, vaultWidth);
      System.out.println("Slicestor Harddrive Limit (sequential access) Mbps: " + 8.0
            * harddriveSequential / MB);
      //////////////////////////////////////////////////////////////////////////////////////////////

      System.out.println("");

      //////////////////////////////////////////////////////////////////////////////////////////////
      // Estimate network throughput
      double networkThroughput = estimateNetworkThroughput(blowUp);
      System.out.println("Network throughput limit writing (Mbps): " + (8 * networkThroughput / MB));
      System.out.println("Network throughput limit reading (Mbps): "
            + (8 * THROUGHPUT_ACCESSER_TO_SLICESTOR / MB));
      //////////////////////////////////////////////////////////////////////////////////////////////

      System.out.println("");

      //////////////////////////////////////////////////////////////////////////////////////////////
      // Estimate full stack throughput
      double fullWriteStack = estimateFullStackThroughput(blowUp, blockSize, false, vaultWidth);
      System.out.println("Per thread write throughput (Mbps): " + (8 * fullWriteStack / MB));
      System.out.println("20 thread write throughput (Mbps):  " + (20 * 8 * fullWriteStack / MB));
      //////////////////////////////////////////////////////////////////////////////////////////////

      System.out.println("");

      //////////////////////////////////////////////////////////////////////////////////////////////
      // Determine the write bottleneck and display it
      String bottleNeck = "Encoding Stack";
      double maxWriteSpeed = encodeStackPerformance;

      if (maxWriteSpeed > harddriveSequential)
      {
         bottleNeck = "Slicestor Harddrive";
         maxWriteSpeed = harddriveSequential;
      }

      if (maxWriteSpeed > networkThroughput)
      {
         bottleNeck = "Network Throughput";
         maxWriteSpeed = networkThroughput;
      }

      System.out.println("System's Write Bottleneck:      " + bottleNeck);
      System.out.println("Maximum sequential write speed: " + (maxWriteSpeed / MB) + " MB/s");
      //////////////////////////////////////////////////////////////////////////////////////////////

      System.out.println("");

      //////////////////////////////////////////////////////////////////////////////////////////////
      // Determine the read bottleneck and display it
      bottleNeck = "Decoding Stack";
      double maxReadSpeed = decodeStackPerformance;

      if (maxReadSpeed > harddriveSequential)
      {
         bottleNeck = "Slicestor Harddrive";
         maxReadSpeed = harddriveSequential;
      }

      if (maxReadSpeed > THROUGHPUT_ACCESSER_TO_SLICESTOR)
      {
         bottleNeck = "Network Throughput";
         maxReadSpeed = THROUGHPUT_ACCESSER_TO_SLICESTOR;
      }

      System.out.println("System's Read Bottleneck:       " + bottleNeck);
      System.out.println("Maximum sequential read speed:  " + (maxReadSpeed / MB) + " MB/s");
      //////////////////////////////////////////////////////////////////////////////////////////////

      // TODO: Possibly obtain a sample for network requests
      // Send some NoOp requests to the slice servers to get performance and latency
      // Assume same stats for initiator <-> accesser connection

      /*
       * // Calculate throughput of protocol (random access) double throughputRandom =
       * estimateFullStackThroughput(blowUp, blockSize, true, vaultWidth); System.out.println("I/O
       * request stack performance (random access) Mbps: " + 8.0 * throughputRandom / MB); //
       * Calculate throughput of protocol (sequential access) double throughputSequential =
       * estimateFullStackThroughput(blowUp, blockSize, false, vaultWidth); System.out.println("I/O
       * request stack performance (sequential access) Mbps: " + 8.0 * throughputSequential / MB);
       * 
       * System.out.println(""); // Determine Bottle Necks boolean sequentialReadCPULimited =
       * (throughputSequential > decodeStackPerformance); boolean sequentialWriteCPULimited =
       * (throughputSequential > encodeStackPerformance); boolean randomReadCPULimited =
       * (throughputRandom > decodeStackPerformance); boolean randomWriteCPULimited =
       * (throughputRandom > encodeStackPerformance); // Print out information on scenarios String
       * sequentialReadBottleNeck = sequentialReadCPULimited ? "Accessor (CPU)" : "IO Stack
       * (Latency, Bandwidth, Server Hard Drive)"; System.out.println("Sequential read bottle neck: " +
       * sequentialReadBottleNeck); double utilization = sequentialReadCPULimited ? 1.0 :
       * throughputSequential / decodeStackPerformance; System.out.println("Sequential read CPU
       * utilization: " + utilization * 100 + "%"); double sequentialReadThroughput =
       * (sequentialReadCPULimited ? decodeStackPerformance : throughputSequential);
       * System.out.println("Sequential read throughput: " + (8.0 * sequentialReadThroughput / MB) + "
       * Mbps"); System.out.println("Sequential read network utilization: " + (100.0 *
       * sequentialReadThroughput / THROUGHPUT_ACCESSER_TO_SLICESTOR) + "%");
       * 
       * System.out.println("");
       * 
       * String sequentialWriteBottleNeck = sequentialWriteCPULimited ? "Accessor (CPU)" : "IO Stack
       * (Latency, Bandwidth, Server Hard Drive)"; System.out.println("Sequential write bottle neck: " +
       * sequentialWriteBottleNeck); utilization = sequentialWriteCPULimited ? 1.0 :
       * throughputSequential / encodeStackPerformance; System.out.println("Sequential write CPU
       * utilization: " + utilization * 100 + "%"); double sequentialWriteThroughput =
       * (sequentialWriteCPULimited ? encodeStackPerformance : throughputSequential);
       * System.out.println("Sequential write throughput: " + (8.0 * sequentialWriteThroughput / MB) + "
       * Mbps"); System.out.println("Sequential write network utilization: " + (100.0 *
       * sequentialWriteThroughput / THROUGHPUT_ACCESSER_TO_SLICESTOR) + "%");
       * 
       * 
       * System.out.println("");
       * 
       * String randomReadBottleNeck = randomReadCPULimited ? "Accessor (CPU)" : "IO Stack (Latency,
       * Bandwidth, Server Hard Drive)"; System.out.println("Random read bottle neck: " +
       * randomReadBottleNeck); utilization = randomReadCPULimited ? 1.0 : throughputRandom /
       * decodeStackPerformance; System.out.println("Random read CPU utilization: " + utilization *
       * 100 + "%"); double randomReadThroughput = (randomReadCPULimited ? decodeStackPerformance :
       * throughputRandom); System.out.println("Random read throughput: " + (8.0 *
       * randomReadThroughput / MB) + " Mbps"); System.out.println("Random read network utilization: " +
       * (100.0 * randomReadThroughput / THROUGHPUT_ACCESSER_TO_SLICESTOR) + "%");
       * 
       * 
       * System.out.println("");
       * 
       * String randomWriteBottleNeck = randomWriteCPULimited ? "Accessor (CPU)" : "IO Stack
       * (Latency, Bandwidth, Server Hard Drive)"; System.out.println("Random write bottle neck: " +
       * randomWriteBottleNeck); utilization = sequentialReadCPULimited ? 1.0 : throughputRandom /
       * encodeStackPerformance; System.out.println("Random write CPU utilization: " + utilization *
       * 100 + "%"); double randomWriteThroughput = (randomWriteCPULimited ? encodeStackPerformance :
       * throughputRandom); System.out.println("Random write throughput: " + (8.0 *
       * randomWriteThroughput / MB) + " Mbps"); System.out.println("Random write network
       * utilization: " + (100.0 * randomWriteThroughput / THROUGHPUT_ACCESSER_TO_SLICESTOR) + "%"); //
       * TODO: Also print file paradigm theoretical performance (use larger block size and
       * sequential hard drive write)
       * 
       */
   }

}
