//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: wleggette
//
// Date: Jan 24, 2008
//---------------------

package org.cleversafe.util;

import java.util.Arrays;
import java.util.HashSet;
import java.util.IllegalFormatException;
import java.util.Set;

import org.apache.commons.codec.binary.Hex;

import com.martiansoftware.jsap.ParseException;
import com.martiansoftware.jsap.StringParser;


public class IQNStringParser extends StringParser
{
   private static final Character[] invalidCharacters = {
      // RFC 3454 Table C.1.1
      '\u0020',
      // RFC 3454 Table C.1.2 (excluding range 2000 - 200B)
      '\u00A0',
      '\u1680',
      '\u202F',
      '\u205F',
      '\u3000',
      // RFC 3454 Table C.2.1 (excluding range 0000 - 001F)
      '\u007F',
      // RFC 3454 Table C.2.2 (excluding range 0080 - 009F and 206A - 206F and FFF9 - FFFC)
      '\u06DD',
      '\u070F',
      '\u180E',
      '\u200C',
      '\u200D',
      '\u2028',
      '\u2029',
      '\u2060',
      '\u2061',
      '\u2062',
      '\u2063',
      '\uFEFF',
      // RFC 3454 Table C.8
      '\u0340',
      '\u0341',
      '\u200E',
      '\u200F',
      '\u202A',
      '\u202B',
      '\u202C',
      '\u202D',
      '\u202E',
      '\u206A',
      '\u206B',
      '\u206C',
      '\u206D',
      '\u206E',
      '\u206F',
      // RFC 3722 Section 6.1
      '\u3002',
      // RFC 3722 Section 6.2 (excluding range 0000-002C and others)
      '\u002F'
   };
   
   private static final Set<Character> invalidSet = new HashSet<Character>();
   
   static
   {
      invalidSet.addAll(Arrays.asList(invalidCharacters));
   }
   
   private static void checkCharacterSet(String str) throws ParseException
   {
      for (char chr : str.toCharArray())
      {
         if (  invalidSet.contains(chr) ||
               chr < '\u002C' ||    // RFC 3722 Section 6.2 and RFC 3454 Table C.2.1
               (chr >= '\u003B' && chr <= '\u0040') || // RFC 3722 Section 6.2
               (chr >= '\u005B' && chr <= '\u0060') ||
               (chr >= '\u007B' && chr <= '\u007F') ||
               (chr >= '\u2000' && chr <= '\u200B') || // RFC 3454 Table C.1.2
               (chr >= '\u0080' && chr <= '\u009F') || // RFC 3454 Table C.2.2
               (chr >= '\u206A' && chr <= '\u206F') ||
               (chr >= '\uFFF9' && chr <= '\uFFFC') )
         {
            final String hex = Integer.toHexString((int)chr);
            throw new ParseException(
                  "invalid IQN: contains illegal character '" + chr + "' (" + hex + ")");
            
         }
      }
   }
   

   @Override
   public Object parse(String arg0) throws ParseException
   {
      final String properFormat = "\nexpected string like 'iqn.2001-04.com.example:target-name'";
      final String iqn = arg0 != null ? arg0.toLowerCase() : null;
      
      if (iqn == null || "".equals(iqn))
         throw new ParseException("invalid IQN: <none>" + properFormat);
      
      if (iqn.length() < 13) // IQN must at least contain "iqn.YYYY-MM."
         throw new ParseException("invalid IQN: " + iqn + properFormat);
      
      checkCharacterSet(iqn);
      
      int idx = iqn.indexOf(':');
      
      final String authority = iqn.substring(0, idx < iqn.length() - 1 ? iqn.length() : idx);
      
      idx = authority.indexOf('.');
      if (idx == -1)
         throw new ParseException("invalid IQN: " + iqn + properFormat);
      if (!"iqn".equals(authority.substring(0,idx)))
         throw new ParseException("invalid IQN: " + iqn + properFormat);
      int idx2 = authority.indexOf('.', idx+1);
      if (idx2 == -1)
         throw new ParseException("invalid IQN: " + iqn + properFormat);
      final String date = authority.substring(idx+1, idx2);
      idx += idx + date.length() - 1;
      try
      {
         if (date.length() != 7)
            throw new NumberFormatException();
         Integer.valueOf(date.substring(0,4));
         int month = Integer.valueOf(date.substring(5,7));
         if (month < 1 || month > 12)
            throw new NumberFormatException();
      }
      catch (NumberFormatException e)
      {
         throw new ParseException("invalid IQN: illegal date component: " + date + properFormat);
      }
      
      // No need to check for domain string format
      // final String domain = authority.substring(idx);
      
      idx = authority.length();
      final String local = iqn.substring(idx);
      
      if ( local.length() > 0 && (local.length() < 2 || !local.substring(0,1).equals(":")))
         throw new ParseException(
               "invalid IQN: must specify local name after colon (or omit colon)" + properFormat);
      
      return iqn;
   }

}


