//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: jquigley
//
// Date: Oct 22, 2007
//---------------------

package org.cleversafe.layer.target.tasks;

import java.nio.ByteBuffer;

import org.apache.log4j.Logger;
import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.layer.block.exceptions.BlockIOException;
import org.cleversafe.layer.block.exceptions.BlockTransactionException;
import org.cleversafe.util.RandomNumberUtils;
import org.cleversafe.util.rangelock.RangeReadWriteLock;
import org.jscsi.scsi.protocol.Command;
import org.jscsi.scsi.protocol.cdb.AbstractTransferCDB;
import org.jscsi.scsi.protocol.cdb.Write10;
import org.jscsi.scsi.protocol.cdb.Write12;
import org.jscsi.scsi.protocol.cdb.Write16;
import org.jscsi.scsi.protocol.cdb.Write6;
import org.jscsi.scsi.protocol.inquiry.InquiryDataRegistry;
import org.jscsi.scsi.protocol.mode.ModePageRegistry;
import org.jscsi.scsi.protocol.sense.exceptions.InvalidCommandOperationCodeException;
import org.jscsi.scsi.protocol.sense.exceptions.LogicalBlockAddressOutOfRangeException;
import org.jscsi.scsi.protocol.sense.exceptions.SenseException;
import org.jscsi.scsi.protocol.sense.exceptions.SynchronousDataTransferErrorException;
import org.jscsi.scsi.tasks.Status;
import org.jscsi.scsi.transport.TargetTransportPort;

/**
 * @author John Quigley <jquigley@cleversafe.com>
 * @version $Id$
 */
public class WriteGridTask extends GridTask
{
   private static Logger _logger = Logger.getLogger(WriteGridTask.class);

   private static final int MIN_RETRY_SLEEP = 1000;
   private static final int MAX_RETRY_SLEEP = 5000;

   /**
    * SBC CDB Write command types
    */
   private enum WriteType
   {
      WRITE6, WRITE10, WRITE12, WRITE16, WRITE32;
   };

   /**
    * Creates a new WriteGridTask.
    */
   public WriteGridTask()
   {
      super("WriteGridTask");
   }

   /**
    * Creates a new WriteGridTask.
    * 
    * @param targetPort
    * @param command
    * @param modePageRegistry
    * @param inquiryDataRegistry
    * @param device
    */
   public WriteGridTask(
         final TargetTransportPort targetPort,
         final Command command,
         final ModePageRegistry modePageRegistry,
         final InquiryDataRegistry inquiryDataRegistry,
         final BlockDeviceController device,
         final RangeReadWriteLock rangeReadWriteLock)
   {
      super("WriteGridTask", targetPort, command, modePageRegistry, inquiryDataRegistry, device,
            rangeReadWriteLock);
   }

   /**
    * Determines the CDB Write type based upon an operation code.
    * 
    * @param operationCode The operation code we're inspecting
    * 
    * @return The ReadType of this operation
    * @throws InvalidCommandOperationCodeException 
    */
   private WriteType getWriteType(final int operationCode)
         throws InvalidCommandOperationCodeException
   {
      if (operationCode == Write6.OPERATION_CODE)
      {
         return WriteType.WRITE6;
      }
      else if (operationCode == Write10.OPERATION_CODE)
      {
         return WriteType.WRITE10;
      }
      else if (operationCode == Write12.OPERATION_CODE)
      {
         return WriteType.WRITE12;
      }
      else if (operationCode == Write16.OPERATION_CODE)
      {
         return WriteType.WRITE16;
      }
      else
      {
         // WRITE32 and any other write types are not currently supported
         throw new InvalidCommandOperationCodeException();
      }
   }

   /**
    * Executes this particular task from within the TaskManager.
    * 
    * @throws InterruptedException
    * @throws SenseException
    */
   @Override
   protected void execute() throws InterruptedException, SenseException
   {
      if (_logger.isDebugEnabled())
      {
         _logger.debug("Executing grid write task: " + toString());
      }
      
      final AbstractTransferCDB cdb =
            (AbstractTransferCDB) getCommand().getCommandDescriptorBlock();
      final long transferLength = cdb.getTransferLength();

      final long deviceSize = getBlockDeviceController().getDeviceSize();
      final int blockSize = getBlockDeviceController().getBlockSize();

      // TODO: transferLength in the third logical operation differs from usage in ReadGridTask#execute()
      if (transferLength < 0 || transferLength > Integer.MAX_VALUE
            || (transferLength + cdb.getLogicalBlockAddress()) > deviceSize / blockSize)
      {
         _logger.error("invalid transfer of lba size, transferLength: " + transferLength
               + ", lba: " + cdb.getLogicalBlockAddress());
         _logger.error("block device block size: " + getBlockDeviceController().getBlockSize());

         final WriteType writeType =
               getWriteType(getCommand().getCommandDescriptorBlock().getOperationCode());

         if (writeType.equals(WriteType.WRITE6))
         {
            throw new LogicalBlockAddressOutOfRangeException(true, true, (byte) 4, 1);
         }
         else if (writeType.equals(WriteType.WRITE10) || writeType.equals(WriteType.WRITE12)
               || writeType.equals(WriteType.WRITE16))
         {
            // field pointer is 2 for Write10, Write12, Write16
            throw new LogicalBlockAddressOutOfRangeException(true, true, 2);
         }
         else if (writeType.equals(WriteType.WRITE32))
         {
            throw new LogicalBlockAddressOutOfRangeException(true, true, 12);
         }
      }

      final ByteBuffer outData = ByteBuffer.allocate((int) transferLength * blockSize);

      boolean readSuccess = false;

      readSuccess = readData(outData);

      if (!readSuccess)
      {
         _logger.error("Unable to properly read data from target transport: "
               + getTargetTransportPort());
         throw new SynchronousDataTransferErrorException();
      }

      _logger.debug("Successfully read bytes from target transport port: " + outData.limit());

      final long lba = cdb.getLogicalBlockAddress();

      final RangeReadWriteLock.Lock writeLock =
            getRangeReadWriteLock().getWriteRangeLock(lba, transferLength);
      writeLock.acquire();
      BlockIOException lastBlockIOException = null;
      BlockTransactionException lastBlockTransactionException = null;
      int numIOErrors = 0;
      int numTransactionErrors = 0;
      while (true)
      {
         if (isAborted())
         {
            writeLock.release();
            if (_logger.isDebugEnabled())
            {
               _logger.debug("Grid write task was aborted: " + toString());
               if (lastBlockIOException != null)
               {
                  _logger.debug(numIOErrors
                        + " write IO errors occurred before abort, last IO exception: ",
                        lastBlockIOException);
                  _logger.info(String.format("scsi I/O occured during write operation at lba %d", lba));
               }
               if (lastBlockTransactionException != null)
               {
                  _logger.debug(
                        numTransactionErrors
                              + " write IO transaction errors occurred before abort, last transaction exception: ",
                        lastBlockTransactionException);
                  _logger.info(String.format("scsi I/O occured during write operation at lba %d", lba));
               }
            }
            return;
         }

         try
         {
            getBlockDeviceController().writeBlocks(lba, (int) transferLength, outData.array());
            break;
         }
         catch (final BlockIOException e)
         {
            _logger.trace("A block level write IO exception occurred writing blocks (" + lba + ", "
                  + transferLength + "), retrying", e);
            numIOErrors++;
            lastBlockIOException = e;
         }
         catch (final BlockTransactionException e)
         {
            _logger.trace("A block level write IO transaction exception occurred writing blocks (lba:" + lba
                  + ", numBlocks: " + transferLength + "), retrying", e);
            numTransactionErrors++;
            lastBlockTransactionException = e;
         }

         Thread.sleep(RandomNumberUtils.getRandomInt(MIN_RETRY_SLEEP, MAX_RETRY_SLEEP));
         // _logger.debug("Write attempt failed (lba:" + lba
         // + ", numBlocks: " + transferLength + "), retrying");
      }

      writeLock.release();
      writeResponse(Status.GOOD, null);
      if (_logger.isDebugEnabled())
      {
         _logger.debug("Grid write task completed successfully: " + toString());
      }
   }
}
