//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
//@author: jquigley
//
//Date: Oct 22, 2007
//---------------------

package org.cleversafe.layer.target.tasks;

import java.util.HashMap;
import java.util.Map;

import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.util.rangelock.RangeReadWriteLock;
import org.jscsi.scsi.protocol.Command;
import org.jscsi.scsi.protocol.cdb.CDB;
import org.jscsi.scsi.protocol.cdb.Inquiry;
import org.jscsi.scsi.protocol.cdb.ModeSense10;
import org.jscsi.scsi.protocol.cdb.ModeSense6;
import org.jscsi.scsi.protocol.cdb.Read10;
import org.jscsi.scsi.protocol.cdb.Read12;
import org.jscsi.scsi.protocol.cdb.Read16;
import org.jscsi.scsi.protocol.cdb.Read6;
import org.jscsi.scsi.protocol.cdb.ReadCapacity10;
import org.jscsi.scsi.protocol.cdb.ReadCapacity16;
import org.jscsi.scsi.protocol.cdb.TestUnitReady;
import org.jscsi.scsi.protocol.cdb.Verify10;
import org.jscsi.scsi.protocol.cdb.Write10;
import org.jscsi.scsi.protocol.cdb.Write12;
import org.jscsi.scsi.protocol.cdb.Write16;
import org.jscsi.scsi.protocol.cdb.Write6;
import org.jscsi.scsi.protocol.inquiry.InquiryDataRegistry;
import org.jscsi.scsi.protocol.mode.ModePageRegistry;
import org.jscsi.scsi.protocol.sense.exceptions.IllegalRequestException;
import org.jscsi.scsi.protocol.sense.exceptions.InvalidCommandOperationCodeException;
import org.jscsi.scsi.tasks.Task;
import org.jscsi.scsi.tasks.TaskFactory;
import org.jscsi.scsi.tasks.lu.InquiryTask;
import org.jscsi.scsi.tasks.lu.LUTaskFactory;
import org.jscsi.scsi.tasks.lu.ModeSenseTask;
import org.jscsi.scsi.tasks.lu.TestUnitReadyTask;
import org.jscsi.scsi.transport.TargetTransportPort;

/**
 * @author John Quigley <jquigley@cleversafe.com>
 * @version $Id$
 */
public class GridTaskFactory implements TaskFactory
{
   // map of tasks that the GridLogicalUnit is configured to respond to
   private static Map<Class<? extends CDB>, Class<? extends GridTask>> gridTasks =
         new HashMap<Class<? extends CDB>, Class<? extends GridTask>>();

   // registries of SCSI mode and inquiry data
   private final ModePageRegistry modePageRegistry;
   private final InquiryDataRegistry inquiryDataRegistry;

   // supervisor for grid-based I/O
   private final BlockDeviceController blockDeviceController;

   // task factory for production of generic LogicalUnitTasks
   private final LUTaskFactory luTaskFactory;

   // Read write lock to be passed to grid tasks
   private final RangeReadWriteLock rangeReadWriteLock;

   // the following maps in all CDBs that this LogicalUnit will directly support (besides the
   // generic
   // tasks suppored by the DefaultLogicalUnit
   static
   {
      GridTaskFactory.gridTasks.put(Read6.class, ReadGridTask.class);
      GridTaskFactory.gridTasks.put(Read10.class, ReadGridTask.class);
      GridTaskFactory.gridTasks.put(Read12.class, ReadGridTask.class);
      GridTaskFactory.gridTasks.put(Read16.class, ReadGridTask.class);
      GridTaskFactory.gridTasks.put(Verify10.class, VerifyGridTask.class);
      GridTaskFactory.gridTasks.put(Write6.class, WriteGridTask.class);
      GridTaskFactory.gridTasks.put(Write10.class, WriteGridTask.class);
      GridTaskFactory.gridTasks.put(Write12.class, WriteGridTask.class);
      GridTaskFactory.gridTasks.put(Write16.class, WriteGridTask.class);
      GridTaskFactory.gridTasks.put(ReadCapacity10.class, ReadCapacityGridTask.class);
      GridTaskFactory.gridTasks.put(ReadCapacity16.class, ReadCapacityGridTask.class);
   }

   /**
    * Creates a new GridTaskFactory.
    * 
    * @param controller The controller for grid I/O
    * @param modePageRegistry The registry of SCSI mode pages
    * @param inquiryDataRegistry The registry of SCSI inquiry data
    */
   public GridTaskFactory(
         final BlockDeviceController controller,
         final ModePageRegistry modePageRegistry,
         final InquiryDataRegistry inquiryDataRegistry,
         final RangeReadWriteLock rangeReadWriteLock)
   {
      this.blockDeviceController = controller;
      this.modePageRegistry = modePageRegistry;
      this.inquiryDataRegistry = inquiryDataRegistry;
      this.rangeReadWriteLock = rangeReadWriteLock;

      // the generic logical unit should respond with tasks that use the very same registries we'll use to generic
      // tasks locally within this factory; all tasks supplied by this factory should be consistent in their
      // use of particular mode and inquiry registries
      this.luTaskFactory = new LUTaskFactory(this.modePageRegistry, this.inquiryDataRegistry);
   }

   /**
    * Produces a new Task wrapping the particular command here specified parametrically,
    * communicating with the ISCSI layer via the TargetTransportPort port.
    * 
    * @param port The Transport used for ISCSI-SCSI I/O communication
    * @param command The command to be wrapped as a Task
    */
   public Task getInstance(final TargetTransportPort port, final Command command)
         throws IllegalRequestException
   {
      // XXX: this entire switch block should be removed, as the LUTaskFactory will now properly respond to these
      switch (command.getCommandDescriptorBlock().getOperationCode())
      {
         case ModeSense6.OPERATION_CODE :
         case ModeSense10.OPERATION_CODE :
            return new ModeSenseTask(port, command, this.modePageRegistry, this.inquiryDataRegistry);
         case TestUnitReady.OPERATION_CODE :
            return new TestUnitReadyTask(port, command, this.modePageRegistry,
                  this.inquiryDataRegistry);
         case Inquiry.OPERATION_CODE :
            return new InquiryTask(port, command, this.modePageRegistry, this.inquiryDataRegistry);
      }

      // check the generic task factory to determine whether it supports the command
      if (this.luTaskFactory.respondsTo(command.getCommandDescriptorBlock().getClass()))
      {
         return this.luTaskFactory.getInstance(port, command);
      }

      // check our local map of grid-specific tasks to check whether we support the command
      final Class<? extends GridTask> taskClass =
            gridTasks.get(command.getCommandDescriptorBlock().getClass());

      // we may reach this point without a valid task responder, in which case we know the command is not supported
      if (taskClass != null)
      {
         Task task = null;
         try
         {
            task =
                  taskClass.newInstance().load(port, command, this.modePageRegistry,
                        this.inquiryDataRegistry, this.blockDeviceController,
                        this.rangeReadWriteLock);
         }
         catch (final InstantiationException e)
         {
            throw new RuntimeException("Unable to instantiate class with default constructor"
                  + taskClass.getName(), e);
         }
         catch (final IllegalAccessException e)
         {
            throw new RuntimeException("Unable to instantiate class with default constructor"
                  + taskClass.getName(), e);
         }

         return task;
      }
      else
      {
         // we don't support this command
         throw new InvalidCommandOperationCodeException();
      }
   }

   /**
    * Acquires the BlockDeviceController associated with this task, to be used for grid I/O.
    * 
    * @return The associated BlockDeviceController
    */
   public BlockDeviceController getBlockDeviceController()
   {
      return this.blockDeviceController;
   }

   /**
    * Indicates whether this logical unit supports a particular CDB
    * 
    * @param cls The CDB class we're inspecting
    * 
    * @return True if supported, false otherwise
    */
   public boolean respondsTo(final Class<? extends CDB> cls)
   {
      return this.luTaskFactory.respondsTo(cls) || gridTasks.containsKey(cls);
   }

   /**
    * Gets the string representation of this class.
    * 
    * @return The string representation
    */
   @Override
   public String toString()
   {
      return "<GridTask>";
   }
}
