//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
//Author: jquigley
//
//Date: May 24, 2007
//---------------------

package org.cleversafe.layer.iscsi;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.security.DigestException;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Logger;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.PropertiesProvider;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.layer.iscsi.authentication.ISCSIAuthenticationSession;
import org.cleversafe.layer.iscsi.authentication.SourceOfSecrets;
import org.cleversafe.layer.iscsi.authentication.SourceOfSecretsConfigurationException;
import org.cleversafe.layer.iscsi.authentication.SourceOfSecretsFactory;
import org.cleversafe.layer.iscsi.authentication.ISCSIAuthenticationSession.SecurityNegotiationStatus;
import org.cleversafe.layer.iscsi.authentication.chap.ChapAuthenticationSession;
import org.cleversafe.layer.iscsi.exceptions.ConnectionClosedException;
import org.cleversafe.layer.iscsi.exceptions.ISCSILayerException;
import org.cleversafe.util.Tuple3;
import org.jscsi.connection.SerialArithmeticNumber;
import org.jscsi.parser.AbstractMessageParser;
import org.jscsi.parser.InitiatorMessageParser;
import org.jscsi.parser.OperationCode;
import org.jscsi.parser.ProtocolDataUnit;
import org.jscsi.parser.ProtocolDataUnitFactory;
import org.jscsi.parser.TargetMessageParser;
import org.jscsi.parser.data.DataInParser;
import org.jscsi.parser.data.DataOutParser;
import org.jscsi.parser.datasegment.DataSegmentFactory;
import org.jscsi.parser.datasegment.IDataSegment;
import org.jscsi.parser.datasegment.IDataSegmentIterator;
import org.jscsi.parser.datasegment.OperationalTextKey;
import org.jscsi.parser.datasegment.SettingsMap;
import org.jscsi.parser.datasegment.TextParameterDataSegment;
import org.jscsi.parser.exception.InternetSCSIException;
import org.jscsi.parser.login.ISID;
import org.jscsi.parser.login.LoginRequestParser;
import org.jscsi.parser.login.LoginResponseParser;
import org.jscsi.parser.login.LoginStage;
import org.jscsi.parser.login.LoginStatus;
import org.jscsi.parser.logout.LogoutResponse;
import org.jscsi.parser.logout.LogoutResponseParser;
import org.jscsi.parser.nop.NOPInParser;
import org.jscsi.parser.r2t.Ready2TransferParser;
import org.jscsi.parser.scsi.SCSICommandParser;
import org.jscsi.parser.scsi.SCSIResponseParser;
import org.jscsi.parser.scsi.SCSIStatus;
import org.jscsi.parser.text.TextResponseParser;
import org.jscsi.parser.tmf.TaskManagementFunctionRequestParser;
import org.jscsi.parser.tmf.TaskManagementFunctionResponseParser;
import org.jscsi.scsi.protocol.Command;
import org.jscsi.scsi.protocol.cdb.CDB;
import org.jscsi.scsi.protocol.cdb.CDBFactory;
import org.jscsi.scsi.protocol.cdb.Inquiry;
import org.jscsi.scsi.protocol.cdb.ReportLuns;
import org.jscsi.scsi.protocol.cdb.ReportSupportedTaskManagementFunctions;
import org.jscsi.scsi.protocol.sense.exceptions.LogicalUnitNotSupportedException;
import org.jscsi.scsi.target.Target;
import org.jscsi.scsi.tasks.Status;
import org.jscsi.scsi.tasks.TaskAttribute;
import org.jscsi.scsi.tasks.management.TaskManagementFunction;
import org.jscsi.scsi.transport.Nexus;
import org.jscsi.scsi.transport.TargetTransportPort;

public class PrototypeiSCSITarget implements Callable<Void>, TargetTransportPort
{
   private static Logger _logger = Logger.getLogger(PrototypeiSCSITarget.class);

   private ISCSIPortal portal;
   private Target scsiTarget;
   private CDBFactory cdbFactory;

   protected AtomicInteger taskQueueSize;
   protected AtomicInteger taskQueueSuccesses;
   protected AtomicInteger cmdRef;

   protected int targetPortalGroupTag = 1;
   protected short targetSessionIdentifyingHandle = 0x0100;
   protected String portalHost;
   protected int portalPort;
   protected String targetName = "-unknown-";

   protected Socket socket;
   protected boolean discoverySession;
   protected boolean loginSuccessful;

   protected String initiatorName;
   protected SerialArithmeticNumber expectedCmdSN;
   protected SerialArithmeticNumber expectedStatSN;
   protected ISID sessionID;
   protected int maxDataSegmentLength = 8192;

   protected ISCSIAuthenticationSession authenticationSession;

   //  Setting this to true means the target will not allow Login without authentication
   protected boolean targetRequiresAuthentication;

   //  This property provides the lookup of the secret for an individual user name
   protected SourceOfSecrets sourceOfSecrets;

   //  This secret is used to authenticate the target against the initiator.
   //  One could imagine a use case where this should be typed SourceOfSecrets, but it depends on how complicated you want to get
   protected byte[] targetAuthenticationSecret;

   //  This is the name used to authenticate the target against the initiator
   protected String targetAuthenticationName;

   protected ProtocolDataUnitFactory pduFactory = new ProtocolDataUnitFactory();

   // Pending tasks (Initiator task tag => Task)
   ConcurrentMap<Integer, Task> pendingTasks = new ConcurrentHashMap<Integer, Task>();

   // Pending writes-in-progress (Initiator task tag => WriteInProgress)
   ConcurrentMap<Integer, WriteInProgress> writesInProgress =
         new ConcurrentHashMap<Integer, WriteInProgress>();

   // TODO: describe the use of this
   Lock recvLock = new ReentrantLock();

   // Task tag must be set on PDU
   Lock sendLock = new ReentrantLock();

   /////////////////////////////////////////////////////////////////////////////
   // Constructor(s)

   public PrototypeiSCSITarget(
         ISCSIPortal portal,
         Socket socket,
         String portalHost,
         int portalPort,
         int taskQueueSize)
   {
      this.portal = portal;
      this.socket = socket;
      this.portalHost = portalHost;
      this.portalPort = portalPort;

      this.expectedCmdSN = new SerialArithmeticNumber(0);
      this.expectedStatSN = new SerialArithmeticNumber(0);
      this.loginSuccessful = false;

      this.cdbFactory = new CDBFactory();

      this.taskQueueSize = new AtomicInteger(taskQueueSize);

      targetRequiresAuthentication = getBooleanConfigProperty("org.cleversafe.layer.iscsi.target.requires-authentication", false);      
      targetAuthenticationSecret = getStringConfigProperty("org.cleversafe.layer.iscsi.target.chap.mutual.secret", "").getBytes();
      targetAuthenticationName = getStringConfigProperty("org.cleversafe.layer.iscsi.target.chap.mutual.target-name", null);

      String secretsImpl = getStringConfigProperty("org.cleversafe.layer.iscsi.target.chap.secrets.implementation", null);
      List<String> secretsImplParameters = getStringArrayConfigProperty("org.cleversafe.layer.iscsi.target.chap.secrets.filename", null);
      
      try
      {
         sourceOfSecrets = SourceOfSecretsFactory.createSourceOfSecrets(secretsImpl, secretsImplParameters);
      }
      catch (SourceOfSecretsConfigurationException e)
      {
         _logger.warn("Encountered an error loading SourceOfSecrets: " + e);
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // Target methods

   public Void call() throws Exception
   {
      boolean done = false;
      ProtocolDataUnit pdu;

      try
      {
         while (!done && (pdu = this.recv()) != null)
         {
            //  Disallow any command prior to login succeeding
            if (!this.loginSuccessful)
            {
               //  Login has to be allowed
               if (pdu.getBasicHeaderSegment().getOpCode() != OperationCode.LOGIN_REQUEST)
               {
                  _logger.debug("Initiator attempted issuing command without having logged in");
                  return null;
               }
            }

            Task task = this.createTask(pdu);
            boolean immediate = pdu.getBasicHeaderSegment().isImmediateFlag();

            if (immediate)
            {
               _logger.debug("Processing immediate iSCSI task: " + task);

               done = task.handleRequest();
            }
            else
            {
               // DataOut tasks do not have their own ITT
               if (!(task instanceof DataOutTask))
               {
                  this.pendingTasks.put(pdu.getBasicHeaderSegment().getInitiatorTaskTag(), task);
               }

               _logger.debug(String.format("%s handling iSCSI task %s (pending: %d)", this, task,
                     this.pendingTasks.size()));

               // These tasks can be executed in parallel if desired, but
               // there is currently little benefit since all SCSI tasks
               // get parallelized at the SCSI layer and iSCSI tasks are
               // generally short-lived
               done = task.handleRequest();
            }
         }
         _logger.debug(String.format("%s exiting gracefully", this));
      }
      catch (ConnectionClosedException ex)
      {
         _logger.warn(String.format("%s disconnected unexpectedly: %s", this, this.initiatorName));
      }
      catch (Exception ex)
      {
         _logger.warn(String.format("%s disconnected unexpectedly: %s", this, this.initiatorName), ex);
      }
      finally
      {
         this.purgeSCSIState();
      }

      _logger.debug(String.format("%s unloading", this));
      return null;
   }

   /**
    * Create a Task for the given request PDU
    * @param request
    * @return
    */
   private Task createTask(ProtocolDataUnit request)
   {
      _logger.debug(String.format("%s got pdu: %s", this,
            request.getBasicHeaderSegment().getOpCode()));
      switch (request.getBasicHeaderSegment().getOpCode())
      {
         case LOGIN_REQUEST :
            return new LoginTask(request);
         case TEXT_REQUEST :
            return new TextTask(request);
         case LOGOUT_REQUEST :
            return new LogoutTask(request);
         case SCSI_COMMAND :
            return new SCSITask(request);
         case NOP_OUT :
            return new NopTask(request);
         case SCSI_DATA_OUT :
            return new DataOutTask(request);
         case SCSI_TM_REQUEST :
            return new TMTask(request);
         default :
            throw new RuntimeException("Received an unknown iSCSI protocol request:\n" + request);
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // TargetTransportPort implementation

   /*
    * @see org.jscsi.scsi.transport.TargetTransportPort#writeResponse(org.jscsi.scsi.transport.Nexus, long, org.jscsi.scsi.tasks.Status, java.nio.ByteBuffer)
    */
   public void writeResponse(
         Nexus nexus,
         long commandReferenceNumber,
         Status status,
         ByteBuffer senseData)
   {
      ProtocolDataUnit oPdu, iPdu;
      int initiatorTaskTag = (int) commandReferenceNumber;

      // Associate with pending request
      Task task = this.pendingTasks.get(initiatorTaskTag);
      if (task == null)
      {
         _logger.error(String.format("%s no pending task for response: %x", this, initiatorTaskTag));

         if (_logger.isDebugEnabled())
         {
            _logger.debug("DEBUG: (info)  no pending task for response is indicative of serious problem!");
            _logger.debug("DEBUG: (info)  method parameters follow:");
            _logger.debug("DEBUG: (param) nexus: " + nexus.toString());
            _logger.debug("DEBUG: (param) commandReferenceNumber: " + commandReferenceNumber);
            _logger.debug("DEBUG: (param) status: " + status.toString());
            _logger.debug("DEBUG: (param) senseData: " + senseData.toString());
            _logger.debug("DEBUG: (info)  stack trace follows: ");
            for (StackTraceElement stackElem : (new Throwable()).getStackTrace())
            {
               _logger.debug("DEBUG: (stack) " + stackElem.toString());
            }
         }

         return;
      }

      iPdu = task.getRequest();

      // Build SCSI Response PDU
      oPdu = this.pduFactory.create(false, true, OperationCode.SCSI_RESPONSE, "None", "None");
      SCSIResponseParser oParser = (SCSIResponseParser) oPdu.getBasicHeaderSegment().getParser();

      // Response bookkeeping
      oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
            iPdu.getBasicHeaderSegment().getInitiatorTaskTag());
      oParser.setResponse(SCSIResponseParser.ServiceResponse.COMMAND_COMPLETED_AT_TARGET);

      // Set the over or under-transfer amount
      oParser.setResidualCount(task.getResidualCount());

      // FIXME: Eventually SCSIStatus and Status will be merged
      oParser.setStatus(SCSIStatus.valueOf(status.value()));

      // TODO: this is really a hack, I don't understand why this must be set
      // iSWAT complained that we were sending the wrong ExpDataSN in the SCSI response
      // as we were sending 0, while it expected 1 ...
      oParser.setExpectedDataSequenceNumber(1);

      // If CHECK_CONDITION, include autosense data
      if (status == Status.CHECK_CONDITION)
      {
         // Autosense
         // FIXME: Assumes that sense data is less than maxDataSegmentLength bytes
         // FIXME: For now, we create a separate buffer to append the sense data length
         // This could probably be made more efficient (perhaps SCSI layer puts this in?)
         ByteBuffer autosense = ByteBuffer.allocate(senseData.remaining() + 2);
         autosense.putShort((short) senseData.remaining());
         autosense.put(senseData);
         autosense.rewind();

         IDataSegment oSegment =
               DataSegmentFactory.create(autosense, DataSegmentFactory.DataSegmentFormat.BINARY,
                     this.maxDataSegmentLength);
         IDataSegmentIterator iterator = oSegment.iterator();
         oPdu.setDataSegment(iterator.next(this.maxDataSegmentLength));
         _logger.debug("preparing to return autosense data to initiator");
      }

      // Send response
      try
      {
         this.send(oPdu);
      }
      catch (Exception ex)
      {
         _logger.error("Error sending response", ex);
      }
      finally
      {
         // Remove pending task after any response is sent
         this.pendingTasks.remove(initiatorTaskTag);

         _logger.debug(String.format("%s completed SCSI task %s with status %s", this, task, status));
      }
   }

   /*
    * @see org.jscsi.scsi.transport.TargetTransportPort#readData(org.jscsi.scsi.transport.Nexus,
    *      long, java.nio.ByteBuffer) NOTE: Called by target once to get data for a WRITE
    */
   public boolean readData(Nexus nexus, long commandReferenceNumber, ByteBuffer output)
   {
      _logger.debug("iSCSI TransportPort readData requested");
      int initiatorTaskTag = (int) commandReferenceNumber;

      ProtocolDataUnit iPdu;
      SCSICommandParser iParser;

      // Associate with pending request
      Task task = this.pendingTasks.get(initiatorTaskTag);
      if (task == null)
      {
         _logger.error("No pending task for response: " + initiatorTaskTag);
         return false;
      }
      iPdu = task.getRequest();
      iParser = (SCSICommandParser) iPdu.getBasicHeaderSegment().getParser();

      // Create a new WriteInProgress
      WriteInProgress write =
            new WriteInProgress(iParser.getExpectedDataTransferLength(), initiatorTaskTag);

      // Initiate data transfer
      this.writesInProgress.put(initiatorTaskTag, write);
      try
      {
         write.start();
         if (write.join())
         {
            // Data was received successfully
            output.put(write.getData());

            _logger.debug("Successfully received data out for task: " + initiatorTaskTag);

            return true;
         }
         else
         {
            _logger.debug("Read was cancelled for task: " + initiatorTaskTag);
            return false;
         }
      }
      catch (Exception e)
      {
         _logger.error("Error reading data for grid write task: " + initiatorTaskTag, e);
         return false;
      }
      finally
      {
         this.writesInProgress.remove(initiatorTaskTag);
      }
   }

   /*
    * @see org.jscsi.scsi.transport.TargetTransportPort#terminateDataTransfer(org.jscsi.scsi.transport.Nexus)
    */
   public void terminateDataTransfer(Nexus nexus, long commandReferenceNumber)
   {
      _logger.debug("iSCSI TransportPort terminateDataTransfer requested");
      int initiatorTaskTag = (int) commandReferenceNumber;

      // Associate with pending request
      WriteInProgress write = this.writesInProgress.get(initiatorTaskTag);
      if (write == null)
      {
         // FIXME: There does not appear to be a way to indicate error here
         _logger.trace("No pending data transfer to terminate for ITT: " + initiatorTaskTag);
         return;
      }

      write.cancel();
      _logger.debug("iSCSI TransportPort terminateDataTransfer completed");
   }

   /*
    * @see org.jscsi.scsi.transport.TargetTransportPort#writeData(org.jscsi.scsi.transport.Nexus,
    *      long, java.nio.ByteBuffer) NOTE: Called by target once to submit data from a READ
    */
   public boolean writeData(Nexus nexus, long commandReferenceNumber, ByteBuffer input)
   {
      ProtocolDataUnit oPdu, iPdu;
      SCSICommandParser iParser;
      int initiatorTaskTag = (int) commandReferenceNumber;

      // Associate with pending request
      Task task = this.pendingTasks.get(initiatorTaskTag);
      if (task == null)
      {
         _logger.error("No pending iSCSI task for response: " + initiatorTaskTag);
         return false;
      }
      iPdu = task.getRequest();
      iParser = (SCSICommandParser) iPdu.getBasicHeaderSegment().getParser();

      // Limit to expected transfer length
      int expectedTransferLength = iParser.getExpectedDataTransferLength();
      task.setResidualCount(input.remaining() - expectedTransferLength);
      ByteBuffer sendBuffer = input.asReadOnlyBuffer();
      sendBuffer.rewind();
      sendBuffer.limit(Math.min(sendBuffer.position() + expectedTransferLength, sendBuffer.limit()));

      IDataSegment oSegment =
            DataSegmentFactory.create(sendBuffer, DataSegmentFactory.DataSegmentFormat.BINARY,
                  this.maxDataSegmentLength);

      long bytesLeft = oSegment.getLength();
      IDataSegmentIterator iterator = oSegment.iterator();

      // Send PDUs
      int pduNum = 0;
      boolean stop = false;
      while (!stop)
      {
         boolean isFinal = (bytesLeft <= this.maxDataSegmentLength);
         oPdu = this.pduFactory.create(false, isFinal, OperationCode.SCSI_DATA_IN, "None", "None");
         DataInParser oParser = (DataInParser) oPdu.getBasicHeaderSegment().getParser();

         oParser.setTargetTransferTag(isFinal ? 0xffffffff : 0x1);

         int residualCount = sendBuffer.limit() - iParser.getExpectedDataTransferLength();
         oParser.setResidualCount(residualCount);

         // Multiple-PDU responses
         oParser.setBufferOffset(pduNum * this.maxDataSegmentLength);
         oParser.setDataSequenceNumber(pduNum);

         try
         {
            oPdu.setDataSegment(iterator.next(this.maxDataSegmentLength));
         }
         catch (NoSuchElementException e)
         {
            _logger.warn("data segment is unexpectedly empty");
         }

         try
         {
            oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
                  iPdu.getBasicHeaderSegment().getInitiatorTaskTag());

            // Response transmission
            this.send(oPdu);
         }
         catch (Exception ex)
         {
            _logger.error("Unable to send DataIn", ex);
            return false;
         }

         bytesLeft -= this.maxDataSegmentLength;
         pduNum++;

         stop = (iterator != null) ? !iterator.hasNext() : true;

      }
      return true;
   }

   /////////////////////////////////////////////////////////////////////////////
   // Task handlers

   abstract class Task
   {
      protected ProtocolDataUnit request;
      protected AtomicBoolean isCanceled = new AtomicBoolean(false);
      protected PrototypeiSCSITarget parent;

      // Number of bytes over or under the expected transfer length
      protected int residualCount = 0;

      public Task(ProtocolDataUnit request)
      {
         this.request = request;
         this.parent = PrototypeiSCSITarget.this;
      }

      public abstract boolean handleRequest() throws Exception;

      public ProtocolDataUnit getRequest()
      {
         return this.request;
      }

      public void setResidualCount(int residualCount)
      {
         this.residualCount = residualCount;
      }

      public int getResidualCount()
      {
         return this.residualCount;
      }

      public boolean isCanceled()
      {
         return this.isCanceled.get();
      }

      public void cancel()
      {
         _logger.debug(String.format("%s cancelling task: %s", this.parent, this));
         this.isCanceled.set(true);
      }

      protected Tuple3<AbstractMessageParser, IDataSegment, SettingsMap> helperPDU(
            final ProtocolDataUnit pdu) throws DigestException, InternetSCSIException, IOException
      {
         AbstractMessageParser parser = pdu.getBasicHeaderSegment().getParser();
         IDataSegment segment =
               (TextParameterDataSegment) DataSegmentFactory.create(pdu.getDataSegment(),
                     parser.getDataSegmentFormat(), this.parent.maxDataSegmentLength);
         SettingsMap settings = segment.getSettings();

         return new Tuple3<AbstractMessageParser, IDataSegment, SettingsMap>(parser, segment,
               settings);
      }

      public String toString()
      {
         return String.format("<%s: opCode: %s, initiatorTaskTag: %x>",
               this.getClass().getSimpleName(), this.request.getBasicHeaderSegment().getOpCode(),
               this.request.getBasicHeaderSegment().getInitiatorTaskTag());
      }
   }

   class LoginTask extends Task
   {
      public LoginTask(ProtocolDataUnit request)
      {
         super(request);
      }

      public boolean handleRequest() throws IOException, DigestException, InternetSCSIException,
            InterruptedException, ISCSILayerException
      {
         ProtocolDataUnit iPdu, oPdu;
         LoginRequestParser iParser;
         LoginResponseParser oParser;
         IDataSegment oSegment;
         SettingsMap iSettings, oSettings = null;
         Tuple3<AbstractMessageParser, IDataSegment, SettingsMap> tuple;

         // First PDU
         iPdu = this.request;
         tuple = this.helperPDU(iPdu);
         iParser = (LoginRequestParser) tuple.getFirst();
         iSettings = tuple.getThird();

         if (_logger.isDebugEnabled())
            _logger.debug("performing iSCSI login");

         if (_logger.isTraceEnabled())
         {
            _logger.trace("Leading Login PDU: " + iPdu);
            dumpSettingsMap("Leading Login", iSettings);
         }

         boolean is_authentication_required = this.parent.targetRequiresAuthentication;
         boolean is_security_negotiated = false;

         this.parent.initiatorName = iSettings.get(OperationalTextKey.INITIATOR_NAME);
         if (this.parent.sessionID == null)
         {
            this.parent.sessionID = iParser.getInitiatorSessionID();
         }

         // Find associated SCSI target for non-discovery sessions
         this.parent.discoverySession =
               "Discovery".equals(iSettings.get(OperationalTextKey.SESSION_TYPE));

         if (!this.parent.discoverySession)
         {
            String targetName = iSettings.get(OperationalTextKey.TARGET_NAME);
            if (targetName == null)
            {
               throw new InternetSCSIException("No target name provided");
            }

            // TODO: we simply drop the connection if the iSCSI target is 
            //       already active, investigate proper behavior
            if (this.parent.portal.isTargetActivated(targetName))
            {
               _logger.warn("another client has an active session with target: " + targetName);
            }

            Target scsiTarget = this.parent.portal.getSCSITarget(targetName);
            if (scsiTarget == null)
            {
               // FIXME: Should this be handled more gracefully?
               throw new InternetSCSIException("unknown target: " + targetName);
            }

            this.parent.portal.activateTarget(targetName);
            this.parent.scsiTarget = scsiTarget;
            this.parent.targetName = targetName;
         }

         if (_logger.isDebugEnabled())
         {
            _logger.debug("Stage: " + iParser.getCurrentStageNumber() + " -  "
                  + iParser.getNextStageNumber());
         }

         SecurityNegotiationStatus security_negotiation_status =
               SecurityNegotiationStatus.IN_PROGRESS;
         boolean is_leading_login = true;

         // Security negotiation phase
         while (iParser.getCurrentStageNumber() == LoginStage.SECURITY_NEGOTIATION)
         {
            // Text data fields
            oSettings = new SettingsMap();

            if (is_leading_login)
            {
               oSettings.add(OperationalTextKey.TARGET_PORTAL_GROUP_TAG,
                     Integer.toString(this.parent.targetPortalGroupTag));
               is_leading_login = false;
            }

            String initiator_auth_method = iSettings.get(OperationalTextKey.AUTH_METHOD);

            //  Determine the authentication method to use
            if (is_authentication_required && initiator_auth_method != null
                  && this.parent.authenticationSession == null)
            {
               String[] initiator_auth_methods = initiator_auth_method.split(",");

               //  Find the first auth method presented by the client that we support
               for (String auth_method : initiator_auth_methods)
               {
                  //  The only thing the target supports is CHAP.  This could be extended
                  //  and moved into a separate factory class in the future.
                  if (auth_method.equals(ChapAuthenticationSession.AUTH_METHOD_CHAP))
                  {
                     this.parent.authenticationSession =
                           new ChapAuthenticationSession(targetAuthenticationSecret,
                                 targetAuthenticationName);
                     break;
                  }
               }
            }

            if (is_authentication_required)
            {
               //  If authentication is required but no acceptable AuthMethod was supplied,
               //  we're done with security negotiation
               if (this.parent.authenticationSession == null)
               {
                  if (_logger.isDebugEnabled())
                     _logger.debug("Authentication is required but initiator did not present acceptable AuthMethod");

                  break;
               }
               else
               {
                  //  Take a step through security negotiation
                  security_negotiation_status =
                        this.parent.authenticationSession.handleAuthenticationCheck(iSettings,
                              oSettings, this.parent.sourceOfSecrets);

                  //  If we have reached the point of fatal authentication error, break out of
                  //  the negotiation loop.  The authentication failure will be sent back to the
                  //  initiator from the check right after this loop.
                  if (security_negotiation_status == SecurityNegotiationStatus.COMPLETE_FAILURE)
                  {
                     if (_logger.isDebugEnabled())
                        _logger.debug("Breaking out of login sequence because authentication has failed");

                     break;
                  }
               }
            }
            else
            {
               //  No authentication is required, so negotiate to use None as the AuthMethod
               oSettings.add(OperationalTextKey.AUTH_METHOD, "None");
            }

            //  See if we have completed security negotation successfully
            is_security_negotiated =
                  !is_authentication_required
                        || security_negotiation_status == SecurityNegotiationStatus.COMPLETE_SUCCESS;

            // Send
            oSegment =
                  DataSegmentFactory.create(oSettings.asByteBuffer(),
                        DataSegmentFactory.DataSegmentFormat.TEXT, this.parent.maxDataSegmentLength);

            if (_logger.isDebugEnabled())
               dumpSettingsMap("Login Response", oSettings);

            long bytesLeft = oSegment.getLength();
            IDataSegmentIterator iterator = oSegment.iterator();
            
            boolean tmp_in_loop = true;

            //  A data segment can require multiple PDUs, so send them back here
            while (tmp_in_loop)
            {
               boolean isFinal = (bytesLeft < this.parent.maxDataSegmentLength);

               //  Only include the T (transit) flag on the last chunk of this PDU
               //  if we can move on to the next stage of Login
               boolean is_transit = isFinal && is_security_negotiated;

               oPdu =
                     this.parent.pduFactory.create(false, is_transit, OperationCode.LOGIN_RESPONSE,
                           "None", "None");
               oParser = (LoginResponseParser) oPdu.getBasicHeaderSegment().getParser();

               oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
                     iPdu.getBasicHeaderSegment().getInitiatorTaskTag());
               oParser.setCurrentStageNumber(iParser.getCurrentStageNumber());

               //  If we've negotiated security, set NSG to operational negotation
               if (is_security_negotiated)
               {
                  oParser.setNextStageNumber(LoginStage.LOGIN_OPERATIONAL_NEGOTIATION);
               }
               else
               {
                  oParser.setNextStageNumber(LoginStage.SECURITY_NEGOTIATION);
               }

               oParser.setInitiatorSessionID(this.parent.sessionID);
               oParser.setTargetSessionIdentifyingHandle(this.parent.targetSessionIdentifyingHandle);

               //  Login status SUCCESS only means that it is going along as planned
               oParser.setStatus(LoginStatus.SUCCESS);
               oParser.setContinueFlag(!isFinal);
               
               if (iterator.hasNext())
               {
                  oPdu.setDataSegment(iterator.next(this.parent.maxDataSegmentLength));
               }

               if (_logger.isTraceEnabled())
                  _logger.trace("oPdu: " + oPdu);

               this.parent.send(oPdu);
               bytesLeft -= this.parent.maxDataSegmentLength;
               
               tmp_in_loop = iterator.hasNext();
            }

            // Get next PDU in Login sequence
            iPdu = this.parent.recv();
            tuple = this.helperPDU(iPdu);
            iParser = (LoginRequestParser) tuple.getFirst();
            iSettings = tuple.getThird();

            if (_logger.isDebugEnabled())
            {
               _logger.trace("Next PDU in Login sequence: " + iPdu);
               dumpSettingsMap("Next PDU in Login sequence", iSettings);
            }
         }

         //  Handle any case where we are requiring authentication but couldn't work it out 
         //  with the initiator after the above security negotiation
         if (is_authentication_required && !is_security_negotiated)
         {
            oPdu =
                  this.parent.pduFactory.create(false, false, OperationCode.LOGIN_RESPONSE, "None",
                        "None");
            oParser = (LoginResponseParser) oPdu.getBasicHeaderSegment().getParser();

            oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
                  iPdu.getBasicHeaderSegment().getInitiatorTaskTag());
            oParser.setCurrentStageNumber(iParser.getCurrentStageNumber());
            oParser.setNextStageNumber(iParser.getCurrentStageNumber());
            oParser.setInitiatorSessionID(this.parent.sessionID);
            oParser.setTargetSessionIdentifyingHandle(this.parent.targetSessionIdentifyingHandle);
            oParser.setStatus(LoginStatus.AUTHENTICATION_FAILURE);

            oParser.setContinueFlag(false);

            if (_logger.isDebugEnabled())
               _logger.debug("oPdu: " + oPdu);

            this.parent.send(oPdu);
         }

         else
         {
            // Operational negotiation phase
            if (iParser.getCurrentStageNumber() == LoginStage.LOGIN_OPERATIONAL_NEGOTIATION)
            {
               String setting;

               // Max datasegment length
               if ((setting = iSettings.get(OperationalTextKey.MAX_RECV_DATA_SEGMENT_LENGTH)) != null)
               {
                  this.parent.maxDataSegmentLength = Integer.parseInt(setting);
               }

               oSettings = iSettings;
               oSettings.add(OperationalTextKey.TARGET_PORTAL_GROUP_TAG,
                     Integer.toString(this.parent.targetPortalGroupTag));

               // Remove unwanted keys
               if ((setting = iSettings.get(OperationalTextKey.INITIATOR_NAME)) != null)
               {
                  oSettings.remove(OperationalTextKey.INITIATOR_NAME);
               }
               if ((setting = iSettings.get(OperationalTextKey.INITIATOR_ALIAS)) != null)
               {
                  oSettings.remove(OperationalTextKey.INITIATOR_ALIAS);
               }
               if ((setting = iSettings.get(OperationalTextKey.SESSION_TYPE)) != null)
               {
                  oSettings.remove(OperationalTextKey.SESSION_TYPE);
               }
               if ((setting = iSettings.get(OperationalTextKey.TARGET_NAME)) != null)
               {
                  oSettings.remove(OperationalTextKey.TARGET_NAME);
               }

               // Parrot back settings
               if ((setting = iSettings.get(OperationalTextKey.HEADER_DIGEST)) != null)
               {
                  oSettings.add(OperationalTextKey.HEADER_DIGEST, "None");
               }
               if ((setting = iSettings.get(OperationalTextKey.DATA_DIGEST)) != null)
               {
                  oSettings.add(OperationalTextKey.DATA_DIGEST, "None");
               }
               if ((setting = iSettings.get(OperationalTextKey.ERROR_RECOVERY_LEVEL)) != null)
               {
                  oSettings.add(OperationalTextKey.ERROR_RECOVERY_LEVEL, "0");
               }
               if ((setting = iSettings.get(OperationalTextKey.INITIAL_R2T)) != null)
               {
                  oSettings.add(OperationalTextKey.INITIAL_R2T, "Yes");
               }
               if ((setting = iSettings.get(OperationalTextKey.IMMEDIATE_DATA)) != null)
               {
                  oSettings.add(OperationalTextKey.IMMEDIATE_DATA, "No");
               }
               if ((setting = iSettings.get(OperationalTextKey.MAX_CONNECTIONS)) != null)
               {
                  oSettings.add(OperationalTextKey.MAX_CONNECTIONS, "1");
               }
               if ((setting = iSettings.get(OperationalTextKey.MAX_OUTSTANDING_R2T)) != null)
               {
                  oSettings.add(OperationalTextKey.MAX_OUTSTANDING_R2T, "1");
               }
               if ((setting = iSettings.get(OperationalTextKey.DATA_PDU_IN_ORDER)) != null)
               {
                  oSettings.add(OperationalTextKey.DATA_PDU_IN_ORDER, "Yes");
               }
               if ((setting = iSettings.get(OperationalTextKey.DATA_SEQUENCE_IN_ORDER)) != null)
               {
                  oSettings.add(OperationalTextKey.DATA_SEQUENCE_IN_ORDER, "Yes");
               }

               oSettings.add(OperationalTextKey.DEFAULT_TIME_2_WAIT, "2");
               oSettings.add(OperationalTextKey.DEFAULT_TIME_2_RETAIN, "20");
               if(!this.parent.discoverySession)
               {
            	   oSettings.add(OperationalTextKey.IMMEDIATE_DATA, "No");
               }
               
               if (_logger.isDebugEnabled())
                  dumpSettingsMap("output from operational negotiation", oSettings);

               // Send
               oSegment =
                     DataSegmentFactory.create(oSettings.asByteBuffer(),
                           DataSegmentFactory.DataSegmentFormat.TEXT,
                           this.parent.maxDataSegmentLength);

               long bytesLeft = oSegment.getLength();
               IDataSegmentIterator iterator = oSegment.iterator();
               while (iterator.hasNext())
               {
                  boolean isFinal = (bytesLeft < this.parent.maxDataSegmentLength);
                  oPdu =
                        this.parent.pduFactory.create(false, isFinal, OperationCode.LOGIN_RESPONSE,
                              "None", "None");
                  oParser = (LoginResponseParser) oPdu.getBasicHeaderSegment().getParser();

                  // Response bookkeeping
                  oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
                        iPdu.getBasicHeaderSegment().getInitiatorTaskTag());
                  oParser.setCurrentStageNumber(iParser.getCurrentStageNumber());
                  oParser.setNextStageNumber(LoginStage.FULL_FEATURE_PHASE);
                  oParser.setInitiatorSessionID(this.parent.sessionID);
                  oParser.setTargetSessionIdentifyingHandle(this.parent.targetSessionIdentifyingHandle);
                  oParser.setStatus(LoginStatus.SUCCESS);
                  oParser.setContinueFlag(!isFinal);

                  oPdu.setDataSegment(iterator.next(this.parent.maxDataSegmentLength));
                  this.parent.send(oPdu);
                  bytesLeft -= this.parent.maxDataSegmentLength;
               }
            }

            // Login accepted
            String targetNameMessage =
                  this.parent.scsiTarget != null ? "target "
                        + this.parent.scsiTarget.getTargetName() : "for discovery";

            _logger.info(this.parent.initiatorName + " logged in " + targetNameMessage);

            this.parent.loginSuccessful = true;
         }

         return false;
      }
   }

   class LogoutTask extends Task
   {
      public LogoutTask(ProtocolDataUnit request)
      {
         super(request);
      }

      public boolean handleRequest() throws IOException, DigestException, InternetSCSIException,
            InterruptedException
      {
         ProtocolDataUnit oPdu =
               this.parent.pduFactory.create(false, true, OperationCode.LOGOUT_RESPONSE, "None",
                     "None");
         LogoutResponseParser oParser =
               (LogoutResponseParser) oPdu.getBasicHeaderSegment().getParser();

         oParser.setResponse(LogoutResponse.CONNECTION_CLOSED_SUCCESSFULLY);

         // Response bookkeeping
         oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
               this.request.getBasicHeaderSegment().getInitiatorTaskTag());

         this.parent.send(oPdu);
         String targetNameMessage =
               this.parent.scsiTarget != null
                     ? " target " + this.parent.scsiTarget.getTargetName()
                     : " for discovery";

         _logger.info(String.format("%s performing logout: %s", this.parent, targetNameMessage));

         return true;
      }
   }

   class TextTask extends Task
   {
      public TextTask(ProtocolDataUnit request)
      {
         super(request);
      }

      public boolean handleRequest() throws IOException, DigestException, InternetSCSIException,
            InterruptedException
      {
         ProtocolDataUnit iPdu, oPdu;
         TextResponseParser oParser;
         IDataSegment oSegment;
         SettingsMap iSettings;
         Tuple3<AbstractMessageParser, IDataSegment, SettingsMap> tuple;

         // First PDU
         iPdu = this.request;
         tuple = this.helperPDU(iPdu);
         iSettings = tuple.getThird();

         // Discovery request
         if (this.parent.discoverySession
               && iSettings.get(OperationalTextKey.SEND_TARGETS).equals("All"))
         {
            // Targets for discovery
            Set<String> targets = this.parent.portal.getDiscoveryList();

            // Text data fields

            ByteArrayOutputStream targetIQNStream = new ByteArrayOutputStream();

            for (String targetIQN : targets)
            {
               SettingsMap oSettings = new SettingsMap();
               _logger.debug("discovered target iqn: " + targetIQN);
               oSettings.add(OperationalTextKey.TARGET_NAME, targetIQN);
               oSettings.add(OperationalTextKey.TARGET_ADDRESS, String.format("%s:%d,%d",
                     this.parent.portalHost, this.parent.portalPort,
                     this.parent.targetPortalGroupTag));

               targetIQNStream.write(oSettings.asByteBuffer().array());
            }

            // Send
            oSegment =
                  DataSegmentFactory.create(ByteBuffer.wrap(targetIQNStream.toByteArray()),
                        DataSegmentFactory.DataSegmentFormat.TEXT, this.parent.maxDataSegmentLength);

            IDataSegmentIterator iterator = oSegment.iterator();

            // TODO: If more than one PDU is needed for response, special handling is required
            boolean isFinal = true;
            oPdu =
                  this.parent.pduFactory.create(false, isFinal, OperationCode.TEXT_RESPONSE,
                        "None", "None");
            oParser = (TextResponseParser) oPdu.getBasicHeaderSegment().getParser();

            // Response bookkeeping
            oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
                  iPdu.getBasicHeaderSegment().getInitiatorTaskTag());
            oParser.setTargetTransferTag(0xffffffff);

            // in the case where a target has not been loaded by the service,
            // well have a zero-length set upon discovery, we check for that here
            if (iterator.hasNext())
               oPdu.setDataSegment(iterator.next(this.parent.maxDataSegmentLength));

            this.parent.send(oPdu);

            return false;
         }

         return false;
      }
   }

   class NopTask extends Task
   {
      public NopTask(ProtocolDataUnit request)
      {
         super(request);
      }

      public boolean handleRequest() throws IOException, DigestException, InternetSCSIException,
            InterruptedException
      {
         // No response required
         if (this.request.getBasicHeaderSegment().getInitiatorTaskTag() == 0xffffffff)
         {
            _logger.debug("Noop: No response required");
            return false;
         }

         ProtocolDataUnit oPdu =
               this.parent.pduFactory.create(false, true, OperationCode.NOP_IN, "None", "None");
         NOPInParser oParser = (NOPInParser) oPdu.getBasicHeaderSegment().getParser();

         oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
               this.request.getBasicHeaderSegment().getInitiatorTaskTag());
         oParser.setTargetTrasferTag(0xffffffff);

         this.parent.send(oPdu);
         return false;
      }
   }

   class TMTask extends Task
   {
      public TMTask(ProtocolDataUnit request)
      {
         super(request);
      }

      public boolean handleRequest() throws IOException, DigestException, InternetSCSIException,
            InterruptedException
      {
         TaskManagementFunctionResponseParser.ResponseCode responseCode;

         // Cancel task
         TaskManagementFunctionRequestParser iParser =
               (TaskManagementFunctionRequestParser) this.request.getBasicHeaderSegment().getParser();

         // If the message isn't ABORT_TASK, don't log task-specific information
         // (the case where ABORT_TASK is called is handled later). 
         if(iParser.getFunction() != TaskManagementFunctionRequestParser.FunctionCode.ABORT_TASK)
         {
        	 _logger.warn(String.format("%s received task management function: %s for itt: %x",
        		     this.parent.toString(), iParser.getFunction(), iParser.getReferencedTaskTag()));
         }
         
         if(iParser.getLogicalUnitNumber() == 0)
         {	 
	         if (iParser.getFunction() == TaskManagementFunctionRequestParser.FunctionCode.ABORT_TASK)
	         {
	            Task task = this.parent.pendingTasks.get(iParser.getReferencedTaskTag());
	            if (task != null)
	            {
	            	_logger.warn(String.format("%s received ABORT_TASK function: %s for itt: %x  task: %s",
	              		     this.parent.toString(), iParser.getFunction(), iParser.getReferencedTaskTag(), task.toString()));
	            	Nexus nexus =
	                     new Nexus(this.parent.initiatorName, this.parent.scsiTarget.getTargetName(),
	                           iParser.getLogicalUnitNumber(), iParser.getReferencedTaskTag());
	               this.parent.scsiTarget.execute(nexus, TaskManagementFunction.ABORT_TASK);
	               task.cancel();
	               responseCode = TaskManagementFunctionResponseParser.ResponseCode.FUNCTION_COMPLETE;
	            }
	            else
	            {
	               _logger.warn(String.format("%s cannot find task to cancel with task tag: %s",
	                     this.parent, iParser.getReferencedTaskTag()));
	               responseCode = TaskManagementFunctionResponseParser.ResponseCode.TASK_DOES_NOT_EXIST;
	            }
	         }
	         else if (iParser.getFunction() == TaskManagementFunctionRequestParser.FunctionCode.LOGICAL_UNIT_RESET)
	         {
	            Nexus nexus =
	                  new Nexus(this.parent.initiatorName, this.parent.scsiTarget.getTargetName(), iParser.getLogicalUnitNumber());
	
	            this.parent.scsiTarget.execute(nexus, TaskManagementFunction.LOGICAL_UNIT_RESET);
	
	            // Abort all tasks
	            for (Map.Entry<Integer, Task> entry : this.parent.pendingTasks.entrySet())
	            {
	               entry.getValue().cancel();
	            }
	
	            // Abort all writes in progress
	            for (Map.Entry<Integer, WriteInProgress> entry : this.parent.writesInProgress.entrySet())
	            {
	               entry.getValue().cancel();
	            }
	
	            responseCode = TaskManagementFunctionResponseParser.ResponseCode.FUNCTION_COMPLETE;
	         }
	         else
	         {
	            _logger.warn(String.format("%s received unknown task management function: %s",
	                  this.parent, iParser.getFunction()));
	            responseCode =
	                  TaskManagementFunctionResponseParser.ResponseCode.TASK_MANAGEMENT_FUNCTION_NOT_SUPPORTED;
	         }
         }
         else
         {
        	 _logger.warn(String.format("%s received task management function for nonexistent logical unit %d: %s",
	                  this.parent, iParser.getLogicalUnitNumber(), iParser.getFunction()));
        	 responseCode = TaskManagementFunctionResponseParser.ResponseCode.LUN_DOES_NOT_EXIST;
         }
         
         // Send response
         ProtocolDataUnit oPdu =
               this.parent.pduFactory.create(false, true, OperationCode.SCSI_TM_RESPONSE, "None",
                     "None");
         TaskManagementFunctionResponseParser oParser =
               (TaskManagementFunctionResponseParser) oPdu.getBasicHeaderSegment().getParser();

         oPdu.getBasicHeaderSegment().setInitiatorTaskTag(
               this.request.getBasicHeaderSegment().getInitiatorTaskTag());
         oParser.setResponse(responseCode);

         this.parent.send(oPdu);
         _logger.debug(String.format("%s responded to task management function: %s for itt: %s",
               this.parent, iParser.getFunction(), iParser.getReferencedTaskTag()));
         return false;
      }
   }

   class SCSITask extends Task
   {
      // TODO: consider removing this in favor of a private CDB member
      protected String cdbString = "-unparsed-";

      public SCSITask(ProtocolDataUnit request)
      {
         super(request);
      }

      public boolean handleRequest() throws IOException
      {
         ByteBuffer iCDB;
         CDB cdb;
         Nexus nexus;
         ProtocolDataUnit iPdu;
         SCSICommandParser iParser;

         // First PDU
         iPdu = this.request;
         iParser = (SCSICommandParser) iPdu.getBasicHeaderSegment().getParser();
         iCDB = iParser.getCDB();

         _logger.debug(String.format("%s handling incoming SCSI Request: %s", this.parent.toString(),
               iPdu.getBasicHeaderSegment().getOpCode()));
                 
         // Build SCSI command
         try
         {
            cdb = this.parent.cdbFactory.decode(iCDB);
         }
         catch (IOException ex)
         {
            _logger.error("scsi cdb decoding error");
            throw ex;
         }
         
         this.cdbString = cdb.toString();
         int opCode = cdb.getOperationCode();
         int initiatorTaskTag = iPdu.getBasicHeaderSegment().getInitiatorTaskTag();

         SCSICommandParser.TaskAttributes iscsiTaskAttributes = iParser.getTaskAttributes();
         boolean isITNexus = false;
         switch (opCode)
         {
            // I_T Nexus
            case ReportLuns.OPERATION_CODE :
            case ReportSupportedTaskManagementFunctions.OPERATION_CODE :
               nexus = new Nexus(this.parent.initiatorName, this.parent.scsiTarget.getTargetName());
               isITNexus = true;
               break;

            // I_T_L Nexus
            case Inquiry.OPERATION_CODE :
               nexus =
                     new Nexus(this.parent.initiatorName, this.parent.scsiTarget.getTargetName(), iParser.getLogicalUnitNumber());
               break;
               // I_T_L{_Q} Nexus
            default :
               if (iParser.getTaskAttributes() == SCSICommandParser.TaskAttributes.UNTAGGED)
               {
                  // UNTAGGED iSCSI requests are SIMPLE SCSI requests on an I_T_L nexus
                  nexus =
                        new Nexus(this.parent.initiatorName,
                              this.parent.scsiTarget.getTargetName(), iParser.getLogicalUnitNumber());
                  iscsiTaskAttributes = SCSICommandParser.TaskAttributes.SIMPLE;
               }
               else
               {
                  // All other iSCSI requests should have an InitiatorTaskTag
                  nexus =
                        new Nexus(this.parent.initiatorName,
                              this.parent.scsiTarget.getTargetName(), iParser.getLogicalUnitNumber(), initiatorTaskTag);
               }
               break;
         }
         
  
         // FIXME: Converting from iSCSI TaskAttributes to SCSI TaskAttribute
         TaskAttribute scsiTaskAttribute;
         switch (iscsiTaskAttributes)
         {
            case SIMPLE :
            case UNTAGGED :
               scsiTaskAttribute = TaskAttribute.SIMPLE;
               break;
            case ACA :
               scsiTaskAttribute = TaskAttribute.ACA;
               break;
            case HEAD_OF_QUEUE :
               scsiTaskAttribute = TaskAttribute.HEAD_OF_QUEUE;
               break;
            case ORDERED :
               scsiTaskAttribute = TaskAttribute.ORDERED;
               break;
            default :
               throw new IllegalArgumentException("Unknown iSCSI task attribute: "
                     + iscsiTaskAttributes);
         }

         Command command = new Command(nexus, cdb, scsiTaskAttribute, initiatorTaskTag, 1);
         
         // The target does not support non-zero logical unit numbers.  Since the MS initiator doesn't trust
         // REPORT LUNS when it gets into an error state, it performs an inquiry walk and attempts to
         // address invalid logical unit numbers (which we erroneously respond to with the information for LUN 0).
         // This code fixes that issue in a hacky manner (in the sense that the check should really be done at the 
         // SCSI target level rather than here).
         //
         // However, if we're addressing an I_T nexus, we don't care what the LUN is.  We put this check way down here to
         // make logging easier.
         if((iParser.getLogicalUnitNumber() != 0) && (!isITNexus))
         {
        	 _logger.warn(String.format("Initiator attempted to address invalid logical unit, command: %s", command));
        	 this.parent.writeResponse(nexus, initiatorTaskTag, Status.CHECK_CONDITION,
                     ByteBuffer.wrap((new LogicalUnitNotSupportedException()).encode()));
        	 return false;
         }
         _logger.debug(String.format("enqueing command: %s", command));
         this.parent.scsiTarget.enqueue(this.parent, command);

         return false;
      }

      @Override
      public String toString()
      {
         return String.format("<%s: cdb: %s, initiatorTaskTag: %x>",
               this.getClass().getSimpleName(), this.cdbString,
               this.request.getBasicHeaderSegment().getInitiatorTaskTag());
      }
   }

   class DataOutTask extends Task
   {
      public DataOutTask(ProtocolDataUnit request)
      {
         super(request);
      }

      public boolean handleRequest() throws Exception
      {
         // Find pending write-in-progress
         int task = this.request.getBasicHeaderSegment().getInitiatorTaskTag();
         WriteInProgress write = this.parent.writesInProgress.get(task);
         if (write == null)
         {
            // This write has not begun or is already complete
            return false;
         }

         // Process DataOut
         write.processDataOut(this.request);
         return false;
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // Network send/recv

   protected ProtocolDataUnit recv() throws DigestException, InternetSCSIException,
         ConnectionClosedException, IOException
   {
      recvLock.lock();
      try
      {
         ProtocolDataUnit pdu = this.pduFactory.create("None", "None");

         int bytesRead = pdu.read(this.socket);
         if (-1 == bytesRead)
         {
            // Socket has been closed
            throw new ConnectionClosedException("Connection closed by remote host");
         }

         InitiatorMessageParser parser =
               (InitiatorMessageParser) pdu.getBasicHeaderSegment().getParser();

         int expectedStatSN = parser.getExpectedStatusSequenceNumber();
         if (expectedStatSN > this.expectedStatSN.getValue())
         {
            this.expectedStatSN.setValue(expectedStatSN);
         }

         int cmdSN = parser.getCommandSequenceNumber();
         if (cmdSN > 0 || this.expectedCmdSN.getValue() == 0)
         {
            SerialArithmeticNumber tmp = new SerialArithmeticNumber(cmdSN);
            tmp.increment();
            this.expectedCmdSN.setValue(tmp.getValue());
         }

         return pdu;
      }
      finally
      {
         recvLock.unlock();
      }
   }

   protected void send(ProtocolDataUnit pdu) throws InternetSCSIException, IOException,
         InterruptedException
   {
      sendLock.lock();
      try
      {
         // See if this task has been canceled
         int initiatorTaskTag = pdu.getBasicHeaderSegment().getInitiatorTaskTag();
         Task task = this.pendingTasks.get(initiatorTaskTag);
         if (task != null && task.isCanceled())
         {
            _logger.debug(String.format("%s task cancelled and suppressing response: %s", this,
                  initiatorTaskTag));
            return;
         }

         TargetMessageParser parser = (TargetMessageParser) pdu.getBasicHeaderSegment().getParser();

         if (pdu.getBasicHeaderSegment().isFinalFlag())
         {
            parser.setStatusSequenceNumber(this.expectedStatSN.getValue());

            // Some commands do not advance the expected StatSN
            switch (pdu.getBasicHeaderSegment().getOpCode())
            {
               // Do not advance StatSN
               case R2T :
               case SCSI_DATA_IN :
                  break;

               // Conditionally advance StatSN
               case NOP_IN :
                  if (pdu.getBasicHeaderSegment().getInitiatorTaskTag() != 0xffffffff)
                  {
                     this.expectedStatSN.increment();
                  }
                  break;

               // Advance StatSN
               default :
                  this.expectedStatSN.increment();
                  break;
            }
         }
         else
         {
            parser.setStatusSequenceNumber(0);
         }

         SerialArithmeticNumber maxCmdSN = new SerialArithmeticNumber(expectedCmdSN.getValue());
         parser.setExpectedCommandSequenceNumber(maxCmdSN.getValue());
         for (int i = Math.max(this.taskQueueSize.get() - this.pendingTasks.size(), 0); i > 0; --i)
         {
            maxCmdSN.increment();
         }
         parser.setMaximumCommandSequenceNumber(maxCmdSN.getValue());

         pdu.write(this.socket);
      }
      finally
      {
         sendLock.unlock();
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // Write handling

   protected class WriteInProgress
   {
      protected static final int TARGET_TRANSFER_TAG = 0xcafebabe;

      // Task tag
      protected int initiatorTaskTag;

      // Current serial number of r2t messages
      protected int r2tsn;

      // Data to write
      protected byte[] data;

      // Current length of data read into the data buffer
      protected int read;

      // Expected length of write
      private int length;

      // Parent target instance
      private PrototypeiSCSITarget parent;

      // Info for last r2t sent
      protected int lastR2TOffset;
      protected int lastR2TLength;

      // Was the transfer canceled?
      private AtomicBoolean isCanceled;

      // Latch is zero when all data is available
      protected CountDownLatch recvDone;

      public WriteInProgress(int length, int initiatorTaskTag)
      {
         this.parent = PrototypeiSCSITarget.this;

         this.length = length;
         this.initiatorTaskTag = initiatorTaskTag;

         this.r2tsn = 0;
         this.read = 0;
         this.data = new byte[this.length];

         this.lastR2TOffset = -1;
         this.lastR2TLength = -1;

         this.recvDone = new CountDownLatch(1);
         this.isCanceled = new AtomicBoolean(false);
      }

      /**
       * Start transfer operation
       * @throws InternetSCSIException
       * @throws IOException
       * @throws InterruptedException
       */
      public void start() throws InternetSCSIException, IOException, InterruptedException
      {
         // Send first R2T
         this.sendNextR2T();
      }

      /**
       * Returns true iff the transfer is complete
       * 
       * @return True iff the transfer is complete
       */
      public boolean isDone()
      {
         return (this.read >= this.length);
      }

      /**
       * Get the next R2T PDU in transfer
       * 
       * @return Next R2T PDU in transfer
       * @throws InterruptedException
       * @throws IOException
       * @throws InternetSCSIException
       */
      public void sendNextR2T() throws InternetSCSIException, IOException, InterruptedException
      {
         ProtocolDataUnit r2t =
               this.parent.pduFactory.create(false, true, OperationCode.R2T, "None", "None");
         Ready2TransferParser r2tParser =
               (Ready2TransferParser) r2t.getBasicHeaderSegment().getParser();

         r2t.getBasicHeaderSegment().setInitiatorTaskTag(this.initiatorTaskTag);
         r2tParser.setTargetTransferTag(TARGET_TRANSFER_TAG);
         r2tParser.setReady2TransferSequenceNumber(this.r2tsn);

         this.lastR2TLength = Math.min(this.parent.maxDataSegmentLength, this.length - this.read);

         this.lastR2TOffset = this.read;
         r2tParser.setBufferOffset(this.lastR2TOffset);
         r2tParser.setDesiredDataTransferLength(this.lastR2TLength);

         this.r2tsn++;
         this.parent.send(r2t);
      }

      /**
       * Process an incoming DataOut PDU
       * 
       * @param dataOut
       *           Incoming DataOut PDU
       * @throws InterruptedException
       * @throws IOException
       * @throws InternetSCSIException
       */
      public void processDataOut(ProtocolDataUnit dataOut) throws InternetSCSIException,
            IOException, InterruptedException
      {
         DataOutParser dataParser = (DataOutParser) dataOut.getBasicHeaderSegment().getParser();
         IDataSegment dataSegment =
               DataSegmentFactory.create(dataOut.getDataSegment(),
                     dataParser.getDataSegmentFormat(), this.parent.maxDataSegmentLength);

         if (dataParser.getTargetTransferTag() != TARGET_TRANSFER_TAG)
         {
            // This request is unsolicited
            return;
         }

         int bufferOffset = dataParser.getBufferOffset();
         int bufferLength = dataSegment.getLength();

         // If this is not the most recently requested DataOut, it was sent unsolicited
         if ((bufferOffset != this.lastR2TOffset) || (bufferLength != this.lastR2TLength))
         {
            throw new InternetSCSIException("Out of order Data-Out");
         }

         ByteBuffer buffer = dataOut.getDataSegment();
         buffer.rewind();
         buffer.get(this.data, bufferOffset, bufferLength);

         read += bufferLength;

         // Send next R2T
         if (!this.isDone())
         {
            this.sendNextR2T();
         }
         else
         {
            this.recvDone.countDown();
         }
      }

      public byte[] getData()
      {
         return this.data;
      }

      public long getLength()
      {
         return this.length;
      }

      public int getInitiatorTaskTag()
      {
         return this.initiatorTaskTag;
      }

      /**
       * Block until all data is received
       * @throws InterruptedException
       * @return boolean True if operation finished, false if canceled
       * @throws InterruptedException 
       */
      public boolean join() throws InterruptedException
      {
         try
         {
            this.recvDone.await();
         }
         catch (InterruptedException ex)
         {
            this.cancel();
         }
         return !this.isCanceled.get();
      }

      /**
       * Cancel operation and notify
       */
      public void cancel()
      {
         this.isCanceled.set(true);
         this.recvDone.countDown();
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   // Utils

   private static boolean getBooleanConfigProperty(String name, boolean defaultValue)
   {
      try
      {
         PropertiesProvider propProvider = ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);
         String[] property = name.split("\\.");
         return propProvider.getBooleanValue(property);
      }
      catch (ConfigurationException e)
      {
         return defaultValue;
      }
   }
   
   private static String getStringConfigProperty(String name, String defaultValue)
   {
      try
      {
         PropertiesProvider propProvider = ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);
         String[] property = name.split("\\.");
         return propProvider.getStringValue(property);
      }
      catch (ConfigurationException e)
      {
         return defaultValue;
      }
   }

   private static List<String> getStringArrayConfigProperty(String name, List<String> defaultValue)
   {
      try
      {
         PropertiesProvider propProvider = ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);
         String[] property = name.split("\\.");
         return propProvider.getStringValuesList(property);
      }
      catch (ConfigurationException e)
      {
         return defaultValue;
      }
   }

   private static void dumpSettingsMap(String pSettingsTitle, SettingsMap pSettings)
   {
      if (_logger.isTraceEnabled())
      {
         _logger.trace("Settings for " + pSettingsTitle + ":");

         for (Map.Entry<OperationalTextKey, String> entry : pSettings.entrySet())
         {
            _logger.trace("  " + entry.getKey() + ": " + entry.getValue());
         }
      }
   }

   private final void purgeSCSIState()
   {
      if (this.scsiTarget != null)
      {
         Nexus abortNexus = new Nexus(this.initiatorName, this.scsiTarget.getTargetName(), 0);
         _logger.debug(String.format("target calling ABORT_TASK_SET on nexus: %s", abortNexus));
         this.scsiTarget.execute(abortNexus, TaskManagementFunction.ABORT_TASK_SET);

         _logger.debug("successfully purged scsi state");
      }
   }

   public String getTargetName()
   {
      return this.targetName;
   }

   public String toString()
   {
      return String.format("<ISCSITarget name: %s initiator: %s>", this.targetName,
            this.initiatorName);
   }
}
