//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: bcilfone
//
// Date: Dec 19, 2007
//---------------------

package org.cleversafe.layer.iscsi;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.cleversafe.layer.iscsi.exceptions.ISCSIBinaryValueDecodingException;

/**
 * Contains some methods to do encoding and decoding of binary values for the format
 * defined by the iSCSI RFC
 */

public class ISCSIUtils
{
   /**
    * Encode the given byte as an iSCSI large-binary-value in our preferred encoding (hex)
    * 
    * @param b
    * @return
    */
   public static String encodeBinaryValue(byte b)
   {
      return encodeBinaryValue(new byte[]{
         b
      });
   }

   /**
    * Encode the given byte array as an iSCSI large-binary-value in our preferred encoding (hex)
    * 
    * @param bytes
    * @return
    */
   public static String encodeBinaryValue(byte[] bytes)
   {
      return encodeBinaryValueHex(bytes);
   }

   /**
    * Encode the given byte array as an iSCSI large-binary-value in hex
    * 
    * @param bytes
    * @return
    */
   public static String encodeBinaryValueHex(byte[] bytes)
   {
      StringBuffer sb = new StringBuffer("0x");

      sb.append(String.copyValueOf(Hex.encodeHex(bytes)));

      return sb.toString();
   }

   /**
    * Encode the given byte array as an iSCSI large-binary-value in Base 64
    * 
    * @param bytes
    * @return
    */
   public static String encodeBinaryValueBase64(byte[] bytes)
   {
      StringBuffer sb = new StringBuffer("0b");

      sb.append(new String(Base64.encodeBase64(bytes)));

      return sb.toString();
   }

   /**
    * Decode the given string into a byte array using the following rules:
    *  If the string starts with "0x" or "0X", decode the remainder of the string
    *  as byte hex codes.
    *  If the string starts with "0b" or "0B", decode the remainder of the string
    *  as base 64 encoded data.
    *  
    * @param value
    * @return
    * @throws ISCSIBinaryValueDecodingException 
    */
   public static byte[] decodeBinaryValue(String value) throws ISCSIBinaryValueDecodingException
   {
      try
      {
         if (value.startsWith("0b") || value.startsWith("0B"))
         {
            return Base64.decodeBase64(value.substring(2).getBytes());
         }
         else if (value.startsWith("0x") || value.startsWith("0X"))
         {
            return Hex.decodeHex(value.substring(2).toCharArray());
         }
         else
         {
            //  If the string does not begin with the Base64 or hex indicator, it is a
            //  decimal representation of a byte
            int byte_val = Integer.parseInt(value);

            return new byte[]{
               (byte) byte_val
            };
         }
      }
      catch (Exception e)
      {
         throw new ISCSIBinaryValueDecodingException("Could not decode value: " + value, e);
      }
   }
}
