//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: gdhuse
//
// Date: May 4, 2007
//---------------------

package org.cleversafe.layer.block.dsd.message;
import java.io.*;

/**
 * Base message class for communication between the block layer and 
 * system device driver
 */
public abstract class DeviceMessage
{  
   /**
    * Deserialize object from an input stream
    * @param in
    */
   public abstract void deserialize( DataInputStream in ) throws IOException;
   
   /**
    * Serialize object to an output stream
    * @param out
    */
   public abstract void serialize( DataOutputStream out ) throws IOException;
   
   /**
    * Message header found at the beginning of every message.  The message's 
    * (de)serialize methods are responsible for reading or writing this 
    * header and using its data to construct the message.
    */
   public static class Header
   {
      // Protocol magic number
      protected static final int DSD_MAGIC = 0xD5;
      protected static final int MESSAGE_LENGTH = 6; //8;
      
      // Message type (FIXME: This should probably be an enum) 
      public int messageType;
      
      // Message payload length, in bytes
      public int length;
      
      /**
       * Construct an empty message header
       */
      public Header()
      {
         this.messageType = -1;
         this.length = -1;
      }

      /**
       * Construct a new message header
       * @param messageType Messsage ID
       * @param length Length of payload in bytes
       */
      public Header( int messageType, int length )
      {
         this.messageType = messageType;
         this.length = length;
      }

      /**
       * Read Header from an input stream
       * @param in
       * @param peek If true, do not consume data from the stream
       * @throws IOException
       * @throws ClassNotFoundException
       */
      public void deserialize( DataInputStream in, boolean peek ) 
         throws IOException
      {
         // Make sure the stream can be marked
         if( peek && !in.markSupported() )
         {
            throw new IOException( "Peek is not supported on this stream" );
         }
         
         // Mark the stream for later reset
         if( peek )
         {
            in.mark( MESSAGE_LENGTH );
         }
         
         // Check magic number
         int magic = in.readUnsignedByte();
         if( magic != DSD_MAGIC )
         {
            throw new IOException( "Bad magic number: " + magic );
         }
             
         this.messageType = in.readUnsignedByte();
         //in.skip( 2 );  // Align to 4-byte boundary
         this.length = in.readInt();  
         
         // Roll stream back to mark
         if( peek )
         {
            in.reset();
         }
      }

      /**
       * Write header to an output stream
       * @param out
       * @throws IOException
       */
      public void serialize( DataOutputStream out ) throws IOException
      {
         out.writeByte( DSD_MAGIC );
         out.writeByte( this.messageType );
         //out.writeByte( 0x0 );   // Align to 4-byte boundary
         //out.writeByte( 0x0 );
         out.writeInt( this.length );
      }      
   }      
}


