//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: gdhuse
//
// Date: Jun 14, 2007
//---------------------

package org.cleversafe.layer.block.dsd;

import java.io.EOFException;
import java.io.IOException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.log4j.Logger;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.layer.block.BlockDeviceVault;
import org.cleversafe.layer.block.dsd.message.DeviceMessage;
import org.cleversafe.layer.block.dsd.message.ReadBlocksDeviceMessage;
import org.cleversafe.layer.block.dsd.message.ReadBlocksResponseDeviceMessage;
import org.cleversafe.layer.block.dsd.message.WriteBlocksDeviceMessage;
import org.cleversafe.layer.block.dsd.message.WriteBlocksResponseDeviceMessage;
import org.cleversafe.layer.grid.exceptions.ControllerInvalidParametersException;

/**
 * Controller for a dsd system block device. Mediates requests from the SystemBlockDevice and
 * responses from the SliceStoreController
 */
public class SystemBlockDeviceController extends BlockDeviceController implements Runnable
{
   private static Logger _logger = Logger.getLogger(SystemBlockDeviceController.class);

   // Link to the system device driver
   protected SystemBlockDevice device;

   // Responsible for running DeviceJobs that process
   // DeviceMessages and generate GridJobs
   protected ExecutorService deviceJobExecutor;

   /**
    * Construct a controller for a real system device
    * 
    * @param vault
    *           Vault that represents this device
    * @param controller
    *           Controller responsible for slice storage to the grid
    * @param device
    * @throws ControllerInvalidParametersException 
    */
   public SystemBlockDeviceController(BlockDeviceVault vault, SystemBlockDevice device)
         throws ConfigurationException, ControllerInvalidParametersException
   {
      super(vault);
      this.device = device;
   }

   public void startup()
   {
      super.startup();
      this.deviceJobExecutor = Executors.newSingleThreadExecutor();
   }

   @Override
   public void shutdown()
   {
      super.shutdown();
      this.deviceJobExecutor.shutdown();
   }

   /**
    * Run, continually processing requests from the system and grid FIXME: Add a mechanism for
    * graceful shutdown
    */
   public void run()
   {
      // Process messages
      try
      {
         while (true)
         {
            // Retrieve next request and pass it to the DeviceJob executor
            DeviceMessage message = this.device.getMessage();
            _logger.trace("Got a DeviceMessage: " + message.toString());

            // Submit a job to handle this message
            // FIXME: This is ghetto, n'cest pas?
            if (message instanceof ReadBlocksDeviceMessage)
            {
               this.deviceJobExecutor.submit(new ReadBlocksDeviceJob(
                     (ReadBlocksDeviceMessage) message));
            }
            else if (message instanceof WriteBlocksDeviceMessage)
            {
               this.deviceJobExecutor.submit(new WriteBlocksDeviceJob(
                     (WriteBlocksDeviceMessage) message));
            }
         }
      }
      catch (EOFException e)
      {
         _logger.info("Block device initiated disconnect");
      }
      catch (IOException e)
      {
         _logger.error("System device error", e);
      }
   }

   /**
    * Job to process a read request from the device
    */
   public class ReadBlocksDeviceJob implements Callable<Object>
   {
      // Raw message from the device to process
      protected ReadBlocksDeviceMessage message;

      public ReadBlocksDeviceJob(ReadBlocksDeviceMessage message)
      {
         this.message = message;
      }

      /**
       * Convert read request into one or more ReadBlockGridJobs, then send a response back to the
       * device
       */
      public Object call() throws Exception
      {
         SystemBlockDeviceController sbdc = SystemBlockDeviceController.this;
         byte[] data = sbdc.readBlocks(this.message.getFirstBlock(), this.message.getNumBlocks());

         // Send the result to the device
         ReadBlocksResponseDeviceMessage response =
               new ReadBlocksResponseDeviceMessage(this.message.getRequestTag(), data);
         sbdc.device.sendMessage(response);

         return null;
      }
   }

   /**
    * Job to process a write request from the device
    */
   public class WriteBlocksDeviceJob implements Callable<Object>
   {
      // Raw device request to process
      protected WriteBlocksDeviceMessage message;

      public WriteBlocksDeviceJob(WriteBlocksDeviceMessage message)
      {
         this.message = message;
      }

      /**
       * Convert a write request into one or more WriteBlockGridJobs, then send a response back to
       * the device.
       */
      public Object call() throws Exception
      {
         SystemBlockDeviceController sbdc = SystemBlockDeviceController.this;

         int status = 0;
         try
         {
            sbdc.writeBlocks(this.message.getFirstBlock(), this.message.getNumBlocks(),
                  this.message.getData());
         }
         catch (Exception e)
         {
            // FIXME
            status = -1;
         }

         WriteBlocksResponseDeviceMessage response =
               new WriteBlocksResponseDeviceMessage(this.message.getRequestTag(), status);
         sbdc.device.sendMessage(response);

         return null;
      }
   }
}
