//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: gdhuse
//
// Date: May 9, 2007
//---------------------

package org.cleversafe.layer.block.dsd;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.cleversafe.codec.Codec;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.cauchy.CauchyInformationDispersalCodec;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.layer.block.BlockDeviceVault;
import org.cleversafe.layer.grid.exceptions.ControllerInvalidParametersException;
import org.cleversafe.layer.slicestore.SliceStore;

/**
 * Main class for block device daemon
 */
public class BlockDeviceMain
{
   // Configuration
   public static final int GRID_PILLARS = 16;
   public static final int IDA_THRESHOLD = 12;
   public static final String GRID_PATH = "Grid";
   public static final long DEVICE_SIZE = 32 * 1024 * 1024;
   public static final int BLOCK_SIZE = 4096;

   private static Logger _logger = Logger.getLogger(BlockDeviceMain.class);

   public static void main(String[] args) throws IDAInvalidParametersException
   {
      // Set up log4j
      Logger rootLogger = Logger.getRootLogger();
      rootLogger.setLevel(org.apache.log4j.Level.DEBUG);
      rootLogger.addAppender(new org.apache.log4j.ConsoleAppender(
            new org.apache.log4j.SimpleLayout(), "System.out"));

      // Parse command line parameters
      // FIXME: This is just for debugging
      int staticPort = -1;
      for (String arg : args)
      {
         String opt = "--staticPort=";
         if (arg.startsWith(opt))
         {
            staticPort = Integer.parseInt(arg.substring(opt.length()));
         }
      }

      // Initialize slice storage
      File gridBase = new File(GRID_PATH);
      gridBase.mkdirs();

      List<SliceStore> storage = new ArrayList<SliceStore>(GRID_PILLARS);
      for (int i = 0; i < GRID_PILLARS; ++i)
      {
         File ssPath = new File(gridBase, String.format("ss%d", i));

         try
         {
            // TODO: Fix SliceStore
            SliceStore ss = null; //new BDBSliceStore(ssPath);
            storage.set(i, ss);
         }
         catch (Exception e)
         {
            _logger.error(String.format("Unable to connect to pillar: %d", i), e);
            storage.set(i, null);
         }
      }

      // Initialize IDA
      InformationDispersalCodec ida =
            new CauchyInformationDispersalCodec(GRID_PILLARS, IDA_THRESHOLD);

      // Initialize codecs
      List<Codec> codecs = new LinkedList<Codec>();
      List<Codec> sliceCodecs = new LinkedList<Codec>();

      // Initialize vault
      BlockDeviceVault vault =
            new BlockDeviceVault(BLOCK_SIZE, DEVICE_SIZE / BLOCK_SIZE, ida, codecs, sliceCodecs,
                  storage);

      // Create system block device
      SystemBlockDevice device;
      try
      {
         if (staticPort > 0)
         {
            // Listen on the provided port for a connection from an
            // externally-created system device
            device = new SystemBlockDevice(vault.getNumBlocks(), vault.getBlockSize(), staticPort);
         }
         else
         {
            // Initiate device creation directly
            device = new SystemBlockDevice(vault.getNumBlocks(), vault.getBlockSize());
         }
      }
      catch (IOException e)
      {
         _logger.error("Error creating system block device", e);
         return;
      }

      // Start controller
      SystemBlockDeviceController controller = null;
      try
      {
         controller = new SystemBlockDeviceController(vault, device);
      }
      catch (ConfigurationException e)
      {
         _logger.error("Error creating system block device controller", e);
         return;
      }
      catch (ControllerInvalidParametersException e)
      {
         _logger.error("Error creating system block device controller", e);
         return;
      }

      controller.startup();
      controller.run();
      controller.shutdown();
   }
}
