//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 6, 2007
//---------------------

package org.cleversafe.layer.access.managers;

import java.io.File;
import java.net.URI;
import java.security.KeyPair;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.CredentialsManager;
import org.cleversafe.authentication.PropertiesFileCredentialsManager;
import org.cleversafe.authentication.exceptions.CredentialsException;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.layer.access.GridAccessManager;
import org.cleversafe.layer.access.Service;
import org.cleversafe.layer.access.ServiceInterface;
import org.cleversafe.layer.access.exceptions.AccessConfigurationException;
import org.cleversafe.layer.access.exceptions.AccessIOException;
import org.cleversafe.layer.access.exceptions.AccessLayerException;
import org.cleversafe.layer.access.exceptions.AccessStateModificationException;
import org.cleversafe.layer.access.exceptions.InvalidVaultLoadException;
import org.cleversafe.layer.access.exceptions.InvalidVaultTypeException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceStateException;
import org.cleversafe.layer.access.exceptions.ServiceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceStartStopException;
import org.cleversafe.layer.access.managers.state.LocalFileServiceInterfaceWrapper;
import org.cleversafe.layer.access.managers.state.ServiceInterfaceState;
import org.cleversafe.layer.access.managers.state.ServiceState;
import org.cleversafe.util.Tuple2;
import org.cleversafe.util.Tuple3;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultManager;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.cleversafe.vault.managers.LocalFileVaultManager;
import org.cleversafe.vault.managers.state.VaultState;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.cfg.AnnotationConfiguration;
import org.hibernate.cfg.Environment;
import org.hibernate.classic.Session;

/**
 * A local file implementation of {@link GridAccessManager}. State information is stored in
 * a local directory.
 * <p>
 * This implementation does not allow duplicate service interface types and requires that
 * services have unique (name, type) pairs.
 */
public class LocalFileGridAccessManager implements GridAccessManager
{
   private static Logger _logger = Logger.getLogger(LocalFileGridAccessManager.class);

   public final static String LOCAL_FILE_DATABASE = LocalFileVaultManager.LOCAL_FILE_DATABASE;

   private URI directoryLocation;
   private VaultManager vaultManager;
   
   // Maps of loaded service interfaces (type:obj) and services (name:obj)
   private final ConcurrentMap<String,LocalFileServiceInterfaceWrapper> serviceInterfaces;
   private final ConcurrentMap<Tuple2<String,String>,Service> services;

   
   public LocalFileGridAccessManager()
   {
      this.serviceInterfaces = new ConcurrentHashMap<String,LocalFileServiceInterfaceWrapper>();
      this.services = new ConcurrentHashMap<Tuple2<String,String>,Service>();
   }

   public synchronized GridAccessManager load(URI directory) throws AccessLayerException
   {
      this.directoryLocation = directory;

      loadDirectory(directory);
      
      try
      {
         this.vaultManager = 
            _getBindingImplementation(VaultManager.class, directory.getScheme());
         if (this.vaultManager instanceof LocalFileVaultManager)
         {
            ((LocalFileVaultManager)vaultManager).load(directory, getSessionFactory(directory));
         }
         else
         {
            this.vaultManager.load(directory);
         }
      }
      catch (VaultIOException e)
      {
         throw new AccessIOException("unable to load vault manager", e);
      }
      
      try
      {
         this.loadServiceInterfaces();
      }
      catch (AccessLayerException e)
      {
         try
         {
            this.vaultManager.close();
         }
         catch (VaultIOException ex)
         {
            _logger.info("Could not close connection to vault manager", ex);
         }
         throw e;
      }
      return this;
   }
   
   public void shutdown() throws ServiceInterfaceStartStopException, ServiceStartStopException
   {
      try
      {
         this.vaultManager.close();
      }
      catch (VaultIOException e)
      {
         throw new ServiceInterfaceStartStopException("Could not shutdown vault manager", e);
      }
      for ( ServiceInterface<Service> serviceInterface : this.serviceInterfaces.values() )
      {
         serviceInterface.stop();
      }
      LocalFileGridAccessManager.shutdown(this.directoryLocation);
   }

   protected void checkInitialization()
   {
      if (this.directoryLocation == null || this.vaultManager == null)
      {
         throw new InitializationException("Vault manager not properly initialized");
      }
   }

   ////////////////////////////////////////////////////////////////////////////////////////////////
   
   private static Map<URI, SessionFactory> _sessionFactoryMap;
   private static Properties _configuration;
   
   static
   {
      _sessionFactoryMap = new HashMap<URI, SessionFactory>();
      
      _configuration = new Properties();
      // FIXME: Allow configuration from within code instead of using hibernate.cfg.xml file.
//      Properties c = _configuration;

//      c.setProperty("hibernate.connection.driver_class", "org.apache.derby.jdbc.EmbeddedDriver");
//      c.setProperty("hibernate.dialect", "org.hibernate.dialect.DerbyDialect");
//      c.setProperty("hibernate.hbm2dll.auto", "update");
//
//      c.setProperty("hibernate.c3p0.min_size", "5");
//      c.setProperty("hibernate.c3p0.max_size", "20");
//      c.setProperty("hibernate.c3p0.timeout", "300");
//      c.setProperty("hibernate.c3p0.max_statements", "50");
//      c.setProperty("hibernate.c3p0.idle_test_period", "3000");
//
//      c.setProperty("show_sql", "true");
//      c.setProperty("format_sql", "true");
   }
   
   private static String getDerbyURL(URI directory)
   {
      File dir = new File(directory.getPath());
      return "jdbc:derby:" + dir + File.separator + LOCAL_FILE_DATABASE;
   }
   
   // Attempts to close (unlock) database in case it has become wedged (locked after a process died).
   protected static void cleanupWedgedDatabase(URI directory) throws AccessIOException
   {
      try
      {
         Class.forName("org.apache.derby.jdbc.EmbeddedDriver").newInstance();
         Connection conn = DriverManager.getConnection(getDerbyURL(directory) + ";shutdown=true");
         conn.close();
      }
      catch (SQLException e)
      {
         // should receive an error indicating that database was shutdown.
         _logger.trace("Received expected error after database shutdown", e);
      }
      catch (Exception e)
      {
         throw new AccessIOException("Error unlocking database", e);
      }
   }

   protected static void loadDirectory(URI directory) throws AccessIOException
   {
      synchronized (_sessionFactoryMap)
      {
         if (!directory.getScheme().equals("file"))
         {
            throw new RuntimeException(
                  "Local file grid access manager requires 'file' directory schema");
         }
         
         File dir = new File(directory.getPath());

         _logger.debug("Loading directory from database: " + dir);

         cleanupWedgedDatabase(directory);
         
         _configuration.setProperty(Environment.URL, getDerbyURL(directory) + ";create=True");
         
         _logger.trace("Inserting " + Environment.URL + " = " +
               _configuration.getProperty(Environment.URL));
         
         try
         {
            SessionFactory factory = new AnnotationConfiguration()
                     .addAnnotatedClass(org.cleversafe.vault.managers.state.VaultState.class)
                     .addAnnotatedClass(org.cleversafe.layer.access.managers.state.ServiceState.class)
                     .addAnnotatedClass(org.cleversafe.layer.access.managers.state.ServiceInterfaceState.class)
                     .addProperties(_configuration)
                     .configure()
                     .buildSessionFactory();
            _sessionFactoryMap.put(directory, factory);
         }
         catch (Throwable ex)
         {
            throw new AccessIOException(
                  "Error configuring session factory for directory database", ex);
         }
      }
   }
   
   protected static SessionFactory getSessionFactory(URI directory)
   {
      synchronized (_sessionFactoryMap)
      {
         return _sessionFactoryMap.get(directory); 
      }
   }

   private static void shutdown(URI directory)
   {
      synchronized (_sessionFactoryMap)
      {
         SessionFactory hibernateSessionFactory = _sessionFactoryMap.remove(directory);
         if (hibernateSessionFactory != null)
            hibernateSessionFactory.close();
      }
   }
   
   ////////////////////////////////////////////////////////////////////////////////////////////////
   // FIXME: Static account methods. These methods will be replaced once a full account framework is
   // complete
   
   protected final static String ACCOUNTS_PATH = "accounts";
   protected final static String ACCOUNT_NAME = "dsnet";
   protected final static String ACCOUNT_PASSWORD = "dsnet";
   
   protected static Credentials _getCredentials(URI directory) throws CredentialsException
   {
      CredentialsManager manager = _getCredentialsManager(directory);
      UUID accountIdentifier = manager.getAccountIdentifier(ACCOUNT_NAME);
      return manager.getCredentials(accountIdentifier);
   }
   
   protected static KeyPair _getKeyPair(URI directory) throws CredentialsException
   {
      CredentialsManager manager = _getCredentialsManager(directory);
      UUID accountIdentifier = manager.getAccountIdentifier(ACCOUNT_NAME);
      return manager.getAccountKeyPair(accountIdentifier);
   }
   
   protected static CredentialsManager _getCredentialsManager(URI directory)
   {
      File accountPath = 
         new File(directory.getPath() + File.separator + ACCOUNTS_PATH);
      _logger.trace("Loading credentials manager from " + accountPath);
      PropertiesFileCredentialsManager.setDefaultCredentialsPath(accountPath.getAbsolutePath());
      return new PropertiesFileCredentialsManager();
   }
   
   
   ////////////////////////////////////////////////////////////////////////////////////////////////

   
   private static <T> T _getBindingImplementation(
         final Class<T> interfaceClass,
         final String referral) throws AccessConfigurationException
   {
      try
      {
         return ConfigurationFactory
                     .getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE)
                     .getImplementation(interfaceClass, referral);
      }
      catch (ConfigurationItemNotDefinedException e)
      {
         throw new AccessConfigurationException(
               "Could not load '" + referral + "' " + interfaceClass.getName() + " object", e);
      }
      catch (ObjectInstantiationException e)
      {
         throw new AccessConfigurationException(
               "Could not load '" + referral + "' " + interfaceClass.getName() + " object", e);
      }
      catch (ObjectInitializationException e)
      {
         throw new AccessConfigurationException(
               "Could not load '" + referral + "' " + interfaceClass.getName() + " object", e);
      }
      catch (ConfigurationLoadException e)
      {
         throw new AccessConfigurationException("Could not load bindings configuration", e);
      }
   }

   
   private Tuple2<String, Service> loadService(ServiceState state)
         throws AccessConfigurationException, VaultSecurityException, VaultException,
         InvalidVaultTypeException, CredentialsException, InvalidVaultLoadException
   {
      Service service = _getBindingImplementation(Service.class, state.getType());
      
      Vault vault = this.getVaultManager().loadVault(
            state.getVault().getIdentifier(),
            _getCredentials(this.directoryLocation).getAccountIdentifier(),
            _getCredentialsManager(this.directoryLocation) );
      
      service.load(state.getName(), vault);
      
      // Add new service instance to loaded services map
      this.services.put(new Tuple2<String,String>(state.getType(), state.getName()), service);
      return new Tuple2<String,Service>(state.getName(), service);
   }
   
   
   
   
   @SuppressWarnings("unchecked")
   protected void loadServiceInterfaces() throws AccessLayerException, AccessConfigurationException
   {
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         List<ServiceInterfaceState> states =
                  session.createQuery("from ServiceInterfaceState").list();
         for ( ServiceInterfaceState state : states )
         {
            this.serviceInterfaces.put(state.getType(), this.loadServiceInterface(session, state));
         }
         
         tx.commit();
      }
      catch (AccessConfigurationException e)
      {
         for (ServiceInterface serviceInterface : this.serviceInterfaces.values())
         {
            try
            {
               serviceInterface.stop();
            }
            catch (AccessLayerException ex)
            {
               _logger.info(String.format(
                     "Could not stop %s interface on %s:%i",
                     serviceInterface.getType(),
                     serviceInterface.getHost(),
                     serviceInterface.getPort()), ex);
            }
         }
         throw new AccessConfigurationException(
               "Invalid service interface configuration, " + e.getMessage(), e);
      }
      catch (AccessLayerException e)
      {
         for (ServiceInterface serviceInterface : this.serviceInterfaces.values())
         {
            try
            {
               serviceInterface.stop();
            }
            catch (AccessLayerException ex)
            {
               _logger.info(String.format(
                     "Could not stop %s interface on %s:%i",
                     serviceInterface.getType(),
                     serviceInterface.getHost(),
                     serviceInterface.getPort()), ex);
            }
         }
         throw e;
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }
   
   /**
    * Loads the given service interface and all services currently available through that
    * service interface.
    * @param session An active database session.
    * @param state The state object representing the service interface to be loaded.
    * @return A loaded service interface. The {@link ServiceInterface#start()} method will
    *    have been called if autostart is indicated in the service interface state.
    */
   @SuppressWarnings("unchecked")
   protected LocalFileServiceInterfaceWrapper loadServiceInterface(
         Session session,
         ServiceInterfaceState state) throws AccessConfigurationException,
         AccessStateModificationException, ServiceInterfaceStartStopException,
         ServiceStartStopException
   {
      ServiceInterface<Service> serviceInterface = 
         _getBindingImplementation(ServiceInterface.class, state.getType());
      serviceInterface.load(state.getHost(), state.getPort());
      for (ServiceState serviceState : state.getServices())
      {
         try
         {
            Tuple2<String,Service> tuple = this.loadService(serviceState);
            // Add the service to the service interface we are loading
            serviceInterface.add(tuple.getSecond());
         }
         catch (CredentialsException e)
         {
            throw new ServiceStartStopException(
                  "error loading account credentials prevented service from starting", e);
         }
         catch (VaultException e)
         {
            throw new ServiceStartStopException(
                  "error loading vault prevented service from starting", e);
         }
         catch (InvalidVaultTypeException e)
         {
            throw new ServiceStartStopException(
                  "invalid vault type prevented service from starting", e);
         }
         catch (InvalidVaultLoadException e)
         {
            throw new ServiceStartStopException("service unable to load vault");
         }
      }
      
      if (state.isStartAutomatically())
      {
         serviceInterface.start();
      }
      
      return new LocalFileServiceInterfaceWrapper(
               getSessionFactory(this.directoryLocation),
               state,
               serviceInterface,
               this.services);
   }
   
   ////////////////////////////////////////////////////////////////////////////////////////////////

   public synchronized ServiceInterface<Service> createServiceInterface(
         String type,
         String host,
         int port,
         boolean autostart) throws AccessIOException, AccessConfigurationException,
         ServiceInterfaceStartStopException
   {
      this.checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         ServiceInterfaceState state = (ServiceInterfaceState)session
                  .createQuery("from ServiceInterfaceState s where s.type is :type")
                  .setParameter("type", type)
                  .uniqueResult();
         if ( state != null )
            throw new AccessIOException("duplicate service interface exists: " + type);
         
         state = new ServiceInterfaceState();
         state.setType(type);
         state.setHost(host);
         state.setPort(port);
         state.setStartAutomatically(autostart);
         
         session.save(state);
         
         
         tx.commit();
         
         try
         {
            LocalFileServiceInterfaceWrapper serviceInterface = 
               this.loadServiceInterface(session, state);
            
            this.serviceInterfaces.put(
                  type,
                  serviceInterface);
            
            return serviceInterface;
         }
         catch (ServiceStartStopException e)
         {
            throw new RuntimeException(
                  "bug: newly created (empty) service interface tried to start services");
         }
         
         
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
      
      
      
   }

   public synchronized void deleteServiceInterface(String type)
         throws AccessIOException, ServiceInterfaceStateException
   {
      this.checkInitialization();
      LocalFileServiceInterfaceWrapper serviceInterface = this.serviceInterfaces.get(type);
      if ( serviceInterface != null && serviceInterface.isRunning() )
         throw new ServiceInterfaceStateException("service interface still running");
      
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         ServiceInterfaceState state = 
            (ServiceInterfaceState)session.merge(serviceInterface.getState());
         
         // Remove services themselves if they no longer belong to any service interfaces
         for ( ServiceState service : state.getServices() )
         {
            // If the service has a single interface, that interface is the one now being removed
            if ( service.getServiceInterfaces().size() == 1 )
               session.delete(service);
            // Also remove the service instance from the loaded services map
            this.services.remove(new Tuple2<String,String>(service.getType(),service.getName()));
         }
         
         // Remove any service interface instance from the map
         this.serviceInterfaces.remove(type);
         
         session.delete(state);
         
         tx.commit();
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }

   }
   
   

   public synchronized Service getService(String type, String name) throws AccessIOException,
         AccessConfigurationException, ServiceNotFoundException, ServiceStartStopException
   {
      this.checkInitialization();
      
      Service service = this.services.get(new Tuple2<String,String>(type,name));
      if ( service != null )
         return service;
      
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         ServiceState state = (ServiceState)session
               .createQuery("from ServiceState s where (s.name is :name and s.type is :type)")
               .setParameter("name", name)
               .setParameter("type", type)
               .uniqueResult();
         if ( state == null )
            throw new ServiceNotFoundException("no such service exists: " + name);
         
         tx.commit();
         
         try
         {
            return this.loadService(state).getSecond();
         }
         catch (CredentialsException e)
         {
            throw new ServiceStartStopException(
                  "error loading account credentials prevented service from starting", e);
         }
         catch (VaultException e)
         {
            throw new ServiceStartStopException(
                  "vault loading error prevented service from loading", e);
         }
         catch (InvalidVaultTypeException e)
         {
            throw new ServiceStartStopException(
                  "invalid vault type prevented service from loading", e);
         }
         catch (InvalidVaultLoadException e)
         {
            throw new ServiceStartStopException("service unable to load vault");
         }
         
      }
      finally
      {
         if(!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }
   
   public synchronized Service getService(UUID vaultIdentifier) throws AccessIOException,
         AccessConfigurationException, ServiceNotFoundException, ServiceStartStopException
   {
      this.checkInitialization();
      
      for (Service service : this.services.values())
      {
         if (service.getVaultIdentifier().equals(vaultIdentifier))
            return service;
      }
      
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         VaultState vaultState = (VaultState)session
               .createQuery("from VaultState v where v.identifier is :identifier")
               .setParameter("identifier", vaultIdentifier.toString())
               .uniqueResult();
         ServiceState state = (ServiceState)session
               .createQuery("from ServiceState s where s.vault is :vault")
               .setParameter("vault", vaultState)
               .uniqueResult();
         if (state == null)
            throw new ServiceNotFoundException("No service bound to vault: " + vaultIdentifier);
         
         tx.commit();
         
         try
         {
            return this.loadService(state).getSecond();
         }
         catch (CredentialsException e)
         {
            throw new ServiceStartStopException(
                  "error loading account credentials prevented service from starting", e);
         }
         catch (VaultException e)
         {
            throw new ServiceStartStopException(
                  "vault loading error prevented service from loading", e);
         }
         catch (InvalidVaultTypeException e)
         {
            throw new ServiceStartStopException(
                  "invalid vault type prevented service from loading", e);
         }
         catch (InvalidVaultLoadException e)
         {
            throw new ServiceStartStopException("service unable to load vault");
         }
      }
      finally
      {
         if(!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
      
   }

   public synchronized ServiceInterface<? extends Service> getServiceInterface(String type)
         throws ServiceInterfaceNotFoundException
   {
      this.checkInitialization();
      ServiceInterface<Service> serviceInterface = this.serviceInterfaces.get(type);
      if ( serviceInterface == null )
         throw new ServiceInterfaceNotFoundException("no such service interface: " + type);
      else
         return serviceInterface;
   }
   
   public synchronized List<ServiceInterface<Service>> getServiceInterfaces()
   {
      this.checkInitialization();
      List<ServiceInterface<Service>> ints = new ArrayList<ServiceInterface<Service>>();
      for ( LocalFileServiceInterfaceWrapper s : this.serviceInterfaces.values() )
      {
         ints.add(s);
      }
      return Collections.unmodifiableList(ints);
   }

   @SuppressWarnings("unchecked")
   public synchronized Set<UUID> getVaultIdentifiers(boolean running)
   {
      this.checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         Set <UUID> identifiers = new HashSet<UUID>();
         
         if (running)
         {
            // query running service interfaces for running vaults
            for ( LocalFileServiceInterfaceWrapper serviceInterface : this.serviceInterfaces.values() )
            {
               if ( serviceInterface.isRunning() )
               {
                  ServiceInterfaceState state = 
                     (ServiceInterfaceState)session.merge(serviceInterface.getState());
                  for ( ServiceState service : state.getServices() )
                  {
                     identifiers.add(service.getVault().getIdentifier());
                  }
               }
            }
         }
         else
         {
            // just return a list of vaults
            List<String> ids = session.createQuery("select v.identifier from VaultState v").list();
            for ( String id : ids )
            {
               identifiers.add(UUID.fromString(id));
            }
         }
         
         tx.commit();
         
         return identifiers;
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }
   
   @SuppressWarnings("unchecked")
   public synchronized Set<Tuple3<String,String,UUID>> getServiceInfo(boolean running)
   {
      this.checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         Set<Tuple3<String,String,UUID>> info = new HashSet<Tuple3<String,String,UUID>>();
         
         if (running)
         {
            // query running service interfaces for service information
            for (LocalFileServiceInterfaceWrapper serviceInterface : this.serviceInterfaces.values())
            {
               if (serviceInterface.isRunning())
               {
                  ServiceInterfaceState state = 
                     (ServiceInterfaceState)session.merge(serviceInterface.getState());
                  for (ServiceState service : state.getServices())
                  {
                     info.add(new Tuple3<String,String,UUID>(
                           service.getType(),
                           service.getName(),
                           service.getVault().getIdentifier()));
                  }
               }
            }
         }
         else
         {
            // just return list of services
            List<ServiceState> services = session.createQuery("from ServiceState").list();
            for (ServiceState service : services)
            {
               info.add(new Tuple3<String,String,UUID>(
                     service.getType(),
                     service.getName(),
                     service.getVault().getIdentifier()));
            }
         }
         
         tx.commit();
         
         return info;
         
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }
   
   

   public synchronized VaultManager getVaultManager()
   {
      this.checkInitialization();
      // FIXME: Wrap vault manager so that on success we delete existing services that use
      // the vault. Also, fail (or stop?) the delete operation if the service is still running.
      return this.vaultManager;
   }



}
