//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jan 3, 2008
//---------------------

package org.cleversafe.layer.access.cli;
import java.io.IOException;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.management.InstanceNotFoundException;

import org.cleversafe.layer.access.GridAccessConnector;
import org.cleversafe.layer.access.jmx.VaultStatusMBean;
import org.cleversafe.util.TablePrinter;
import org.cleversafe.util.Tuple2;

// TODO: Describe class or interface
public class ListVaults extends AbstractCLI
{
   private static final String COMMAND_NAME = "dsnet-list-vaults";
   
   private static final String CSV_OUTPUT_FORMAT_OPTION = "csv";
   
   private static final String VAULT_ACTIVE   = "active";
   private static final String VAULT_INACTIVE = "      ";

   
   ////////////////////////////////////////////////////////////////////////////////////////////////
   
   private final boolean csvFormat;
   
   public ListVaults(String[] args) throws ExitStatusException
   {
      super(ListVaults.class,
            COMMAND_NAME,
            "listing_client.jsap",
            "Lists dsNet vaults",
            args);
      this.csvFormat = this.getJSAPResult().getBoolean(CSV_OUTPUT_FORMAT_OPTION);
   }
   
   public ListVaults(String jmxHost, int jmxPort, boolean csvFormat)
   {
      super(COMMAND_NAME, jmxHost, jmxPort);
      this.csvFormat = csvFormat;
   }
   
   private static void printCSVFormat(Map<String,Tuple2<String,String>> map)
   {
      for (Map.Entry<String,Tuple2<String,String>> entry : map.entrySet())
      {
         System.out.println(entry.getKey() + "," + 
               entry.getValue().getFirst() + "," + entry.getValue().getSecond());
      }
   }
   
   public void run()
   {
      GridAccessConnector connector = this.getGridAccessConnector();
      
      try
      {
         connector.connect();
         
         // Get a list of vaults
         VaultStatusMBean vs = connector.getVaultStatusMBean();
         SortedMap<String,Tuple2<String,String>> statusMap = 
            new TreeMap<String,Tuple2<String,String>>();
         
         // Get names and status for each indicated vault
         for (String vaultIdentifier : vs.getVaultList())
         {
            statusMap.put(vaultIdentifier, new Tuple2<String,String>(
                     vs.getVaultName(vaultIdentifier),
                     vs.isVaultLoaded(vaultIdentifier) ? VAULT_ACTIVE : VAULT_INACTIVE));
         }
         
         if (this.csvFormat)
         {
            printCSVFormat(statusMap);
         }
         else
         {
            TablePrinter printer = new TablePrinter(3);
            for (Map.Entry<String, Tuple2<String,String>> entry : statusMap.entrySet())
            {
               printer.add(
                     entry.getValue().getFirst(), 
                     entry.getKey(), 
                     entry.getValue().getSecond());
            }
            printer.print(System.out);
         }
      }
      catch (InstanceNotFoundException e)
      {
         throw new ExitStatusException("Error connecting to access server", IO_ERROR, e);
      }
      catch (IOException e)
      {
         throw new ExitStatusException(
               "Could not connect to vault manager on access server", IO_ERROR, e);
      }
      catch (Exception e)
      {
         throw new ExitStatusException(
               "Error listing vaults on access server", UNKNOWN_ERROR, e);
      }
      finally
      {
         try
         {
            if (connector != null)
            {
               connector.close();
            }
         }
         catch (IOException e) {}
      }
   }
   
   public static void main(String[] args)
   {
      try
      {
         ListVaults client = new ListVaults(args);
         client.finish();
      }
      catch (ExitStatusException e)
      {
         System.exit(e.getStatus());
      }
   }
   
   
   
}


