//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 3, 2007
//---------------------

package org.cleversafe.layer.access;

import java.util.List;

import org.cleversafe.layer.access.exceptions.AccessIOException;
import org.cleversafe.layer.access.exceptions.AccessStateModificationException;
import org.cleversafe.layer.access.exceptions.ServiceConfigurationException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceStillRunningException;

/**
 * An interface through which services are made available to external entities. Individually
 * configured services are made externally available when they are added to a service interface.
 * <p>
 * Service interfaces generally represent an address or port through which services are accessed.
 * Examples of a service interface include a iSCSI Portal or a CIFS server.
 * <p>
 * The <code>ServiceInterface</code> class is generic to provide type safety for the developer.
 * Additionally, service interface implementations must require that for any added services
 * {@link Service#getType()} is equal to {@link ServiceInterface#getType()}.
 */
public interface ServiceInterface<T extends Service>
{

   /**
    * Returns the service type supported by this service interface.
    * <p>
    * This method can be called before {@link #load(String, int)}.
    */
   String getType();
   

   /**
    * Configures this service interface.
    * 
    * @param host
    *           The host or address this service interface will bind to.
    * @param port
    *           The port this service interface will bind to.
    * @throws ServiceConfigurationException
    *            If a runtime library binding error occurs.
    */
   ServiceInterface<T> load(String host, int port) throws ServiceConfigurationException;

   /**
    * Returns the host or address this service interface listens on.
    */
   String getHost();
   
   /**
    * Returns the port this service interface listens on.
    */
   int getPort();
   
   /**
    * Returns <code>true</code> if the service interface starts automatically on access device
    * startup; <code>false</code> otherwise.
    * 
    * @throws AccessIOException If an I/O error occurs while querying this value.
    */
   boolean isStartsAutomatically() throws AccessIOException;
   
   
   /**
    * Sets whether this service interface starts automatically on access device startup. This
    * value may be set while the service interface is running.
    * 
    * @throws AccessStateModificationException If an I/O error occurs while setting this value.
    */
   ServiceInterface<T> setStartsAutomatically(boolean autostart) throws AccessStateModificationException;
   
   
   
   /**
    * Starts this service interface. Starts all available services.
    * 
    * @throws ServiceInterfaceStartStopException
    *            If the service interface does not load correctly.
    * @throws ServiceStartStopException
    *            If the service interface loads correctly but one or more services fail.
    */
   void start() throws ServiceInterfaceStartStopException, ServiceStartStopException;

   /**
    * Stops this service interface. Stops any running services. Any services that were running
    * before shutdown will be restarted automatically the next time the service interface is
    * restarted.
    * 
    * @throws ServiceInterfaceStartStopException
    *            If the service interface does not stop correctly.
    * @throws ServiceStartStopException
    *            If the service interface stops correctly but one or more services fail to stop.
    */
   void stop() throws ServiceInterfaceStartStopException, ServiceStartStopException;

   /**
    * Returns <code>true</code> if the service interface is currently running; returns
    * <code>false</code> otherwise.
    */
   boolean isRunning();

   /**
    * Makes a service available through this service interface.
    * 
    * @param service
    *           The service to add. The service's type must be identical to this service interface.
    * @throws AccessStateModificationException
    *            If an I/O error occurs while saving new access device state.
    */
   void add(T service) throws AccessStateModificationException;

   /**
    * Removes an existing service from this service interface.
    * 
    * @param service
    *           The service to remove.
    * @throws ServiceStillRunningException
    *            If the given service has not yet been stopped.
    * @throws ServiceNotFoundException
    *            If no service with the given name is currently available through this service
    *            interface.
    * @throws AccessStateModificationException
    * @returns The service which was removed from the service interface.
    */
   T remove(T service) throws ServiceStillRunningException,
         AccessStateModificationException, ServiceNotFoundException;

   /**
    * Returns a list of services available through this service interface.
    */
   List<T> getServices();

   /**
    * Returns the names of all services available through this service interface.
    */
   String[] getServiceNames();

   /**
    * Returns the indicated service.
    * 
    * @param serviceName
    *           The name of the service to return.
    * @throws ServiceNotFoundException
    *            If no service with the given name is currently available through this service
    *            interface.
    */
   T getService(String serviceName) throws ServiceNotFoundException;

}
