/**
 * Cleversafe open-source code header - Version 1.1 - December 1, 2006
 *
 * Cleversafe Dispersed Storage(TM) is software for secure, private and
 * reliable storage of the world's data using information dispersal.
 *
 * Copyright (C) 2005-2007 Cleversafe, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 *
 * Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
 * Chicago IL 60616
 * email licensing@cleversafe.org
 *
 * Author: Greg Dhuse <gdhuse@cleversafe.com>
 *
 */

#include "dsdbus.h"
#include "dsdnet.h"


/**
 * Called when the driver is loaded
 */
NTSTATUS
DriverEntry( IN PDRIVER_OBJECT driverObject, 
             IN PUNICODE_STRING registryPath )
{
   NTSTATUS status;
   WDF_DRIVER_CONFIG config;
   WDF_OBJECT_ATTRIBUTES attributes;
   
   KdPrint(( DSDBUS_TAG "--> DriverEntry\n" ));

   // Set driver attributes
   WDF_OBJECT_ATTRIBUTES_INIT( &attributes );

   WDF_DRIVER_CONFIG_INIT( &config, Bus_EvtDeviceAdd );
   status = WdfDriverCreate( driverObject,
                             registryPath,
                             &attributes,
                             &config,
                             WDF_NO_HANDLE );
   if( !NT_SUCCESS(status) )
   {
      KdPrint(( DSDBUS_TAG "WdfDriverCreate failed with status 0x%08x\n",
               status ));
   }
   
   KdPrint(( DSDBUS_TAG "<-- DriverEntry\n" ));
   return status;
}

/**
 * Called when a new device stack that involves this driver is
 * being constructed
 */
NTSTATUS
Bus_EvtDeviceAdd( IN WDFDRIVER driver, 
                  IN PWDFDEVICE_INIT deviceInit )
{
   WDFQUEUE queue;
   NTSTATUS status;
   WDFDEVICE device;
   PNP_BUS_INFORMATION busInfo;
   ANSI_STRING deviceNameAnsi;
   UNICODE_STRING deviceNameUnicode;
   WDF_IO_QUEUE_CONFIG queueConfig;
   WDF_CHILD_LIST_CONFIG childListConfig;

   UNREFERENCED_PARAMETER( driver );
   
   KdPrint(( DSDBUS_TAG "--> Bus_EvtDeviceAdd\n" ));
   
   // Setup bus device
   WdfDeviceInitSetDeviceType( deviceInit, FILE_DEVICE_BUS_EXTENDER );
   WdfDeviceInitSetExclusive( deviceInit, TRUE );

   RtlInitAnsiString( &deviceNameAnsi, DSDBUS_DEVICE_NAME );
   RtlAnsiStringToUnicodeString( &deviceNameUnicode, &deviceNameAnsi, TRUE );
   status = WdfDeviceInitAssignName( deviceInit, &deviceNameUnicode );
   if( !NT_SUCCESS(status) )
   {
      KdPrint(( DSDBUS_TAG "Unable to assign tag: 0x%08x\n", status ));
      return status;
   }

   // Initialize child list
   WDF_CHILD_LIST_CONFIG_INIT( &childListConfig, 
                               sizeof(DSDBUS_CHILD_ID),
                               Bus_EvtChildListCreateDevice );
   WdfFdoInitSetDefaultChildListConfig( deviceInit, 
                                        &childListConfig,
                                        WDF_NO_OBJECT_ATTRIBUTES );

   // Create device
   status = WdfDeviceCreate( &deviceInit, 
                             WDF_NO_OBJECT_ATTRIBUTES, 
                             &device );
   if( !NT_SUCCESS(status) )
   {
      KdPrint(( DSDBUS_TAG "WdfDeviceCreate failed with status 0x%08x\n",
                status ));
      return status;
   }

   // Create event queue
   WDF_IO_QUEUE_CONFIG_INIT_DEFAULT_QUEUE( &queueConfig, 
      WdfIoQueueDispatchSequential );
   queueConfig.EvtIoDeviceControl = Bus_EvtDeviceIoDeviceControl;
   
   status = WdfIoQueueCreate( device, 
                              &queueConfig, 
                              WDF_NO_OBJECT_ATTRIBUTES, 
                              &queue );
   if( !NT_SUCCESS(status) )
   {
      KdPrint(( DSDBUS_TAG "WdfIoQueueCreate failed with status 0x%08x\n", 
         status ));
      return status;
   }

   // Create device interface for bus
   status = WdfDeviceCreateDeviceInterface( 
      device, 
      &GUID_DEVINTERFACE_BUSENUM_DISPERSED_STORAGE,
      NULL );
   if( !NT_SUCCESS(status) )
   {
      KdPrint(( DSDBUS_TAG "WdfDeviceCreateDeviceInterface failed 0x%08x\n",
         status ));
      return status;
   }

   // Uniquely identify this bus
   busInfo.LegacyBusType = PNPBus;
   busInfo.BusNumber     = 0;

   WdfDeviceSetBusInformationForChildren( device, &busInfo );

   KdPrint(( DSDBUS_TAG "<-- Bus_EvtDeviceAdd\n" ));
   return status;
}

/**
 * Enumerate a new child device
 */
NTSTATUS 
Bus_PlugInDevice( IN WDFDEVICE device, 
                  IN uint64_t numBlocks, 
                  IN uint16_t blockSize, 
                  IN int socket )
{
   NTSTATUS status;
   DSDBUS_CHILD_ID childId;   

   KdPrint(( DSDBUS_TAG "--> Bus_PlugInDevice\n" ));

   // Initialize child ID
   RtlZeroMemory( &childId, sizeof(childId) );
   WDF_CHILD_IDENTIFICATION_DESCRIPTION_HEADER_INIT( &childId.header, 
                                                     sizeof(childId) );
   childId.numBlocks    = numBlocks;
   childId.blockSize    = blockSize;
   childId.socket       = socket;

   KdPrint(( DSDBUS_TAG "Debug: socket %d\n", socket ));

   // Report new child
   WdfChildListBeginScan( WdfFdoGetDefaultChildList( device ) );
   status = WdfChildListAddOrUpdateChildDescriptionAsPresent( 
               WdfFdoGetDefaultChildList( device ),
               &childId.header,
               NULL );
   WdfChildListEndScan( WdfFdoGetDefaultChildList( device ) );

   KdPrint(( DSDBUS_TAG "<-- Bus_PlugInDevice\n" ));
   return status;
}

/**
 * Called by the framework when a new child is added
 */
NTSTATUS
Bus_EvtChildListCreateDevice( 
   IN WDFCHILDLIST childList, 
   IN OUT PWDF_CHILD_IDENTIFICATION_DESCRIPTION_HEADER ident,
   IN OUT PWDFDEVICE_INIT childInit )
{
   PDSD_PDO pdo;
   NTSTATUS status;
   WDFDEVICE childDevice;
   PDSDBUS_CHILD_ID childId;
   WDF_OBJECT_ATTRIBUTES childAttributes;
   DECLARE_CONST_UNICODE_STRING( deviceId, DSD_DEVICE_ID );

   UNREFERENCED_PARAMETER( childList );

   KdPrint(( DSDBUS_TAG "--> Bus_EvtChildListCreateDevice\n" ));

   childId = (PDSDBUS_CHILD_ID)CONTAINING_RECORD( ident, 
                                                  DSDBUS_CHILD_ID, 
                                                  header );

   // Assign device and hardware IDs.  This is the device type
   status = WdfPdoInitAssignDeviceID( childInit, &deviceId );
   if( !NT_SUCCESS(status) )
   {
      // FIXME
   }

   status = WdfPdoInitAddHardwareID( childInit, &deviceId );
   if( !NT_SUCCESS(status) )
   {
      // FIXME
   }

#if 0
   /* FIXME: This should eventually be the vault UUID */
   status =  RtlIntegerToUnicodeString( 
      InterlockedIncrement( &instanceIDCounter ),
      10,   /* Base-10 */
      &instanceID );
   if( !NT_SUCCESS(status) )
   {
      // FIXME
   }

   status = WdfPdoInitAssignInstanceID( childInit, &instanceID );
   if( !NT_SUCCESS(status) )
   {
      // FIXME
   }
#endif

   // Initialize device attributes
   WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE( &childAttributes, DSD_PDO );
   status = WdfDeviceCreate( &childInit, &childAttributes, &childDevice );
   if( !NT_SUCCESS(status) )
   {
      // FIXME
   }

   pdo = GetDsdPdo( childDevice );
   pdo->numBlocks    = childId->numBlocks;
   pdo->blockSize    = childId->blockSize;
   pdo->socket       = childId->socket;

   KdPrint(( DSDBUS_TAG "<-- Bus_EvtChildListCreateDevice\n" ));
   return STATUS_SUCCESS;
}

/**
 * Handle bus ioctls
 */
VOID 
Bus_EvtDeviceIoDeviceControl( IN WDFQUEUE queue, 
                              IN WDFREQUEST request, 
                              IN size_t outputBufferLength,
                              IN size_t inputBufferLength,
                              IN ULONG ioctl )
{
   PVOID buffer;
   NTSTATUS status;   
   WDFDEVICE device;

   UNREFERENCED_PARAMETER( inputBufferLength );
   UNREFERENCED_PARAMETER( outputBufferLength );

   KdPrint(( DSDBUS_TAG "--> Bus_EvtDeviceIoDeviceControl\n" ));

   device = WdfIoQueueGetDevice( queue );

   switch( ioctl )
   {
      // Create a new child device
      case DSD_BUS_IOCTL_CREATE_DEVICE:
      {
         int socket;
         struct dsd_bus_ioctl_create_device* devParams;
         struct dsd_bus_ioctl_create_device_rsp* response;

         KdPrint(( DSDBUS_TAG "DSD_BUS_IOCTL_CREATE_DEVICE\n" ));

         // Input buffer
         status = WdfRequestRetrieveInputBuffer( 
            request, sizeof(struct dsd_bus_ioctl_create_device), 
            &buffer, NULL );
         if( status == STATUS_BUFFER_TOO_SMALL )
         {
            KdPrint(( DSDBUS_TAG "Buffer too small: 0x%08x\n", status ));
            break;
         }
         
         devParams = (struct dsd_bus_ioctl_create_device*)buffer;

         // Connect to daemon
         status = DsdNetConnect( devParams->ip_addr, 
                                 devParams->port, 
                                 &socket );
         if( !NT_SUCCESS(status) )
         {
            KdPrint(( DSDBUS_TAG "Unable to connect to daemon\n" ));
            break;
         }

         // Add device to bus
         Bus_PlugInDevice( device,
                           devParams->sectors,
                           devParams->sector_size,
                           socket );

         // Output buffer
         status = WdfRequestRetrieveOutputBuffer( 
            request, sizeof(struct dsd_bus_ioctl_create_device_rsp), 
            &buffer, NULL );
         if( status == STATUS_BUFFER_TOO_SMALL )
         {
            KdPrint(( DSDBUS_TAG "Buffer too small: 0x%08x\n", status ));
            break;
         }
         
         response = (struct dsd_bus_ioctl_create_device_rsp*)buffer;
         RtlZeroMemory( response, sizeof(struct dsd_bus_ioctl_create_device_rsp) );
         response->status = 0;
         RtlStringCbPrintfA( response->device_name, 64, "dynamic" );         

         WdfRequestSetInformation( request, sizeof(struct dsd_bus_ioctl_create_device_rsp) );
         status = STATUS_SUCCESS;
         KdPrint(( DSDBUS_TAG "Success!\n" ));
      }
      break;

      default:
         KdPrint(( DSDBUS_TAG "Unknown ioctl: 0x%08x\n", ioctl ));
         status = STATUS_NOT_IMPLEMENTED;         
         break;
   }

   WdfRequestComplete( request, status ); 
   KdPrint(( DSDBUS_TAG "<-- Bus_EvtDeviceIoDeviceControl\n" ));
}

