/**
 * Cleversafe open-source code header - Version 1.1 - December 1, 2006
 *
 * Cleversafe Dispersed Storage(TM) is software for secure, private and
 * reliable storage of the world's data using information dispersal.
 *
 * Copyright (C) 2005-2007 Cleversafe, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 *
 * Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
 * Chicago IL 60616
 * email licensing@cleversafe.org
 *
 * Author: Greg Dhuse <gdhuse@cleversafe.com>
 *
 */

#include "dsd.h"

int dsd_bus_major = 0;

/* Bus operations */
static int dsd_bus_ioctl( struct inode* inode,
                          struct file* filp,
                          unsigned int ioctl,
                          unsigned long arg );


/* Kernel device operations structure */
static struct file_operations dsd_bus_ops = 
{
   .owner         = THIS_MODULE,
   .ioctl         = dsd_bus_ioctl,
};

/**
 * Handle bus ioctl calls
 */
static int dsd_bus_ioctl( struct inode* inode, 
                          struct file* filp,
                          unsigned int ioctl,
                          unsigned long arg )
{
   int status = -ENOTTY;

   switch( ioctl )
   {
      /* Create device */
      case DSD_BUS_IOCTL_CREATE_DEVICE:
      {
         struct socket* sk;
         struct dsd_dev* dev;
         struct dsd_bus_ioctl_create_device dev_params;
         struct dsd_bus_ioctl_create_device_rsp response;

         memset( &response, 0, sizeof(response) );

         #ifdef DSD_DEBUG
         printk( KERN_ALERT DSD_TAG "ioctl( DSD_BUS_IOCTL_CREATE_DEVICE )\n" );
         #endif

         if(unlikely( copy_from_user( &dev_params,
                                      (struct dsd_bus_ioctl_create_device*)arg,
                                       sizeof(struct dsd_bus_ioctl_create_device) ) ))
         {
            return -EFAULT;
         }

         /* Connect to daemon */
         sk = dsd_connect( dev_params.ip_addr, 
                           dev_params.port );
         if(unlikely( !sk ))
         {
            printk( KERN_ALERT DSD_TAG "Unable to connect to daemon\n" );
            return -EINVAL;
         }

         /* Create device */
         status = dsd_add_device( dev_params.sectors,
                                  dev_params.sector_size,
                                  sk, &dev );
         if(unlikely( status < 0 ))
         {
            printk( KERN_ALERT DSD_TAG "Unable to create device(%d)\n", status );
            return status;
         }

         snprintf( response.device_name, 64, "/dev/%s", dev->gd->disk_name );

         /* Response */
         response.status = 0;
         if(unlikely( copy_to_user( (struct dsd_bus_ioctl_create_device_rsp*)arg,
                                    &response,
                                     sizeof( response ) ) ))
         {
            return -EINVAL;
         }

         status = 0;
      }
      break;

      default:
      {
         printk( KERN_ALERT DSD_TAG "Unhandled ioctl(0x%08x)\n", ioctl );
      }
      break;
   }

   return status;
}

/**
 * Bus initialization
 */
int __init dsd_bus_init( void )
{
   dsd_bus_major = register_chrdev( dsd_bus_major, 
                                    DSD_BUS_DEVICE_NAME, 
                                    &dsd_bus_ops );
   if( dsd_bus_major <= 0 )
   {
      printk( KERN_ALERT DSD_TAG "Unable to register bus device\n" );
      return -EBUSY;
   }
   
   return 0;
}

/**
 * Bus cleanup
 */
void __exit dsd_bus_exit( void )
{
   unregister_chrdev( dsd_bus_major, DSD_BUS_DEVICE_NAME );
}
