package Netpkg; 

# Copyright Jean-Philippe Guillemin <jp.guillemin@free.fr>. This program is free software; you can redistribute
# it and/or modify it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at your option)
# any later version. Please take a look at http://www.gnu.org/copyleft/gpl.htm
#
# Netpkg is a tool for easily install or upgrade packages via the network. With Netpkg,
# you can make a minimal installation of Zenwalk Linux and install/upgrade just the
# packages you need most.
#
#

use strict;
use POSIX;
use LWP::UserAgent ();
use LWP::MediaTypes qw(guess_media_type media_suffix);
use URI ();
use HTTP::Date ();
use POSIX qw/strftime/;
use Locale::gettext;
use Encode;
use Unicode::Normalize; #for accent-insensitive sort
use Storable; 
use threads ('yield',
             'stack_size' => 64*4096,
             'exit' => 'threads_only',
             'stringify');
use threads::shared;
my $semaphorus : shared = 0;
my $command : shared = "sleep";

# Internationalization
setlocale(LC_ALL, "");
textdomain("netpkg");

####################################################################
############################ CONSTANTS  ############################
####################################################################

use constant CATEGORY => 0;
use constant AVAILABLE_PKG => 1;
use constant UID => 2;
use constant SIZE_COMPRESSED => 3;
use constant SIZE_UNCOMPRESSED => 4;
use constant DEPENDENCIES => 5;
use constant AVAILABLE_VERSION => 6;
use constant AVAILABLE_BUILD => 7;
use constant DESCRIPTION => 8;
use constant INSTALLED_PKG => 9;
use constant INSTALLED_VERSION => 10;
use constant INSTALLED_BUILD => 11;
use constant STATUS => 12;
use constant SELECTED_AS_PKG => 13;
use constant SELECTED_AS_DEP => 14;
use constant BLACKLISTED => 15;
use constant PATH => 16;
use constant USED_BY => 17;
use constant INSTALLED_DEPENDENCIES => 18;
use constant MISSING_DEPENDENCIES => 19;
use constant CATEGORY_ICON => 20;
use constant LOCAL_PKG => 21;
use constant LIBS => 22;
use constant BUFFER => 30;

use constant STATUS_OK => 0;
use constant STATUS_DOWNGRADED => -1;
use constant STATUS_UPDATED => 1;
use constant STATUS_NEW => 2;
use constant STATUS_UNAVAILABLE => -2;

use constant PROCESS_PKGS_ONLY => 1;
use constant PROCESS_PKGS_AND_DEPS => 2;

use constant NONE => 			0b00000000;
use constant ALL => 			0b11111111;
use constant NEW => 			0b00001000;
use constant NOT_NEW => 		0b11110111;
use constant UPDATED => 		0b00000100;
use constant NOT_UPDATED => 	0b11111011;
use constant INSTALLED => 		0b00000010;
use constant NOT_INSTALLED => 	0b11111101;
use constant DOWNGRADED => 		0b00000001;
use constant NOT_DOWNGRADED => 	0b11111110;
use constant MODIFIED => 		0b00000101;
use constant NOT_MODIFIED => 	0b11111010;
use constant ORPHANS => 		0b00010000;
use constant NOT_ORPHANS => 	0b11101111;
use constant SHOW_EXTRA => 		0b00100000;
use constant NOT_SHOW_EXTRA => 	0b11011111;


use constant ICON_COLUMN => 0;
use constant SOFT_COLUMN => 1;
use constant AVAILABLE_PKG_COLUMN => 2;
use constant INSTALLED_PKG_COLUMN => 3;
use constant SELECTED_PKG_COLUMN => 4;

use constant WITH_PKG_INFOS => 0;
use constant WITH_SIZES => 1;

####################################################################
sub new { 
	my $self = {};
	bless ($self);	
	return $self;
}

####################################################################
# $Netpkg->GetMirrors($Hash,$configfile );
sub GetMirrors {
	my $self = shift; 
	my ($Hash,$configfile) = @_ ;
	
    open(FH, "<$configfile");
    
    while(<FH>) {
		chomp($_);    	
		if ( $_ =~ /(\s*Internet_mirror\s*=\s*)(.*)/ ){   
			next if ( $_ =~ /^[ \t]*$/ ) ;
			$Hash->{$2} = 1 ;
		} 			
	}
	close(FH);	
	
	open(FH, "<$self->{Netpkg_dir}/last_mirror");
	my $lastmirror;
	while(<FH>) {
		chomp($_);    
		next if ( $_ =~ /^[ \t]*$/ ) ;	
		$Hash->{$_} = 1 ;
		$lastmirror = $_ ;
	}
	$Hash->{"$lastmirror"} = 0 ;
	close(FH);	
	
	return 0; 
}

####################################################################
# $Netpkg->SaveMirrors($Hash);
sub SaveMirrors {
	my $self = shift; 
	my $Hash = shift; 
	open(FH, ">$self->{Netpkg_dir}/last_mirror");
	for $_ ( sort { $Hash->{$a} cmp $Hash->{$b} } keys %{$Hash} ) { 
		print FH "$_\n";
	}
	
	close(FH);	
	
	return 0; 
}

####################################################################
# $Netpkg->SaveDB(\%db, $dbname);
sub SaveDB {
	my $self = shift; 
	my %db = %{$_[0]};
	my $dbname = $_[1];
	sysopen(DF, "$self->{Netpkg_dir}/$dbname", O_RDWR|O_CREAT, 0666) ;
	store(\%db, "$self->{Netpkg_dir}/$dbname");
	
	return 0; 
}

####################################################################
# $Netpkg->LoadDB(\$result, $dbname);
sub LoadDB {
	my $self = shift; 
	my $result = shift; 
	my $dbname = shift; 
	my %db = [];
	my $key ;
	if (-e "$self->{Netpkg_dir}/$dbname" ) {
		%db = %{ retrieve("$self->{Netpkg_dir}/$dbname") } ; 
		$$result = 0 ;
	}else{
		$$result = -1;
	}
	return \%db;
}

# Will parse the repository data to fill %db related fields
# $href = $Netpkg->BuildDB(\%db)
sub BuildDB {
	my $self = shift; 
	my %db = [];
	my $soft;
	my $desc;
	my $package;
	
	my @keys = split ( / / , $self->{Black_list}); 
	my %Blacklist;
	foreach my $key (@keys) { $Blacklist{$key} = 1 } ;

	open(FH, "$self->{Netpkg_dir}/PACKAGES.TXT") || return -1 ; 
	while (<FH>) {
		
		Gtk2->main_iteration while ( Gtk2->events_pending );
		
		chomp $_;
		# print "$_";
		if ($_ =~ /PACKAGE NAME:\s*([^\s]*)$/) { 
			$package="$1"; 
			$package =~ /^(.*)-([^-]*)-[^-]*-([^-]*).t[glx]z$/;
			$soft="$1"; 
			chomp $1;
			@{$db{$soft}}[AVAILABLE_PKG] = "$package";	
			@{$db{$soft}}[AVAILABLE_VERSION] = "$2";
			@{$db{$soft}}[AVAILABLE_BUILD] = "$3";		
			next;
		}
		if ($_ =~ /PACKAGE LOCATION:\s*\.?\/?([^\s]*)\s*$/) { 
			@{$db{$soft}}[PATH] = "$1";		
			next;
		}	
		if ($_ =~ /PACKAGE SIZE \(c.*:\s*(.*)\s*$/) { 
			@{$db{$soft}}[SIZE_COMPRESSED] = "$1";	
			next;						
		}			
		if ($_ =~ /PACKAGE SIZE \(u.*:\s*(.*)\s*$/) { 
			@{$db{$soft}}[SIZE_UNCOMPRESSED] = "$1";	
			next;							
		}														
		if ($_ =~ /PACKAGE REQUIRED:\s*([^\s]*)\s*$/) { 
			@{$db{$soft}}[DEPENDENCIES] = "$1";	
			@{$db{$soft}}[DEPENDENCIES] =~ s/,/ /g ;
			next;				
		}
		if ($_ =~ /PACKAGE REQUIRED:\s*([^\s]*)\s*$/) { 
			@{$db{$soft}}[DEPENDENCIES] = "$1";	
			@{$db{$soft}}[DEPENDENCIES] =~ s/,/ /g ;
			next;				
		}
		if ($_ =~ /PACKAGE LIBS:\s*(.*)$/) { 
			@{$db{$soft}}[LIBS] = "$1";	
			@{$db{$soft}}[LIBS] =~ s/,/ /g ;
			next;				
		}				
		# @{$db{$soft}}[DESCRIPTION] = "" if ( undef @{$db{$soft}}[DESCRIPTION] ) ;	
		while ($_ =~ /\Q${soft}\E:(.*)/) {
			$desc = $1;
			chomp $desc;
			$desc =~ s/[\ \t]{2,}/ /g ;						
			@{$db{$soft}}[DESCRIPTION] = @{$db{$soft}}[DESCRIPTION]." ".$desc;
			$_ = <SH>;
		}

		# Until we check local packages, everything is NEW
		@{$db{$soft}}[INSTALLED_VERSION] = decode('utf8',gettext("not installed"));
		@{$db{$soft}}[STATUS] = STATUS_NEW ;
		
		# SELECTED PACKAGE BOOLEAN	
		@{$db{$soft}}[SELECTED_AS_PKG] = 0; 
		# SELECTED DEP BOOLEAN	
		@{$db{$soft}}[SELECTED_AS_DEP] = 0; 
		# DELETING REVERSE DEPS
		@{$db{$soft}}[USED_BY] = "";
		
		if ( defined $Blacklist{$soft} ) {
			@{$db{$soft}}[BLACKLISTED] = 1 ;
		}
			
	}
	close(FH); 	

	return \%db;
}


####################################################################
# Will check installed packages to fill %db related fields
# $href = $Netpkg->CheckInstalledPkgs(\%db)
sub CheckInstalledPkgs {
	my $self = shift; 
	my %db = %{$_[0]};
	my $fh;
	my ($package,$soft,$version,$build,$desc,$meta);

	opendir(DH, "$self->{Package_logs}") || return -1 ;
	foreach (sort readdir(DH)) {
		
		$meta = $_;
		open(FH, "<$self->{Package_logs}/$meta") || return -1 ;		
		next if ( length ($meta) <= 3 );
		
		$meta =~ /^(.*)-([^-]*)-[^-]*-([^-]*)$/;
		$soft="$1"; 
		$version="$2";
		$build="$3";
		
		while(<FH>) {
			if ($_ =~ /^PACKAGE LOCATION:.*(\s|\/)([^\/]*t[glx]z)\s*$/) { 				
				@{ $db{$soft} }[INSTALLED_PKG] = "$2";
				last;							
			}					
		}	
		
		@{ $db{$soft} }[INSTALLED_VERSION] = "$version";	
		@{ $db{$soft} }[INSTALLED_BUILD] = "$build";			
		
		if (defined @{$db{$soft}}[AVAILABLE_PKG] ) {

			# STATUS ( -1:downgraded ; 0:installed ; 1:updated )
			@{$db{$soft}}[STATUS] = &ZW::Netpkg::PVfilter::test_package("@{$db{$soft}}[AVAILABLE_PKG]", "@{$db{$soft}}[INSTALLED_PKG]") ;					

		}else{
			@{$db{$soft}}[STATUS] = STATUS_OK;
			@{$db{$soft}}[AVAILABLE_VERSION] = "not on mirror";	
			# @{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Local"));	
			# @{$db{$soft}}[CATEGORY_ICON] = "folder";
			
			
			seek(FH, 0, 0);
			while(<FH>) {
				if ($_ =~ /^COMPRESSED PACKAGE SIZE:\s*(.*)\s*$/) { 
					@{$db{$soft}}[SIZE_COMPRESSED] = "$1";	
					next;				
				}			
				if ($_ =~ /^UNCOMPRESSED PACKAGE SIZE:\s*(.*)\s*$/) { 
					@{$db{$soft}}[SIZE_UNCOMPRESSED] = "$1";
					next;							
				}	
				if ($_ =~ /^PACKAGE LOCATION:\s*(.*)\s*$/) { 
					my $fullpath = "$1";
					my $regex = '^'.$self->{Local_repository}.'/(.*)/[^/]*.t[glx]z\s*$';
					$regex =~ s/\//\\\//g ;
					if ($fullpath =~ /$regex/ ) {
						@{$db{$soft}}[PATH] = "$1";			
					}else{
						@{$db{$soft}}[PATH] = "local";
					}
					next;							
				}	
				
				# @{$db{$soft}}[DESCRIPTION] = "" if ( undef @{$db{$soft}}[DESCRIPTION] ) ;	
				while ($_ =~ /\Q${soft}\E:(.*)/) {
					$desc = $1;
					chomp $desc;
					$desc =~ s/[\ \t]{2,}/ /g ;									
					@{$db{$soft}}[DESCRIPTION] = @{$db{$soft}}[DESCRIPTION]." ".$desc;
					$_ = <FH>;
				}
			}
		}	
		close(FH);	
	}
	closedir(DH); 
	return \%db;;
}

####################################################################
# Will set categories according to the origin of package
# $href = $Netpkg->SetCategories(\%db)
sub SetCategories {
	my $self = shift; 
	my %db = %{$_[0]};
	my $soft;
	for $soft ( keys %db ) {
		
		if ( @{$db{$soft}}[AVAILABLE_VERSION] eq "" ||  @{$db{$soft}}[INSTALLED_VERSION] eq "" ) {
			delete($db{$soft});
			next;
		}				
		
		if ( 
		@{$db{$soft}}[PATH] =~ /^l$/ || 
		@{$db{$soft}}[PATH] =~ /.*\/l$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Runtime libraries"));
			@{$db{$soft}}[CATEGORY_ICON] = "gnome-other";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^d$/ ||
		@{$db{$soft}}[PATH] =~ /.*\/d$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Development libraries"));
			@{$db{$soft}}[CATEGORY_ICON] = "development";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^x$/ ||
		@{$db{$soft}}[PATH] =~ /.*\/x$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Xorg"));
			@{$db{$soft}}[CATEGORY_ICON] = "xorg";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^ap$/ ||
		@{$db{$soft}}[PATH] =~ /.*\/ap$/
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Console Applications"));
			@{$db{$soft}}[CATEGORY_ICON] = "utilities-terminal";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^xap$/ ||
		@{$db{$soft}}[PATH] =~ /.*\/xap$/
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Graphical Applications"));
			@{$db{$soft}}[CATEGORY_ICON] = "applications-graphics";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /.*\/g$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Gnome Desktop"));
			@{$db{$soft}}[CATEGORY_ICON] = "gnome-logo";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /.*\/kde$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("KDE Desktop"));
			@{$db{$soft}}[CATEGORY_ICON] = "konqueror";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^a$/ || 
		@{$db{$soft}}[PATH] =~ /.*\/a$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("System"));
			@{$db{$soft}}[CATEGORY_ICON] = "applications-system";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^locale$/ || 
		@{$db{$soft}}[PATH] =~ /.*\/locale$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Localization"));
			@{$db{$soft}}[CATEGORY_ICON] = "preferences-desktop-locale";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^games$/ || 
		@{$db{$soft}}[PATH] =~ /.*\/games$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Games"));
			@{$db{$soft}}[CATEGORY_ICON] = "applications-games";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^patches$/ || 
		@{$db{$soft}}[PATH] =~ /.*\/patches$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Patches"));
			@{$db{$soft}}[CATEGORY_ICON] = "administration";
		}elsif ( 
		@{$db{$soft}}[PATH] =~ /^n$/ || 
		@{$db{$soft}}[PATH] =~ /.*\/n$/ ||
		@{$db{$soft}}[PATH] =~ /^s$/ || 
		@{$db{$soft}}[PATH] =~ /.*\/s$/ 
		) {
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Network"));
			@{$db{$soft}}[CATEGORY_ICON] = "applications-internet";
		}else{
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Misc"));
			@{$db{$soft}}[CATEGORY_ICON] = "applications-other";
		}
			
		@{$db{$soft}}[UID] = @{$db{$soft}}[CATEGORY].$soft.@{$db{$soft}}[AVAILABLE_VERSION].@{$db{$soft}}[AVAILABLE_BUILD].@{$db{$soft}}[INSTALLED_VERSION].@{$db{$soft}}[INSTALLED_BUILD];	
							
	}
	return \%db;
}

####################################################################
# $Netpkg->GetFilters();
# $filter = $Netpkg->GetFilters()
sub GetFilters {
	my $self = shift; 
	
    open(FH, "<$self->{Netpkg_dir}/last_filter") or return -1 ;
    binmode FH;	
	my $Filter = <FH> ;
	chomp($Filter);    
	close(FH);
	
	return $Filter;
}

####################################################################
# $Netpkg->SaveFilters($Filter);
sub SaveFilters {
	my $self = shift; 
	my $Filter = shift; 
	open(FH, ">$self->{Netpkg_dir}/last_filter");
	binmode FH;	
	print FH "$Filter";
	close(FH);	
	
	return 0; 
}

####################################################################
# $Netpkg->Configure($option, $default, $configfile);
sub Configure {
	my $self = shift; 
	
	# default value
	$self->{"$_[0]"} = $_[1];
	
    open(FH, "<$_[2]");
    while (<FH>) {	
		chomp($_);

		if ( $_ =~ /^\s*${_[0]}\s*=\s*(.*)$/ ){
			$self->{"$_[0]"} = $1;
			last;
		}
	}	
	close(FH);
	return 0; 
}

####################################################################
# $Netpkg->SaveConf($option, $newvalue, $configfile);
sub SaveConf {
	my $self = shift; 
	my $buffer;
	my $found = 0;
    open(FH, "<$_[2]");
    while (<FH>) {	
		chomp($_);

		if ( $_ =~ /^\s*${_[0]}\s*=\s*.*$/ ){
			$buffer = $buffer."$_[0] = $_[1]\n";
			$found = 1;
		}else{
			$buffer = $buffer."$_\n";
		}
	}	
	if ( $found == 0 ) {
		$buffer = $buffer."\n$_[0] = $_[1]\n";
	}
	close(FH);
	open(FH, ">$_[2]");
	print FH $buffer;
	close(FH);	
	return 0; 
}

####################################################################
# $Netpkg->Get ($Url, $Object, $Mode, $Data, $File, $ProgressBar, $Start, $Range);
sub Get {
	my $self = shift;
	my ($Url, $Object, $Mode, $Data, $File, $ProgressBar, $Start, $Range) = @_; 
	
	# Normalize URL
	$Url =~ s/\/\//\//g ;
	$Url =~ s/^[htp:]*\/*(.*)/http:\/\/$1/ ;

	# Create an LWP UserAgent
	my $Browser = LWP::UserAgent->new;

	# Setup authentication details, if necessary
	if($self->{Proxy_Socket} ne '') {
		my $ProxyUrl = "http://"."$self->{Proxy_User}".":"."$self->{Proxy_Password}"."@"."$self->{Proxy_Socket}";
		$Browser->proxy(['http', 'ftp'], $ProxyUrl);
	}
	# if ($Browser->is_protocol_supported("ftp")) {print "ftp ok\n"}
	my $result = $Browser->head($Url);
	if ( ! $result->is_success ) {
		return -1 ;
	}
	
	$ProgressBar->set_text(decode('utf8',gettext("Loading"))." ".$Object);
	my $remote_headers = $result->headers;
	my $total_size = $remote_headers->content_length;
	my $KBtotal_size = int($total_size / 1024) ;
	my $iterate = 0 ;
	if($Mode) {
		$Browser->timeout(20);
		open(FH, ">$File") or die "Cannot open file $File : $!\n";
		binmode FH;	
		my $size;	
		my $res = $Browser->request(HTTP::Request->new(GET => $Url), sub {
				my ($chunk, $res) = @_;
				print FH $chunk;
				$size += length($chunk);
				$iterate++ ;
				if ( $iterate % 20 == 0 ){
					# print "$iterate ";
					my $Progress = $Start + $Range * $size / $total_size ;
					my $percent = int($size / $total_size * 100) ;
					if ($Progress >= 1) { $Progress = 1 }
					$ProgressBar->set_fraction($Progress);
					$ProgressBar->set_text(decode('utf8',gettext("Loading"))." ".$Object." (".$KBtotal_size." kB) ".$percent." %");
					Gtk2->main_iteration while ( Gtk2->events_pending );  
				}			
			});		
		close FH;
	}else{
		$Browser->timeout(20);
		my $res = $Browser->request(HTTP::Request->new(GET => $Url), sub {
				my($chunk, $res) = @_;
				${$Data} .= $chunk;
				$iterate++ ;
				if ( $iterate % 10 == 0 ){	
					# print "$iterate ";			
					my $len= length(${$Data});
					my $Progress = $Start + $Range * ($len / $total_size) ;
					my $percent = int(length(${$Data}) / $total_size * 100) ;
					if ($Progress >= 1) { $Progress = 1 }
					$ProgressBar->set_fraction($Progress);
					$ProgressBar->set_text(decode('utf8',gettext("Loading"))." ".$Object." (".$KBtotal_size." kB) ".$percent." %");
					Gtk2->main_iteration while ( Gtk2->events_pending );  	
				}		
			});
	}
	
	return 0;	
}	


# the shared semaphor is set to '0' when system processing ends
####################################################################
# $Netpkg->Execute($command)
my $thr = threads->new(\&Execute);
$thr->detach;

sub Execute {
	while (1) {
		threads->yield();
		sleep 1;
		$SIG{'INT'} = sub { threads->exit(0); };
		$SIG{'TERM'} = sub { threads->exit(0); };
		$SIG{'KILL'} = sub { threads->exit(0); };
		$SIG{'QUIT'} = sub { threads->exit(0); };
		if ($semaphorus == 1) {
			threads->yield();
			`$command`;
			lock $semaphorus;
			$semaphorus = 0;	
			lock $command;
			$command = "sleep";
			threads->yield();
		}
	}
}

####################################################################
# $Netpkg->Install( $File );
sub Install {
	my $self = shift; 
	my $package = shift;
	
    #print "/sbin/installpkg $package\n";
       
	$command = "/sbin/installpkg $package >/dev/null" ; 
	$semaphorus = 1;
	until ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	} 

	my $date = strftime('%a %b %d %H:%M:%S %Y',localtime); 
	open(FH, ">>$self->{Logfile}");
	print FH "[I] $package $date\n";
	close(FH);
	return 0;
}

####################################################################
# $Netpkg->Update( $File );
sub Update {
	my $self = shift; 
	my $package = shift;
	
    #print "/sbin/upgradepkg --reinstall $package\n";

	$command = "/sbin/upgradepkg $package > /dev/null";
	$semaphorus = 1;
	until ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	}
	my $date = strftime('%a %b %d %H:%M:%S %Y',localtime); 
	open(FH, ">>$self->{Logfile}");
	print FH "[U] $package $date\n";
	close(FH);
	return 0;
}

####################################################################
# $Netpkg->Reinstall( $File );
sub Reinstall {
	my $self = shift; 
	my $package = shift;
	
    #print "/sbin/upgradepkg --reinstall $package\n";
    
	$command = "/sbin/upgradepkg --reinstall $package >/dev/null";
	$semaphorus = 1;
	until ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	}
	my $date = strftime('%a %b %d %H:%M:%S %Y',localtime); 
	open(FH, ">>$self->{Logfile}");
	print FH "[I] $package $date\n";
	close(FH);
	return 0;
}

####################################################################
# $Netpkg->Remove( $soft );
sub Remove {
	my $self = shift; 
	my $package = shift;
    #print "/sbin/removepkg $package\n";
    
	$command = "/sbin/removepkg $package >/dev/null";
	$semaphorus = 1;
	until ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	}
	my $date = strftime('%a %b %d %H:%M:%S %Y',localtime); 
	open(FH, ">>$self->{Logfile}");
	print FH "[R] $package $date\n";
	close(FH);
	return 0;
}

####################################################################
# $Netpkg->AsyncRun( $command );
sub AsyncRun {
	my $self = shift; 
	my $cmd = shift;
	    
	$command = "$cmd";
	$semaphorus = 1;
	until ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	}
	return 0;
}


####################################################################
# $Netpkg->ProcessDeps(\%db);
sub ProcessDeps {
	my $self = shift; 
	my %db = %{$_[0]};
	my $soft; 
	my $dep; 
	for $soft ( keys %db ) {
		if ( defined $db{$soft} ) {	
			@{$db{$soft}}[USED_BY] = "";
			@{$db{$soft}}[INSTALLED_DEPENDENCIES] = "";
			@{$db{$soft}}[MISSING_DEPENDENCIES] = "";			
		}
	}
	for $soft ( keys %db ) {
		if ( defined $db{$soft} ) {
			my @Deps = split (/ /, @{$db{$soft}}[DEPENDENCIES]);	
			foreach $dep (@Deps) {
				if ( defined $db{$dep} ) {			
					if ( @{$db{$soft}}[STATUS] != STATUS_NEW ) {
						@{$db{$dep}}[USED_BY] .= " $soft" ; 
					}
					if ( @{$db{$dep}}[STATUS] == STATUS_OK ) {
						@{$db{$soft}}[INSTALLED_DEPENDENCIES] .= " $dep" ;
					}else{
						@{$db{$soft}}[MISSING_DEPENDENCIES] .= " $dep" ;
					}
				}
			}
		}

		@{$db{$dep}}[USED_BY] =~ s/ {2,}/ /g ;
		@{$db{$soft}}[INSTALLED_DEPENDENCIES] =~ s/ {2,}/ /g ;
		@{$db{$soft}}[MISSING_DEPENDENCIES] =~ s/ {2,}/ /g ;
	}
}



####################################################################
# $Netpkg->SelectMissingDeps(\%db);
sub SelectMissingDeps {
	my $self = shift; 
	my %db = %{$_[0]};
	my $soft; 
	for $soft ( keys %db ) { 
		@{$db{$soft}}[SELECTED_AS_DEP] = 0 ;
	}
	for $soft ( keys %db ) {
		if ( @{$db{$soft}}[SELECTED_AS_PKG] ) {

			# array of missing deps
			my @Deps = split (/ /, @{$db{$soft}}[MISSING_DEPENDENCIES]);
			
			my $dep; 
			foreach $dep (@Deps) {

				@{$db{$dep}}[SELECTED_AS_DEP] = 1 ; 
				
				# one more level
				my @Deps2 = split (/ /, @{$db{$dep}}[MISSING_DEPENDENCIES]);					
				my $dep2; 
				foreach $dep2 (@Deps2) {
					@{$db{$dep2}}[SELECTED_AS_DEP] = 1 ; 					
				}
			}
		}
	}
}

####################################################################
# $Netpkg->SelectDeps(\%db);
sub SelectDeps {
	my $self = shift; 
	my %db = %{$_[0]};
	my $soft; 
	for $soft ( keys %db ) { 
		@{$db{$soft}}[SELECTED_AS_DEP] = 0 ;
	}
	for $soft ( keys %db ) {
		if ( @{$db{$soft}}[SELECTED_AS_PKG] ) {
				
			# array of missing deps
			my @Deps = split (/ /, @{$db{$soft}}[DEPENDENCIES]);
			
			my $dep; 
			foreach $dep (@Deps) {
				@{$db{$dep}}[SELECTED_AS_DEP] = 1 ; 
			}
		}
	}
}



1;

