/*
 *  Print Dialog for Canon LIPS/PS/LIPSLX/UFR2/CAPT Printer.
 *  Copyright (C) 2004-2008 CANON INC.
 *  All Rights Reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "uimain.h"
#include "mltlang.h"

#include "widgets.h"
#include "callbacks.h"
#include "support.h"
#include "holdqueuedlg.h"

const gchar *g_page_selection_button_name[] = {
	"AllPage_radiobutton",
	"OddPages_radiobutton",
	"EvenPages_radiobutton",
	"SelPages_radiobutton",
	NULL
};

const gchar *pageset_options[] = {
	"all",
	"odd",
	"even",
	"range",
	NULL,
};

const gchar *g_orientation_button_name[] = {
	"Portrait_radiobutton",
	"Landscape_radiobutton",
	"ReverseLandscape_radiobutton",
	"ReversePortrait_radiobutton",
	NULL
};

const gchar *g_img_position_button_name[] = {
	"ImgPositionTL_radiobutton",
	"ImgPositionCL_radiobutton",
	"ImgPositionBL_radiobutton",
	"ImgPositionTC_radiobutton",
	"ImgPositionCC_radiobutton",
	"ImgPositionBC_radiobutton",
	"ImgPositionTR_radiobutton",
	"ImgPositionCR_radiobutton",
	"ImgPositionBR_radiobutton",
	NULL
};

const char *position_options[] = {
	"top-left",
	"left",
	"bottom-left",
	"top",
	"center",
	"bottom",
	"top-right",
	"right",
	"bottom-right",
	NULL
};

const gchar *g_filter_frame_name[] = {
	NULL,
	"FilterImage_frame",
	"FilterText_frame",
	"FilterHPGL2_frame",
	NULL
};

const gchar *g_img_reso_scale_button_name[] = {
	"Resolution_radiobutton",
	"Scale_radiobutton",
	"PageScale_radiobutton",
	NULL
};

static int g_curr_signal;

void SigInit(void)
{
	g_curr_signal = 0;
}

gboolean SigEnable(void)
{
	gboolean enable = TRUE;
	if(g_curr_signal > 0){
		g_curr_signal--;
		enable = FALSE;
	}
	return enable;
}

gboolean SigDisable(void)
{
	gboolean enable = TRUE;
	if(g_curr_signal)
		enable = FALSE;
	g_curr_signal++;
	return enable;
}

GtkWidget* getWidget(GtkWidget *window, char *name)
{
	return lookup_widget(window, name);
}

gchar* GetCurrComboText(GtkWidget *window, gchar *combo_entry_name)
{
	GtkWidget *entry;
	char *text;
	entry = getWidget(window, combo_entry_name);

	text = (gchar*)gtk_entry_get_text(GTK_ENTRY(entry));
	return text;
}

gchar* GetTextEntry(GtkWidget *window, char *entry_name)
{
	GtkWidget *entry;

	entry = getWidget(window, entry_name);
	if(entry == NULL)
		return NULL;
	return (gchar*)gtk_entry_get_text(GTK_ENTRY(entry));
}

gfloat GetSpinButtonValueFloat(GtkWidget *window, gchar *spin_button_name)
{
	GtkWidget *spin_button;

	spin_button = getWidget(window, spin_button_name);

	return gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(spin_button));
}

gint GetSpinButtonValue(GtkWidget *window, gchar *spin_button_name, int type)
{
	GtkWidget *spin_button;

	spin_button = getWidget(window, spin_button_name);

	if(type)
		return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_button));
	else
		return (gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(spin_button)) * 1000);
}

gboolean GetToggleButtonActive(GtkWidget *window, gchar *button_name)
{
	GtkWidget *button;

	button = getWidget(window, button_name);

	return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
}

void SetEntryVisibility(GtkWidget *window, gchar *entry_name)
{
	GtkWidget *entry;

	entry = getWidget(window, entry_name);

	gtk_entry_set_visibility(GTK_ENTRY(entry), FALSE);
}

void SetCursolPosition(GtkWidget *window, gchar *entry_name, gint position)
{
	GtkWidget *entry;
	entry = getWidget(window, entry_name);
	if(position == -1){
		gint num = GTK_ENTRY(entry)->text_length;
		gtk_editable_set_position(GTK_EDITABLE(entry), num);
	}else{
		gtk_editable_delete_text(GTK_EDITABLE(entry), position, -1);
		gtk_editable_set_position(GTK_EDITABLE(entry), position);
	}
}

void SetDialogTitle(GtkWidget *window, gchar *title)
{
	if(title != NULL)
		gtk_window_set_title(GTK_WINDOW(window), title);
}

void SetGListToCombo(GtkWidget *window, GList *glist, gchar *combo_name, gchar *curr_name)
{
	GtkWidget *combo, *entry;

	combo = getWidget(window, combo_name);
	entry = GTK_COMBO(combo)->entry;

	if(glist != NULL || curr_name != NULL){
		gtk_combo_set_popdown_strings(GTK_COMBO(combo), glist);
		gtk_entry_set_text(GTK_ENTRY(entry), curr_name);
	}
}

void SetActiveRadioButton(GtkWidget *window, const gchar *button_name[], int set_index)
{
	GtkWidget *button;

	button = getWidget(window, (char *)button_name[set_index]);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
}

void SetWidgetSensitive(GtkWidget *window, gchar *widgetname, gboolean value)
{
	GtkWidget *widget;
	widget = getWidget(window, widgetname);
	gtk_widget_set_sensitive(widget, value);
}

void SetSpinButtonFloat(GtkWidget *window, gchar *spin_name, gfloat value)
{
	GtkWidget *spin;
	spin = getWidget(window, spin_name);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), value);
}

void SetSpinDigits(GtkWidget *window, gchar *spin_name, gint value)
{
	GtkWidget *spin;
	spin = getWidget(window, spin_name);
	gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), value);
}

void SetSpinMaxValueFloat(GtkWidget *window, gchar *spin_name, gfloat max_value)
{
	GtkWidget *spin;
	GtkObject *adj;

	spin = getWidget(window, spin_name);
	adj = GTK_OBJECT(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin)));

	GTK_ADJUSTMENT(adj)->upper = max_value;

	gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(spin), GTK_ADJUSTMENT(adj));
}

void SetSpinButton(GtkWidget *window, gchar *spin_name, gint value)
{
	GtkWidget *spin;
	spin = getWidget(window, spin_name);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), value);
}

void SetSpinMaxValue(GtkWidget *window, gchar *spin_name, gint max_value)
{
	GtkWidget *spin;
	GtkObject *adj;

	spin = getWidget(window, spin_name);
	adj = GTK_OBJECT(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin)));

	GTK_ADJUSTMENT(adj)->upper = max_value;

	gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(spin), GTK_ADJUSTMENT(adj));
}
void SetSpinMaxMinValue(GtkWidget *window, gchar *spin_name, gint max_value, gint min_value)
{
	GtkWidget *spin;
	GtkObject *adj;

	spin = getWidget(window, spin_name);
	adj = GTK_OBJECT(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin)));

	GTK_ADJUSTMENT(adj)->upper = max_value;
	GTK_ADJUSTMENT(adj)->lower = min_value;

	gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(spin), GTK_ADJUSTMENT(adj));
}

void SetSpinMaxMinValueDouble(GtkWidget *window, gchar *spin_name, double max_value_d, double min_value_d)
{
	GtkWidget *spin;
	GtkObject *adj;

	spin = getWidget(window, spin_name);
	adj = GTK_OBJECT(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin)));

	GTK_ADJUSTMENT(adj)->upper = max_value_d;
	GTK_ADJUSTMENT(adj)->lower = min_value_d;

	gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(spin), GTK_ADJUSTMENT(adj));
}

void SetSpinStepIncrementFloat(GtkWidget *window, gchar *spin_name, gfloat increment)
{
	GtkWidget *spin;
	GtkObject *adj;

	spin = getWidget(window, spin_name);
	adj = GTK_OBJECT(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin)));

	GTK_ADJUSTMENT(adj)->step_increment = increment;

	gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(spin), GTK_ADJUSTMENT(adj));
}

void SetActiveCheckButton(GtkWidget *window, gchar *button_name, gboolean on)
{
	GtkWidget *button;

	button = getWidget(window, button_name);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), on);
}

void SetTextEntry(GtkWidget *window, gchar *entry_name, gchar *text)
{
	GtkWidget *entry;

	entry = getWidget(window, entry_name);

	gtk_entry_set_text(GTK_ENTRY(entry), text);
}

void SetTextMaxLength(GtkWidget *window, gchar *entry_name, gint max_num)
{
	GtkWidget *entry;

	entry = getWidget(window, entry_name);

	gtk_entry_set_max_length(GTK_ENTRY(entry), (guint)max_num);
}

void SetHScaleValue(GtkWidget *window, gchar *hscale_name, gint value)
{
	GtkWidget *hscale;
	GtkObject *hscale_adj;

	hscale = getWidget(window, hscale_name);
	hscale_adj = (GtkObject *)gtk_range_get_adjustment(GTK_RANGE(hscale));

	gtk_adjustment_set_value(GTK_ADJUSTMENT(hscale_adj), value);
}

void SetValueToLabel(GtkWidget *window, gchar *label_name, gint value)
{
	GtkWidget *label;
	gchar text[32];

	label = getWidget(window, label_name);
	snprintf(text, 31, "%d", value);
	gtk_label_set_text(GTK_LABEL(label), text);
}

void SetTextToLabel(GtkWidget *window, gchar *label_name, gchar *text)
{
	GtkWidget *label;

	label = getWidget(window, label_name);
	gtk_label_set_text(GTK_LABEL(label), text);
}

void SetButtonLabel(GtkWidget *window, gchar *button_name, gchar *text)
{
	GtkWidget *button;
	button = getWidget(window, button_name);
#ifdef OLD_GTK
	gtk_label_set_text(GTK_LABEL(GTK_BUTTON(button)->child), text);
#else
	gtk_button_set_label(GTK_BUTTON(button), text);
#endif
}

void SetFocus(GtkWidget *window, char *widget_name)
{
	GtkWidget *widget;
	widget = getWidget(window, widget_name);
	gtk_widget_grab_focus(widget);
	gtk_widget_grab_default(widget);
}

void SetGrabDefault(GtkWidget *window, char *widget_name)
{
	GtkWidget *widget;
	widget = getWidget(window, widget_name);
	gtk_widget_grab_default(widget);
}

void ClearTextWidget(GtkWidget *window, gchar *text_name)
{
	GtkWidget *text;
	text = getWidget(window, text_name);
	gtk_editable_delete_text(GTK_EDITABLE(text), 0, -1);
}

void ShowWidget(GtkWidget *window, gchar *widget_name)
{
	GtkWidget *widget;

	if(widget_name == NULL)
		return;
	widget = getWidget(window, widget_name);
	gtk_widget_show(widget);
}

void HideWidget(GtkWidget *window, gchar *widget_name)
{
	GtkWidget *widget;

	widget = getWidget(window, widget_name);
	if(widget != NULL)
		gtk_widget_hide(widget);
}

void SetShowPage(GtkWidget *window, int page_index, gboolean show)
{
	GtkWidget *child, *widget;

	widget = getWidget(window, "PropDlg_notebook");
	if(widget != NULL){
		child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(widget), page_index);
		if(child != NULL){
			if(show)
				gtk_widget_show(child);
			else
				gtk_widget_hide(child);
		}
	}
}

void ChangeShowPage(GtkWidget *window, int page_index)
{
	GtkWidget *widget;

	widget = getWidget(window, "PropDlg_notebook");
	if(widget != NULL)
		gtk_notebook_set_page(GTK_NOTEBOOK(widget), page_index);
}

gboolean ExceptWidget(gchar *except_name, gchar *comp_name)
{
	if(except_name != NULL && comp_name != NULL)
		return strcmp(except_name, comp_name) ? FALSE : TRUE;
	else
		return FALSE;
}

void ComboSignalConnect(GtkWidget *window, gchar *combo_name, GtkSignalFunc sig_func)
{
	GtkWidget *combo;

	combo = getWidget(window, combo_name);

	gtk_widget_realize(combo);
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->popwin), "event", GTK_SIGNAL_FUNC(sig_func), NULL);
}

void HscaleSignalConnect(GtkWidget *window, gchar *hscale_name, GtkSignalFunc sig_func)
{
	GtkWidget *hscale;
	GtkObject *hscale_adj;

	hscale = getWidget(window, hscale_name);
	hscale_adj = (GtkObject *)gtk_range_get_adjustment(GTK_RANGE(hscale));

	gtk_range_set_adjustment(GTK_RANGE(hscale), GTK_ADJUSTMENT(hscale_adj));

	gtk_signal_connect(GTK_OBJECT(hscale_adj), "value_changed", GTK_SIGNAL_FUNC(sig_func), hscale_adj);
}

void ConvPointToMM_Inch(int point, char *text, int locale)
{
	if(locale == 0){
		float inch;
		inch = (float)(point * (float)(0.014));
		snprintf(text, 127, "[point]    %.3f[inch]", inch);
	}else{
		float mm;
		mm = (float)(point * (float)(0.35));
		snprintf(text, 127, "[point]    %.2f[mm]", mm);
	}
}

int GetActive(int id, char *list)
{
	char *active;
	int ret = 0;
	active = GetCurrOpt(id, list);
	if(active != NULL){
		if(strcasecmp(active, "False") == 0
		|| strcasecmp(active, "None") == 0)
			ret = 0;
		else if(strcasecmp(active, "True") == 0)
			ret = 1;
		memFree(active);
	}
	return ret;
}

char *GetCurrOptName(int id)
{
	char *str, *text;

	str = GetCurrOpt(id, NULL);

	if(str == NULL)
		return NULL;

	text = NameToText(id, str);
	if(text == NULL)
		return str;
	else{
		memFree(str);
		return text;
	}
}

int GetValue(int id, char *value, char *name)
{
	char tmp[256], *ptmp, *pvalue;
	int disable = 0;

	pvalue = value;
	ptmp = tmp;
	while(1){
		if(*pvalue == '\0'){
			*ptmp = '\0';
			strcpy(name, tmp);
			break;
		}
		if(*pvalue == '<'){
			*ptmp = '\0';
			strcpy(name, tmp);
			pvalue++;
			ptmp = tmp;
		}
		if(*pvalue == '>'){
			*ptmp = '\0';
			disable = atoi(tmp);
			break;
		}
		*ptmp = *pvalue;
		ptmp++;
		pvalue++;
	}

	return disable;
}


void GetOptToGList(GtkWidget *window, int id, char *widget_name)
{
	GList *glist = NULL;
	char *list, *plist, value[256], *pvalue, *curr = NULL;
	char *text = NULL;
	char tmp[256];
	char *str;
	char *cur;

	list = GetOptionList(id);
	if(list == NULL)
		return;
	plist = list;
	pvalue = value;

	while(1){
		if(*plist == '\0'){
			int disable;
			*pvalue = '\0';
			memset(tmp, 0, 256);
			if((disable = GetValue(id, value, tmp)) == 0){
				text = NameToText(id, tmp);
				if(text != NULL)
					glist = g_list_append(glist, text);
				else{
					str = strdup(tmp);
					glist = g_list_append(glist, str);
				}
			}
			break;
		}
		if(*plist == ','){
			int disable;
			*pvalue = '\0';
			memset(tmp, 0, 256);
			if((disable = GetValue(id, value, tmp)) == 0){
				text = NameToText(id, tmp);
				if(text != NULL)
					glist = g_list_append(glist, text);
				else{
					str = strdup(tmp);
					glist = g_list_append(glist, str);
				}
			}
			plist++;
			pvalue = value;
		}
		if(*plist == ':'){
			*pvalue = '\0';
			cur = NameToText(id, value);
			if(cur != NULL)
				curr = strdup(cur);
			else
				curr = strdup(value);

			plist++;
			pvalue = value;
		}
		*pvalue = *plist;
		pvalue++;
		plist++;
	}
	SetGListToCombo(window, glist, widget_name, curr);
	if(glist == NULL)
		SetWidgetSensitive(window, widget_name, FALSE);
	else
		g_list_free(glist);
	memFree(curr);
	memFree(list);
	return;
}

void UpdateCpcaComboWidget(UIPropDlg *prop, int id, char *combo_name)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	GetOptToGList(window, id, combo_name);
}

int SetCpcaWidgetSensitive(UIPropDlg *prop, int id, char *widget_name)
{
	char *list;
	int disable;
	GtkWidget *window;

	if(strcmp(widget_name, "CopyNumbering_checkbutton") == 0
	 || strcmp(widget_name, "TrustPrint_checkbutton") == 0){
		if(prop->captfin == NULL)
			return 0;
		window = UI_DIALOG(prop->captfin)->window;
	}else if(strcmp(widget_name, "ShiftPosition_checkbutton") == 0){
		window = UI_DIALOG(prop->advsettings)->window;
	}else if(strcmp(widget_name, "CreepUse_checkbutton") == 0){
		window = UI_DIALOG(prop->booklet)->window;
	}else
		window = UI_DIALOG(prop)->window;
	list = GetOptionList(id);
	if(list == NULL)
		return 0;

	disable = GetCurrDisable(id, list);

	if(disable != -1){
		int active;
		disable = disable > 0 ? 0 : 1;
		active = GetActive(id, list);
		if(active > -1)
			SetActiveCheckButton(window, widget_name, active);
		SetWidgetSensitive(window, widget_name, disable);
	}
	memFree(list);
	return disable;
}



void UpdateMainDlgWidgets(UIMainDlg *main_wnd, int id)
{
	GtkWidget *window;
	int num = id;

	window = UI_DIALOG(main_wnd)->window;

	if(num == 0){
		GetOptToGList(window, ID_PRINTERNAME, "PrinterName_combo");
		num = ID_PRINTERNAME;
	}

	if(num == ID_PRINTERNAME){
		gboolean enable;
		enable = main_wnd->prop->ui_type == PRINTER_TYPE_OTHER ? FALSE : TRUE;
		SetWidgetSensitive(window, "Property_button", enable);
	}

	if(num == ID_PAGE_SET || num == ID_PRINTERNAME){
		char *cur;
		int sel = 0;
		int i;
		cur = GetCurrOpt(ID_PAGE_SET, NULL);
		for(i = 0; pageset_options[i] != NULL; i++){
			if(cur != NULL && strcmp(pageset_options[i], cur) == 0)
				sel = i;
		}
		memFree(cur);
		if(num != ID_PAGE_SET)
			SetActiveRadioButton(window, g_page_selection_button_name, sel);
		if(sel != PAGESEL_RANGE){
			SetWidgetSensitive(window, "PageSelection_entry", FALSE);
			ClearTextWidget(window, "PageSelection_entry");
		}else{
			char *range;
			SetWidgetSensitive(window, "PageSelection_entry", TRUE);
			range = GetCurrOpt(ID_PAGE_RANGES, NULL);
			SetTextEntry(window, "PageSelection_entry", range);
			memFree(range);
		}
	}

	if(num == ID_PRINTERNAME){
		gint copies;
		char *text, label[32];
		int max_copies;

		max_copies = GetCurrOptInt(ID_MAX_COPIES, 0);
		if(max_copies != 0){
			text = NameToText(ID_MAX_COPIES, "CopiesNum");
			snprintf(label, 31, text, max_copies);
			SetSpinMaxValue(window, "Copies_spinbutton", max_copies);
		}else{
			switch(main_wnd->prop->ui_type){
			case PRINTER_TYPE_OTHER:
				max_copies = 1;
				break;
			case PRINTER_TYPE_CAPT:
			case PRINTER_TYPE_CAPT_BIND:
				max_copies = 999;
				break;
			default:
				max_copies = 2000;
				break;
			}
			SetSpinMaxValue(window, "Copies_spinbutton", max_copies);
		}

		copies = GetCurrOptInt(ID_CNCOPIES, 1);
		SetSpinButton(window, "Copies_spinbutton", copies);
	}

	if(num == ID_PRINTERNAME){
		gboolean on = 0;
		char *cur;
		cur = GetCurrOpt(ID_OUTPUTORDER, NULL);
		if(cur != NULL){
			on = (strcmp(cur, "reverse") == 0) ? 1 : 0;
			memFree(cur);
		}
		SetActiveCheckButton(window, "ReverseOrder_checkbutton", on);
	}

	if(num == ID_PRINTERNAME){
		gboolean enable;
		enable = main_wnd->prop->ui_type == PRINTER_TYPE_OTHER ? FALSE : TRUE;
		enable = (main_wnd->hasfile != 1) ? FALSE : enable;
		SetWidgetSensitive(window, "MainOK_button", enable);
		if(enable == TRUE)
			SetGrabDefault(window, "MainOK_button");
	}
	if(num == ID_PRINTERNAME){
		gboolean enable;
		gchar *text;
		enable = main_wnd->prop->ui_type == PRINTER_TYPE_OTHER ? FALSE : TRUE;
		SetWidgetSensitive(window, "MainSave_button", enable);
		if(main_wnd->hasfile == 0){
			text = NameToText(ID_PRINTERNAME, "Save");
			SetGrabDefault(window, "MainSave_button");
		}else{
			text = NameToText(ID_PRINTERNAME, "PrintSave");
		}
		SetButtonLabel(window, "MainSave_button", text);
		if(enable == FALSE)
			SetGrabDefault(window, "MainCancel_button");
	}
}


void UpdatePropGeneralWidgets(UIPropDlg *prop, int id)
{
	GtkWidget *window;
	int num = id;

	window = UI_DIALOG(prop)->window;

	if(num == 0){
		GetOptToGList(window, ID_PAGESIZE, "PaperSize_combo");
		num = ID_PAGESIZE;
	}

	if(num == ID_PAGESIZE){
		int sel;
		sel = GetCurrOptInt(ID_ORIENTATION_REQUESTED, 3) - 3;
		SetActiveRadioButton(window, g_orientation_button_name, sel);
	}

	if(num == ID_PAGESIZE){
		GetOptToGList(window, ID_NUMBER_UP, "Nup_combo");
	}

	if(num == ID_PAGESIZE || num == ID_BRIGHTNESS){
		int value;
		value = GetCurrOptInt(ID_BRIGHTNESS, 100);
		if(num != ID_BRIGHTNESS)
			SetHScaleValue(window, "BrightnessValue_hscale", value);
		SetValueToLabel(window, "BrightnessValue_label", value);
	}

	if(num == ID_PAGESIZE || num == ID_GAMMA){
		int value;
		value = GetCurrOptInt(ID_GAMMA, 1000);
		if(num != ID_GAMMA)
			SetHScaleValue(window, "GammaValue_hscale", value);
		SetValueToLabel(window, "GammaValue_label", value);
	}

	if(num == ID_PAGESIZE){
		GetOptToGList(window, ID_JOB_SHEETS_START, "BannerStart_combo");
		GetOptToGList(window, ID_JOB_SHEETS_END, "BannerEnd_combo");
	}


}


void UpdatePropImageWidgets(UIPropDlg *prop, int id)
{
	GtkWidget *window;

	if(id == -1)
		return;

	window = UI_DIALOG(prop)->window;

	if(id == ID_FILTER){
		if(prop->color_type)
			ShowWidget(window, "ImgColorSetting_frame");
		if(!prop->color_type)
			HideWidget(window, "ImgColorSetting_frame");
	}
	if(prop->color_type && (id == ID_FILTER || id == ID_HUE)){
		int value;
		value = GetCurrOptInt(ID_HUE, 0);
		if(id != ID_HUE)
			SetHScaleValue(window, "Hue_hscale", value);
		SetValueToLabel(window, "HueValue_label", value);
	}

	if(prop->color_type && (id == ID_FILTER || id == ID_SATURATION)){
		int value;
		value = GetCurrOptInt(ID_SATURATION, 100);
		if(id != ID_SATURATION)
			SetHScaleValue(window, "Saturation_hscale", value);
		SetValueToLabel(window, "SaturationValue_label", value);
	}
	HideWidget(window, "ColorDefault_button");

	if(id == ID_FILTER || id == ID_RESO_SCALE){
		int sel;
		sel = GetCurrOptInt(ID_RESO_SCALE, 0);
		SetActiveRadioButton(window, g_img_reso_scale_button_name, sel);
		if(sel == 0){
			SetWidgetSensitive(window, "ImgReso_hscale", TRUE);
			SetWidgetSensitive(window, "ImgPercent_hscale", FALSE);
			SetWidgetSensitive(window, "ImgPagePercent_hscale", FALSE);
			SetWidgetSensitive(window, "ResolutionValue_label", TRUE);
			SetWidgetSensitive(window, "PageScaleValue_label", FALSE);
			SetWidgetSensitive(window, "ScaleValue_label", FALSE);
		}else if(sel == 1){
			SetWidgetSensitive(window, "ImgPercent_hscale", TRUE);
			SetWidgetSensitive(window, "ImgPagePercent_hscale", FALSE);
			SetWidgetSensitive(window, "ImgReso_hscale", FALSE);
			SetWidgetSensitive(window, "ResolutionValue_label", FALSE);
			SetWidgetSensitive(window, "PageScaleValue_label", FALSE);
			SetWidgetSensitive(window, "ScaleValue_label", TRUE);
		}else if(sel == 2){
			SetWidgetSensitive(window, "ImgPercent_hscale", FALSE);
			SetWidgetSensitive(window, "ImgPagePercent_hscale", TRUE);
			SetWidgetSensitive(window, "ImgReso_hscale", FALSE);
			SetWidgetSensitive(window, "ResolutionValue_label", FALSE);
			SetWidgetSensitive(window, "PageScaleValue_label", TRUE);
			SetWidgetSensitive(window, "ScaleValue_label", FALSE);
		}
	}

	if(id == ID_FILTER || id == ID_PPI){
		int ppi;
		ppi = GetCurrOptInt(ID_PPI, 128);
		if(id != ID_PPI)
			SetHScaleValue(window, "ImgReso_hscale", ppi);
		SetValueToLabel(window, "ResolutionValue_label", ppi);
	}
	if(id == ID_FILTER || id == ID_SCALING){
		int scaling;
		scaling = GetCurrOptInt(ID_SCALING, 100);
		if(id != ID_SCALING)
			SetHScaleValue(window, "ImgPercent_hscale", scaling);
		SetValueToLabel(window, "ScaleValue_label", scaling);
	}
	if(id == ID_FILTER || id == ID_NATURAL_SCALING){
		int natural_scaling;
		natural_scaling = GetCurrOptInt(ID_NATURAL_SCALING, 100);
		if(id != ID_NATURAL_SCALING){
			SetHScaleValue(window, "ImgPagePercent_hscale", natural_scaling);
		}
		SetValueToLabel(window, "PageScaleValue_label", natural_scaling);
	}

	if(id == ID_FILTER){
		gint sel = 4;
		int i;
		char *cur;
		cur = GetCurrOpt(ID_POSITION, NULL);
		for(i = 0; position_options[i]; i++){
			if(cur != NULL && strcmp(cur, position_options[i]) == 0)
				sel = i;
		}
		memFree(cur);
		SetActiveRadioButton(window, g_img_position_button_name, sel);
	}
}


void UpdatePropTextWidgets(UIPropDlg *prop, int id)
{
	GtkWidget *window;

	if(id == -1)
		return;

	window = UI_DIALOG(prop)->window;

	if(id == ID_FILTER){
		gint cpi;
		cpi = GetCurrOptInt(ID_CPI, 10);
		SetSpinButton(window, "TextCPI_spinbutton", cpi);
	}
	if(id == ID_FILTER){
		gint lpi;
		lpi = GetCurrOptInt(ID_LPI, 6);
		SetSpinButton(window, "TextLPI_spinbutton", lpi);
	}
	if(id == ID_FILTER){
		gint columns;
		columns = GetCurrOptInt(ID_COLUMNS, 1);
		SetSpinButton(window, "TextColumns_spinbutton", columns);
	}
	if(id == ID_FILTER || id == ID_MARGIN){
		gboolean on;
		on = GetCurrOptInt(ID_MARGIN, 0);
		if(id == ID_FILTER){
			SetActiveCheckButton(window, "CustomMargin_checkbutton", on);
		}
		SetWidgetSensitive(window, "Margin_table", on);
	}
	if(id == ID_FILTER || id == ID_PAGE_LEFT){
		gint point;
		gchar text[128];
		point = GetCurrOptInt(ID_PAGE_LEFT, 0);
		if(id != ID_PAGE_LEFT)
			SetSpinButton(window, "MarginLeft_spinbutton", point);
		ConvPointToMM_Inch(point, text, !IsUS());
		SetTextToLabel(window, "MarginLeftValue_label", text);
	}
	if(id == ID_FILTER || id == ID_PAGE_RIGHT){
		gint point;
		gchar text[128];
		point = GetCurrOptInt(ID_PAGE_RIGHT, 0);
		if(id != ID_PAGE_RIGHT)
			SetSpinButton(window, "MarginRight_spinbutton", point);
		ConvPointToMM_Inch(point, text, !IsUS());
		SetTextToLabel(window, "MarginRightValue_label", text);
	}

	if(id == ID_FILTER || id == ID_PAGE_TOP){
		gint point;
		gchar text[128];
			point = GetCurrOptInt(ID_PAGE_TOP, 0);
		if(id != ID_PAGE_TOP)
			SetSpinButton(window, "MarginTop_spinbutton", point);
		ConvPointToMM_Inch(point, text, !IsUS());
		SetTextToLabel(window, "MarginTopValue_label", text);
	}

	if(id == ID_FILTER || id == ID_PAGE_BOTTOM){
		gint point;
		gchar text[128];
		point = GetCurrOptInt(ID_PAGE_BOTTOM, 0);
		if(id != ID_PAGE_BOTTOM)
			SetSpinButton(window, "MarginBottom_spinbutton", point);
		ConvPointToMM_Inch(point, text, !IsUS());
		SetTextToLabel(window, "MarginBottomValue_label", text);
	}

	if(id == ID_FILTER){
		gboolean on = 0;
		char *cur;
		cur = GetCurrOpt(ID_PRETTYPRINT, NULL);
		if(cur != NULL){
			on = (strcmp(cur, "true") == 0) ? 1 : 0;
			memFree(cur);
		}
		SetActiveCheckButton(window, "PrettyPrint_checkbutton", on);
	}
}


void UpdatePropHPGLWidgets(UIPropDlg *prop, int id)
{
	GtkWidget *window;

	if(id == -1)
		return;

	window = UI_DIALOG(prop)->window;

	if(id == ID_FILTER){
		gboolean on = 0;
		gfloat penwidth;
		char *cur;
		if(prop->color_type){
			cur = GetCurrOpt(ID_BLACKPLOT, NULL);
			if(cur != NULL){
				on = (strcmp(cur, "true") == 0) ? 1 : 0;
				memFree(cur);
			}
			ShowWidget(window, "HPGL2BlackPen_checkbutton");
			SetActiveCheckButton(window, "HPGL2BlackPen_checkbutton", on);
		}else{
			HideWidget(window, "HPGL2BlackPen_checkbutton");
		}

		on = 0;
		cur = GetCurrOpt(ID_FITPLOT, NULL);
		if(cur != NULL){
			on = (strcmp(cur, "true") == 0) ? 1 : 0;
			memFree(cur);
		}
		SetActiveCheckButton(window, "HPGL2FitPlot_checkbutton", on);

		penwidth = GetCurrOptInt(ID_PENWIDTH, 1000) / 1000.0;
		if(IsUS()){
			char *text;
			if(penwidth >= 10.0)
				penwidth = 0.4;
			else
				penwidth /= 25.4;

			text = NameToText(ID_PENWIDTH, "PenWidthTitle ");
			SetTextToLabel(window, "label51", text);

			text = NameToText(ID_PENWIDTH, "PenWidthUnit ");
			SetTextToLabel(window, "label52", text);

			SetSpinMaxValueFloat(window, "HPGL2PenWidth_spinbutton", 0.4);
			SetSpinStepIncrementFloat(window, "HPGL2PenWidth_spinbutton", 0.01);
			SetSpinDigits(window, "HPGL2PenWidth_spinbutton", 2);
		}
		SetSpinButtonFloat(window, "HPGL2PenWidth_spinbutton", penwidth);
	}
}


void UpdatePropFilterWidgets(UIPropDlg *prop, int id)
{
	int filter = 0, i = 0;
	GtkWidget *window;
	int num = id;
	char *cur = NULL;

	if(id == -1)
		return;

	window = UI_DIALOG(prop)->window;
#if OLD_GTK
	for(i = 1; i < 4; i++){
		HideWidget(window, (char *)g_filter_frame_name[i]);
	}

	cur = GetCurrOpt(ID_FILTER, NULL);

	if(cur != NULL){
		if(strncmp(cur, "Image", 5) == 0)
			filter = FILTER_IMAGE;
		else if(strncmp(cur, "Text", 4) == 0)
			filter = FILTER_TEXT;
		else if(strncmp(cur, "HP-GL", 5) == 0)
			filter = FILTER_HPGL;
	}

	if(filter){
		ShowWidget(window, (char *)g_filter_frame_name[filter]);
	}
#else
	cur = GetCurrOpt(ID_FILTER, NULL);

	if(cur != NULL){
		if(strncmp(cur, "Image", 5) == 0)
			filter = FILTER_IMAGE;
		else if(strncmp(cur, "Text", 4) == 0)
			filter = FILTER_TEXT;
		else if(strncmp(cur, "HP-GL", 5) == 0)
			filter = FILTER_HPGL;
	}

	for(i = 1; i < 4; i++){
		if(i != filter){
			HideWidget(window, (char *)g_filter_frame_name[i]);
		}
	}

	if(filter){
		ShowWidget(window, (char *)g_filter_frame_name[filter]);
	}
#endif

	if(id == ID_PAGESIZE || id == 0){
		UpdateCpcaComboWidget(prop, ID_FILTER, "Filter_combo");
	}

	if(num == 0)
		num = ID_FILTER;

	switch(filter){
	case FILTER_IMAGE:
		UpdatePropImageWidgets(prop, num);
		break;
	case FILTER_TEXT:
		UpdatePropTextWidgets(prop, num);
		break;
	case FILTER_HPGL:
		UpdatePropHPGLWidgets(prop, num);
		break;
	case FILTER_NONE:
	default:
		break;
	}
}


typedef	struct{
	int id;
	char *widget_name;
	int type;
	void (*func)(UIPropDlg *prop);
}UpdateFuncTable;

#define	COMBO_WIDGET	1
#define	CHECK_WIDGET	2
#define	OTHER_WIDGET	3

void UpdateTonerSaving(UIPropDlg *prop);
void UpdateJobExecMode(UIPropDlg *prop);
void UpdatePrintStyle(UIPropDlg *prop);
void UpdateDuplex(UIPropDlg *prop);
void UpdateBooklet(UIPropDlg *prop);
void UpdateOutputPartition(UIPropDlg *prop);
void UpdateStapleLocation(UIPropDlg *prop);
void UpdateSelectBy(UIPropDlg *prop);
void UpdateTonerDensity(UIPropDlg *prop);
void UpdateJobAccount(UIPropDlg *prop);
void UpdateCopySetNumbering(UIPropDlg *prop);
void UpdateInterleaf(UIPropDlg *prop);
void UpdateEnableJobAccountingBW(UIPropDlg *prop);

void UpdateCaptTonerDensity(UIPropDlg *prop);
void UpdateCaptHalftone(UIPropDlg *prop);

void UpdateColorMode(UIPropDlg *prop);
void UpdateCyanTonerDensity16(UIPropDlg *prop);
void UpdateMagentaTonerDensity16(UIPropDlg *prop);
void UpdateYellowTonerDensity16(UIPropDlg *prop);

void UpdatePureBlackText(UIPropDlg *prop);
void UpdateBlackOverprint(UIPropDlg *prop);
void UpdateRGBSourceProfile(UIPropDlg *prop);
void UpdateRGBMatchingMethod(UIPropDlg *prop);
void UpdateRGBPureBlackProcess(UIPropDlg *prop);
void UpdateCMYKSimulationProfile(UIPropDlg *prop);
void UpdateCMYKPureBlackProcess(UIPropDlg *prop);
void UpdateUseGrayscaleProfile(UIPropDlg *prop);
void UpdateOutputProfile(UIPropDlg *prop);
void UpdateOHPPrintMode(UIPropDlg *prop);

void UpdateFrontCvr(UIPropDlg *prop);
void UpdateFrontCvrPrintSide(UIPropDlg *prop);
void UpdateFrontCvrPaperSrc(UIPropDlg *prop);
void UpdateBackCvr(UIPropDlg *prop);
void UpdateBackCvrPrintSide(UIPropDlg *prop);
void UpdateBackCvrPaperSrc(UIPropDlg *prop);

void UpdateOffsetNumWidgets(UIPropDlg *prop);
void UpdateSharpnessWidgets(UIPropDlg *prop);
void UpdateSharpnessValue(UIPropDlg *prop);
void UpdateAdvSettingsDlgWidgets (UIPropDlg *prop);
void UpdateBookletDlgCreepWidgets(UIPropDlg *prop);
void UpdateBookletDlgDispCorrWidgets(UIPropDlg *prop);
void UpdateTrimmingWidgets(UIPropDlg *prop);
void UpdateAdjustTrimWidgets(UIPropDlg *prop);
void UpdateFinCfold(UIPropDlg *prop);
void UpdateFinCSetting(UIPropDlg *prop);

UpdateFuncTable update_func_table[] = {
	{ID_BINDEDGE, "BindingLocation_combo", COMBO_WIDGET, NULL},
	{ID_COLLATE, "FinCollate_combo", COMBO_WIDGET, NULL},
	{ID_OUTPUTBIN, "PaperDisti_combo", COMBO_WIDGET, NULL},
	{ID_CNOUTPUTFACE, "OutputSide_combo", COMBO_WIDGET, NULL},
	{ID_INPUTSLOT, "PaperSource_combo", COMBO_WIDGET, NULL},
	{ID_MEDIATYPE, "PaperType_combo", COMBO_WIDGET, NULL},
	{ID_RESOLUTION, "QualityReso_combo", COMBO_WIDGET, NULL},
	{ID_CNHALFTONE, "QualityHalftone_combo", COMBO_WIDGET, NULL},
	{ID_CNBLACKSUBSTITUTE, "QualityGrayComp_combo", COMBO_WIDGET, NULL},
	{ID_CNFINISHER, "OutputOption_combo", COMBO_WIDGET, NULL},
	{ID_CNZFOLDING, "FinZfold_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNSADDLESTITCH, "SaddleStich_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNTRIMMING, NULL, OTHER_WIDGET, UpdateTrimmingWidgets},
	{ID_CNINSERTER, "Inserter_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNFOLDER, "PaperFolding_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNPUNCHER, "PuncherUnit_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNINSERTUNIT, "DocInsertUnit_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNTRIMMER, "TrimmerUnit_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNDUPLEXUNIT, "DuplexUnit_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNPUNCH, "FinHolePanch_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNTONERSAVING, NULL, OTHER_WIDGET, UpdateTonerSaving},
	{ID_CNJOBEXECMODE, NULL, OTHER_WIDGET, UpdateJobExecMode},
	{ID_DUPLEX, NULL, OTHER_WIDGET, UpdateDuplex},
	{ID_BOOKLET, NULL, OTHER_WIDGET, UpdateBooklet},
	{ID_CNOUTPUTPARTITION, NULL, OTHER_WIDGET, UpdateOutputPartition},
	{ID_STAPLELOCATION, NULL, OTHER_WIDGET, UpdateStapleLocation},
	{ID_SELECTBY, NULL, OTHER_WIDGET, UpdateSelectBy},
	{ID_DATANAME, NULL, OTHER_WIDGET, UpdateBoxIDDlgWidgets},
	{ID_CNTONERDENSITY, NULL, OTHER_WIDGET, UpdateTonerDensity},
	{ID_CNIMAGESPECIALPROCESS, "ImgSpecialProc_combo", COMBO_WIDGET, NULL},
	{ID_CNCOPYSETNUMBERING, NULL, OTHER_WIDGET, UpdateCopySetNumbering},
	{ID_CNDRAFTMODE, "QualityDraftMode_combo", COMBO_WIDGET, NULL},
	{ID_CNINTERLEAFSHEET, NULL, OTHER_WIDGET, UpdateInterleaf},
	{ID_CNINTERLEAFPRINT, "PrintInterleaf_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNSRCOPTION, "SourceOption_combo", COMBO_WIDGET, NULL},
	{ID_CNCOLORMODE, NULL, OTHER_WIDGET, UpdateColorMode},
	{ID_CNCOLORHALFTONE, "QualityClrHalftone_combo", COMBO_WIDGET, NULL},
	{ID_CNKEEPGRAY, "QualityKeepGray_combo", COMBO_WIDGET, NULL},
	{ID_CNCTONERDENSITY, NULL, OTHER_WIDGET, UpdateCyanTonerDensity16},
	{ID_CNMTONERDENSITY, NULL, OTHER_WIDGET, UpdateMagentaTonerDensity16},
	{ID_CNYTONERDENSITY, NULL, OTHER_WIDGET, UpdateYellowTonerDensity16},
	{ID_CNPUREBLACKTEXT, NULL, OTHER_WIDGET, UpdatePureBlackText},
	{ID_CNBLACKOVERPRINT, NULL, OTHER_WIDGET, UpdateBlackOverprint},
	{ID_CNRGBSOURCEPROFILE, NULL, OTHER_WIDGET, UpdateRGBSourceProfile},
	{ID_CNRGBMATCHINGMETHOD, NULL, OTHER_WIDGET, UpdateRGBMatchingMethod},
	{ID_CNRGBPUREBLACKPROCESS, NULL, OTHER_WIDGET, UpdateRGBPureBlackProcess},
	{ID_CNCMYKSIMULATIONPROFILE, NULL, OTHER_WIDGET, UpdateCMYKSimulationProfile},
	{ID_CNPUREBLACKPROCESS, NULL, OTHER_WIDGET, UpdateCMYKPureBlackProcess},
	{ID_CNUSEGRAYSCALEPROFILE, NULL, OTHER_WIDGET, UpdateUseGrayscaleProfile},
	{ID_CNOUTPUTPROFILE, NULL, OTHER_WIDGET, UpdateOutputProfile},
	{ID_CNGRADATIONSMOOTHING, "QualityGradeSmooth_combo", COMBO_WIDGET, NULL},
	{ID_CNIMAGEREFINEMENT, "QualityImgRefine_combo", COMBO_WIDGET, NULL},
	{ID_CNLINEREFINEMENT, "QualityLineRefine_combo", COMBO_WIDGET, NULL},
	{ID_CNSET_FRONT_COVER, NULL, OTHER_WIDGET, UpdateFrontCvr},
	{ID_CNFRONT_PRINT_SHEET, NULL, OTHER_WIDGET, UpdateFrontCvrPrintSide},
	{ID_CNFRONT_PAPER_SRC, NULL, OTHER_WIDGET, UpdateFrontCvrPaperSrc},
	{ID_CNSET_BACK_COVER, NULL, OTHER_WIDGET, UpdateBackCvr},
	{ID_CNBACK_PRINT_SHEET, NULL, OTHER_WIDGET, UpdateBackCvrPrintSide},
	{ID_CNBACK_PAPER_SRC, NULL, OTHER_WIDGET, UpdateBackCvrPaperSrc},
	{ID_CNSPOTCOLORMATCHING, "SpotColorMatching_combo", COMBO_WIDGET, NULL},
	{ID_CNOHPPRINTMODE, NULL, OTHER_WIDGET, UpdateOHPPrintMode},
	{ID_CNADVANCEDSMOOTHING, "AdvancedSmooth_combo", COMBO_WIDGET, NULL},
	{ID_CNSPECIALPRINTINGMODE, "SpecialPrintMode_combo", COMBO_WIDGET, NULL},
	{ID_CNSMALLSIZECURL, "QualitySmallSizeCurl_combo", COMBO_WIDGET, NULL},
	{ID_HOLDQUEUE_DATANAME, NULL, OTHER_WIDGET, UpdateHoldQueueDlgWidgets},
	{ID_CNOPTIONSTAPLE, "Staple_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNOFFSETNUM, NULL, OTHER_WIDGET, UpdateOffsetNumWidgets},
	{ID_CNSHIFTSTARTPRINTPOSITION, NULL, OTHER_WIDGET, UpdateAdvSettingsDlgWidgets},
	{ID_CNUSESHARPNESS, NULL, OTHER_WIDGET, UpdateSharpnessWidgets},
	{ID_CNSHARPNESS, NULL, OTHER_WIDGET, UpdateSharpnessValue},
	{ID_CNIMAGECOMPRESSION, "ImageCompression_combo", COMBO_WIDGET, NULL},
	{ID_CNCREEP, NULL, OTHER_WIDGET, UpdateBookletDlgCreepWidgets},
	{ID_CNDISPLACEMENTCORRECTION, NULL, OTHER_WIDGET, UpdateBookletDlgDispCorrWidgets},
	{ID_CNADJUSTTRIM, NULL, OTHER_WIDGET, UpdateAdjustTrimWidgets},
	{ID_CNCFOLDING, NULL, OTHER_WIDGET, UpdateFinCfold},
	{ID_CNCFOLDSETTING, NULL, OTHER_WIDGET, UpdateFinCSetting},
	{ID_CNENABLETRUSTPRINT, "EnableTrustPrint_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNTRUSTPRINT, "TrustPrint_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNTONERVOLUMEADJUSTMENT, "TonerVolumeAdjustment_combo", COMBO_WIDGET, NULL},
	{ID_CNLINECONTROL, "LineControl_combo", COMBO_WIDGET, NULL},
	{ID_CNSPECIALPRINTADJUSTMENTA, "SpecialPrintAdjustmentA_combo", COMBO_WIDGET, NULL},
	{ID_CNSPECIALPRINTADJUSTMENTB, "SpecialPrintAdjustmentB_combo", COMBO_WIDGET, NULL},
	{-1, NULL, -1, NULL},
};

UpdateFuncTable capt_update_func_table[] = {
	{ID_INPUTSLOT, "CaptPaperSource_combo", COMBO_WIDGET, NULL},
	{ID_MEDIATYPE, "CaptPaperType_combo", COMBO_WIDGET, NULL},
	{ID_COLLATE, "CaptCollate_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNTONERSAVING, "CaptTonerSave_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNSUPERSMOOTH, "ImgRefinement_checkbutton", CHECK_WIDGET, NULL},
	{ID_CNHALFTONE, NULL, OTHER_WIDGET, UpdateCaptHalftone},
	{ID_CNTONERDENSITY, NULL, OTHER_WIDGET, UpdateCaptTonerDensity},
	{-1, NULL, -1, NULL},
};

void UpdateSharpnessWidgets(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt = NULL;
	gboolean check;

	opt = GetCurrOpt(ID_CNUSESHARPNESS, NULL);
	if(opt != NULL){
		if(strcmp(opt, "True") == 0)
			check = TRUE;
		else
			check = FALSE;
		SetCpcaWidgetSensitive(prop, ID_CNUSESHARPNESS, "Sharpness_checkbutton");
		SetWidgetSensitive(window, "Sharpness_label", check);
		SetWidgetSensitive(window, "label240", check);
		SetWidgetSensitive(window, "Sharpness_hscale", check);
		SetWidgetSensitive(window, "label241", check);
		memFree(opt);
	}
}
void UpdateSharpnessValue(UIPropDlg *prop)
{
	int sharpness_value;

	sharpness_value = GetCurrOptInt(ID_CNSHARPNESS, 0);
	GtkWidget *window = UI_DIALOG(prop)->window;
	SetHScaleValue(window, "Sharpness_hscale", sharpness_value);
	SetValueToLabel(window, "Sharpness_label", sharpness_value);
}

void UpdateFinCfold(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt = NULL;
	gboolean check;

	SetCpcaWidgetSensitive(prop, ID_CNCFOLDING, "FinCfold_checkbutton");
	UpdateCpcaComboWidget(prop, ID_CNCFOLDSETTING, "FinCfoldSetting_combo");
	opt = GetCurrOpt(ID_CNCFOLDING, NULL);
	if(opt != NULL){
		if(strcmp(opt, "True") == 0)
			check = TRUE;
		else
			check = FALSE;
		SetWidgetSensitive(window, "FinCfoldSetting_combo", check);
		if(check == FALSE){
			SetTextEntry(window, "FinCfoldSetting_combo_entry", "------------");
		}
		memFree(opt);
	}
}

void UpdateFinCSetting(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt = NULL;
	gboolean check = FALSE;

	UpdateCpcaComboWidget(prop, ID_CNCFOLDSETTING, "FinCfoldSetting_combo");
	opt = GetCurrOpt(ID_CNCFOLDING, NULL);
	if(opt != NULL){
		if(strcmp(opt, "True") == 0)
			check = TRUE;
		memFree(opt);
	}
	SetWidgetSensitive(window, "FinCfoldSetting_combo", check);
	if(check == FALSE)
		SetTextEntry(window, "FinCfoldSetting_combo_entry", "------------");
}
void UpdateOffsetNumWidgets(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	int offset_num;

	offset_num = GetCurrOptInt(ID_CNOFFSETNUM, 1);
	SetSpinButton(window, "OffsetCopyNum_spinbutton", offset_num);
}
void UpdateTrimmingWidgets(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt = NULL;
	gboolean disable;

	SetCpcaWidgetSensitive(prop, ID_CNTRIMMING, "UseTrimmer_checkbutton");
	opt = GetCurrOpt(ID_CNTRIMMING, NULL);
	if(opt != NULL){
		if(strcmp(opt, "True") == 0)
			disable = TRUE;
		else
			disable = FALSE;
		SetWidgetSensitive(window, "TrimWidthAdjusting_frame", disable);
		UpdateAdjustTrimWidgets(prop);
		memFree(opt);
	}
}
void UpdateAdjustTrimWidgets(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt = NULL, *opt1 = NULL;
	double	trim_value;

	SetWidgetSensitive(window, "TrimWidth_hbox", FALSE);
	opt = GetCurrOpt(ID_CNADJUSTTRIM, NULL);
	if(opt != NULL){
		if(strcmp(opt, "Manual") == 0){
			SetActiveCheckButton(window, "TrimManualSetting_radiobutton", TRUE);
			opt1 = GetCurrOpt(ID_CNTRIMMING, NULL);
			if(opt1 != NULL){
				if(strcmp(opt1, "True") == 0)
					SetWidgetSensitive(window, "TrimWidth_hbox", TRUE);
				memFree(opt1);
			}
		}else if(strcasecmp(opt, "None") == 0){
			SetActiveCheckButton(window, "TrimDeviceSetting_radiobutton",  TRUE);
		}
		memFree(opt);
	}

	trim_value = GetCurrOptDouble(ID_CNADJUSTTRIMNUM, 2.0);
	SetSpinButtonFloat(window, "TrimWidth_spinbutton", trim_value);
}
void UpdateBookletDlgCreepWidgets(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->booklet)->window;
	gboolean disable;
	char *opt = NULL;

	opt = GetCurrOpt(ID_CNCREEP, NULL);
	if(opt != NULL){
		SetCpcaWidgetSensitive(prop, ID_CNCREEP, "CreepUse_checkbutton");
		if(strcmp(opt, "True") == 0)
			disable = TRUE;
		else
			disable = FALSE;
		SetWidgetSensitive(window, "Creep_frame", disable);
		UpdateBookletDlgDispCorrWidgets(prop);
		MemFree(opt);
	}
}
void UpdateBookletDlgDispCorrWidgets(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->booklet)->window;
	char *opt = NULL, *opt1 = NULL;
	gboolean disable;
	double gutter_shift_num;

	SetWidgetSensitive(window, "hbox82", FALSE);
	opt = GetCurrOpt(ID_CNDISPLACEMENTCORRECTION, NULL);
	if(opt != NULL){
		if(strcasecmp(opt, "Manual") == 0){
			SetActiveCheckButton(window, "Manual_radiobutton", TRUE);
			opt1 = GetCurrOpt(ID_CNCREEP, NULL);
			if(opt1 != NULL){
				if(strcmp(opt1, "True") == 0)
					SetWidgetSensitive(window, "hbox82", TRUE);
				memFree(opt1);
			}
		}else if(strcmp(opt, "Device") == 0)
			SetActiveCheckButton(window, "DeviceSetting_radiobutton", TRUE);
		else if(strcmp(opt, "Auto") == 0)
			SetActiveCheckButton(window, "Auto_radiobutton", TRUE);
		MemFree(opt);
	}
	gutter_shift_num = GetCurrOptDouble(ID_CNGUTTERSHIFTNUM, 0.0);
	SetSpinButtonFloat(window, "CorrectWidth_spinbutton", gutter_shift_num);
}
void UpdateAdvSettingsDlgWidgets (UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->advsettings)->window;
	char *option, *val;
	gboolean active;
	int front_long, front_short, back_long, back_short;
	double front_long_d, front_short_d, back_long_d, back_short_d;

	if((val = GetCNUIValue("CNEnableDetailShiftPosition")) != NULL){
		if(strcasecmp(val, "True") == 0){
			front_long_d = GetCurrOptDouble(ID_CNSHIFTFRLONGEDGE, 0);
			SetSpinButtonFloat(window, "FrontLong_spinbutton", front_long_d);
			front_short_d = GetCurrOptDouble(ID_CNSHIFTFRSHORTEDGE, 0);
			SetSpinButtonFloat(window, "FrontShort_spinbutton", front_short_d);
			back_long_d = GetCurrOptDouble(ID_CNSHIFTBKLONGEDGE, 0);
			SetSpinButtonFloat(window, "BackLong_spinbutton", back_long_d);
			back_short_d = GetCurrOptDouble(ID_CNSHIFTBKSHORTEDGE, 0);
			SetSpinButtonFloat(window, "BackShort_spinbutton", back_short_d);
		}
	}
	if(val == NULL || strcasecmp(val, "False") == 0){
		front_long = GetCurrOptInt(ID_CNSHIFTFRLONGEDGE, 0);
		SetSpinButton(window, "FrontLong_spinbutton", front_long);
		front_short = GetCurrOptInt(ID_CNSHIFTFRSHORTEDGE, 0);
		SetSpinButton(window, "FrontShort_spinbutton", front_short);
		back_long = GetCurrOptInt(ID_CNSHIFTBKLONGEDGE, 0);
		SetSpinButton(window, "BackLong_spinbutton", back_long);
		back_short = GetCurrOptInt(ID_CNSHIFTBKSHORTEDGE, 0);
		SetSpinButton(window, "BackShort_spinbutton", back_short);
	}
	MemFree(val);

	option = GetCurrOpt(ID_CNSHIFTSTARTPRINTPOSITION, NULL);
	if(option != NULL){
		SetCpcaWidgetSensitive(prop, ID_CNSHIFTSTARTPRINTPOSITION, "ShiftPosition_checkbutton");
		active = (strcasecmp(option, "True") == 0) ? TRUE : FALSE;
		SetWidgetSensitive(window, "table24", active);
	}
	memFree(option);
}

void UpdateTonerSaving(UIPropDlg *prop)
{
	if(prop->ui_type == PRINTER_TYPE_CAPT_BIND)
		SetCpcaWidgetSensitive(prop, ID_CNTONERSAVING, "Capt21TonerSave_checkbutton");
	else
		UpdateCpcaComboWidget(prop, ID_CNTONERSAVING, "QualityTonerSave_combo");
}

void UpdateJobExecMode(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt;
	UpdateCpcaComboWidget(prop, ID_CNJOBEXECMODE, "OutputMethod_combo");
	opt = GetCurrOpt(ID_CNJOBEXECMODE, NULL);
	if(opt != NULL){
		gboolean disable;
		disable = (strcmp(opt, "print") == 0) ? FALSE : TRUE;
		SetWidgetSensitive(window, "OutputMethod_button", disable);
		memFree(opt);
	}
}

void UpdatePrintStyle(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt;
	int side1 = 0;
	opt = GetCurrOpt(ID_DUPLEX, NULL);
	if(opt != NULL){
		if(strcmp(opt, "None") == 0 || strcmp(opt, "False") == 0){
			SetActiveCheckButton(window, "Sided2Print_radiobutton", FALSE);
			side1++;
		}else{
			SetActiveCheckButton(window, "Sided2Print_radiobutton", TRUE);
		}
		memFree(opt);
	}else{
		side1++;
	}

	opt = GetCurrOpt(ID_BOOKLET, NULL);
	if(opt != NULL){
		if(strcmp(opt, "None") == 0 || strcmp(opt, "False") == 0){
			SetActiveCheckButton(window, "BookletPrint_radiobutton", FALSE);
			HideWidget(window, "FinBooklet_frame");
			ShowWidget(window, "Finishing_frame");
			SetWidgetSensitive(window, "BookletSetting_button", FALSE);
			side1++;
		}else{
			SetActiveCheckButton(window, "BookletPrint_radiobutton", TRUE);
			HideWidget(window, "Finishing_frame");
			ShowWidget(window, "FinBooklet_frame");
			SetWidgetSensitive(window, "BookletSetting_button", TRUE);
		}
		memFree(opt);
	}else{
		side1++;
	}

	if(side1 == 2)
		SetActiveCheckButton(window, "Sided1Print_radiobutton", TRUE);
}

void UpdateDuplex(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	int disable;
	UpdatePrintStyle(prop);
	disable = GetCurrDisable(ID_DUPLEX, NULL);
	if(disable != -1){
		disable = disable > 0 ? 0 : 1;
		SetWidgetSensitive(window, "Sided2Print_radiobutton", disable);
	}
}

void UpdateBooklet(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	int disable;
	gboolean active;
	UpdatePrintStyle(prop);
	disable = GetCurrDisable(ID_BOOKLET, NULL);
	if(disable != -1){
		disable = disable > 0 ? 0 : 1;
		SetWidgetSensitive(window, "BookletPrint_radiobutton", disable);
		active = GetToggleButtonActive(window, "BookletPrint_radiobutton");
		SetWidgetSensitive(window, "BookletSetting_button", active);
	}
}

void UpdateOutputPartition(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *part, *plist;
	char *list = GetOptionList(ID_CNOUTPUTPARTITION);
	SetWidgetSensitive(window, "OffsetCopyNum_frame", FALSE);
	part = GetCurrOpt(ID_CNOUTPUTPARTITION, list);
	if(part != NULL){
		if(strcmp(part, "offset") == 0){
			SetActiveCheckButton(window, "FinOffset_checkbutton", TRUE);
			SetActiveCheckButton(window, "FinRotate_checkbutton", FALSE);
			SetWidgetSensitive(window, "FinOffset_checkbutton", TRUE);
			SetWidgetSensitive(window, "FinRotate_checkbutton", FALSE);
			SetWidgetSensitive(window, "OffsetCopyNum_frame", TRUE);
		}else if(strcmp(part, "rotation") == 0){
			SetActiveCheckButton(window, "FinRotate_checkbutton", TRUE);
			SetActiveCheckButton(window, "FinOffset_checkbutton", FALSE);
			SetWidgetSensitive(window, "FinRotate_checkbutton", TRUE);
			SetWidgetSensitive(window, "FinOffset_checkbutton", FALSE);
		}else{
			int disable = 0;
			SetActiveCheckButton(window, "FinOffset_checkbutton", FALSE);
			if((plist = strstr(list, "offset")) != NULL){
				char *ptr;
				if((ptr = strchr(plist, '<')) != NULL){
					ptr++;
					disable = atoi(ptr);
				}
			}
			disable = disable > 0 ? 0 : 1;
			SetWidgetSensitive(window, "FinOffset_checkbutton", disable);

			SetActiveCheckButton(window, "FinRotate_checkbutton", FALSE);
			if((plist = strstr(list, "rotation")) != NULL){
				char *ptr;
				if((ptr = strchr(plist, '<')) != NULL){
					ptr++;
					disable = atoi(ptr);
				}
			}
			disable = disable > 0 ? 0 : 1;
			SetWidgetSensitive(window, "FinRotate_checkbutton", disable);
		}
		memFree(part);
		memFree(list);
	}
}

void UpdateStapleLocation(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt;
	UpdateCpcaComboWidget(prop, ID_STAPLELOCATION, "StaplePosition_combo");
	opt = GetCurrOpt(ID_COLLATE, NULL);
	if(opt != NULL){
		gboolean disable = FALSE;
		if((strcmp(opt, "Staple") == 0) || (strcmp(opt, "StapleCollate") == 0) || (strcmp(opt, "StapleGroup") == 0))
			disable = TRUE;
		else
			disable = FALSE;
		SetWidgetSensitive(window, "FinStaple_hbox", disable);
		if(disable == FALSE){
			SetTextEntry(window, "StaplePosition_combo_entry", "------------");
		}
		memFree(opt);
	}
}

void UpdateSelectBy(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	int selectby = GetCurrOptInt(ID_SELECTBY, 1);
	switch(selectby){
	case SELECTBY_INPUTSLOT:
		SetActiveCheckButton(window, "PaperSource_radiobutton", TRUE);
		ShowWidget(window, "PaperSource_frame");
		HideWidget(window, "PaperType_frame");
		UpdateCpcaComboWidget(prop, ID_INPUTSLOT, "PaperSource_combo");
		SetActiveCheckButton(window, "PrintInterleaf_checkbutton", FALSE);
		SetActiveCheckButton(window, "Interleaf_checkbutton", FALSE);
		break;
	case SELECTBY_MEDIATYPE:
		SetActiveCheckButton(window, "PaperType_radiobutton", TRUE);
		ShowWidget(window, "PaperType_frame");
		HideWidget(window, "PaperSource_frame");
		UpdateCpcaComboWidget(prop, ID_MEDIATYPE, "PaperType_combo");
		break;
	case SELECTBY_NONE:
		break;
	}
}

void UpdateJobAccount(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	gboolean enable, active;
	enable = (prop->user_id == 0) ? TRUE : FALSE;
	if(GetCurrOptInt(ID_SPECIAL_FUNC, 0)){
		active = GetCurrOptInt(ID_JOBACCOUNT, 0);
		SetActiveCheckButton(window, "JobAccouting_checkbutton", active);
		UpdateEnableJobAccountingBW(prop);
		SetWidgetSensitive(window, "AllowPasswd_checkbutton", active);
		SetWidgetSensitive(window, "JobAccount_button", active);
	}
	SetWidgetSensitive(window, "JobAccounting_frame", enable);
}

void UpdateEnableJobAccountingBW(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	gboolean enable, active;
	if(GetCurrOptInt(ID_SPECIAL_FUNC, 0)){
		enable = GetToggleButtonActive(window, "JobAccouting_checkbutton");
		if(enable){
			active = GetCurrOptInt(ID_DISABLE_JOBACCOUNT_BW, 0);
			SetActiveCheckButton(window, "DisableJobAccountingBW_checkbutton", active);
		}else
			UpdateDataInt(ID_DISABLE_JOBACCOUNT_BW,0);
		SetWidgetSensitive(window, "DisableJobAccountingBW_checkbutton", enable);
	}
}

void UpdateCopySetNumbering(UIPropDlg *prop)
{
	if(prop == NULL ||prop->captfin == NULL)
		return;
	GtkWidget *window = UI_DIALOG(prop->captfin)->window;
	gboolean active;
	gint startnum;

	ShowWidget(window, "CopyNumbering_frame");
	SetCpcaWidgetSensitive(prop, ID_CNCOPYSETNUMBERING, "CopyNumbering_checkbutton");

	active = GetToggleButtonActive(window, "CopyNumbering_checkbutton");
	SetWidgetSensitive(window, "StartNumberTitle_label", active);
	SetWidgetSensitive(window, "StartNumber_spinbutton", active);
	SetWidgetSensitive(window, "StartNumberRange_label", active);

	startnum = GetCurrOptInt(ID_STARTNUM, 1);
	SetSpinButton(window, "StartNumber_spinbutton", startnum);
}

void UpdateInterleaf(UIPropDlg *prop)
{
	int selectby;
	char *cur;
	gboolean active;
	GtkWidget *window = UI_DIALOG(prop)->window;

	UpdateCpcaComboWidget(prop, ID_CNINTERLEAFMEDIATYPE, "InterleafMediaType_combo");

	selectby = GetCurrOptInt(ID_SELECTBY, 1);
	switch(selectby){
	case SELECTBY_INPUTSLOT:
		SetWidgetSensitive(window, "Interleaf_checkbutton", FALSE);
		break;
	case SELECTBY_MEDIATYPE:
	case SELECTBY_NONE:
		SetCpcaWidgetSensitive(prop, ID_CNINTERLEAFSHEET, "Interleaf_checkbutton");
		break;
	}

	cur = GetCurrOpt(ID_CNINTERLEAFSHEET, NULL);
	if(cur != NULL){
		active = (strcmp(cur, "True") == 0) ? TRUE : FALSE;
		SetWidgetSensitive(window, "InterleafMediaType_combo", active);
		SetWidgetSensitive(window, "FrontBackCvrSet_button", !active);
		if(active == FALSE){
			SetTextEntry(window, "InterleafMediaType_combo_entry", "------------");
		}
	}
}

void UpdateBlackTonerDensity16(UIPropDlg *prop, char *value)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	int num = atoi(value);
	SetHScaleValue(window, "Black_hscale", num);
}

void UpdateTonerDensity(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *toner, *list;
	list = GetOptionList(ID_CNTONERDENSITY);
	if(list != NULL){
		toner = GetCurrOpt(ID_CNTONERDENSITY, list);
		if(strstr(list, "16") == NULL){
			if(toner != NULL){
				if(strcmp(toner, "1") == 0)
					SetActiveCheckButton(window, "Capt21TonerDensity1_radiobutton", TRUE);
				else if(strcmp(toner, "2") == 0)
					SetActiveCheckButton(window, "Capt21TonerDensity2_radiobutton", TRUE);
				else if(strcmp(toner, "4") == 0)
					SetActiveCheckButton(window, "Capt21TonerDensity4_radiobutton", TRUE);
				else if(strcmp(toner, "5") == 0)
					SetActiveCheckButton(window, "Capt21TonerDensity5_radiobutton", TRUE);
				else
					SetActiveCheckButton(window, "Capt21TonerDensity3_radiobutton", TRUE);
			}
		}else{
			if(toner != NULL)
				UpdateBlackTonerDensity16(prop, toner);
		}
		memFree(toner);
		memFree(list);
	}
}

void UpdateCyanTonerDensity16(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	int num = GetCurrOptInt(ID_CNCTONERDENSITY, 9);
	SetHScaleValue(window, "Cyan_hscale", num);
}

void UpdateMagentaTonerDensity16(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	int num = GetCurrOptInt(ID_CNMTONERDENSITY, 9);
	SetHScaleValue(window, "Magenta_hscale", num);
}

void UpdateYellowTonerDensity16(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	int num = GetCurrOptInt(ID_CNYTONERDENSITY, 9);
	SetHScaleValue(window, "Yellow_hscale", num);
}

void UpdateCaptHalftone(UIPropDlg *prop)
{
	GtkWidget *window;
	char *halftone;
	window = UI_DIALOG(prop)->window;
	halftone = GetCurrOpt(ID_CNHALFTONE, NULL);
	if(halftone != NULL){
		if(strcmp(halftone, "pattern2") == 0)
			SetActiveCheckButton(window, "CaptHalftoneP2_radiobutton", TRUE);
		else if(strcmp(halftone, "pattern3") == 0)
			SetActiveCheckButton(window, "CaptHalftoneSmooth_radiobutton", TRUE);
		else
			SetActiveCheckButton(window, "CaptHalftoneP1_radiobutton", TRUE);
		memFree(halftone);
	}
}

void UpdateCaptTonerDensity(UIPropDlg *prop)
{
	GtkWidget *window;
	char *toner;
	window = UI_DIALOG(prop)->window;

	toner = GetCurrOpt(ID_CNTONERDENSITY, NULL);
	if(toner != NULL){
		if(strcmp(toner, "1") == 0)
			SetActiveCheckButton(window, "CaptTonerDensity1_radiobutton", TRUE);
		else if(strcmp(toner, "2") == 0)
			SetActiveCheckButton(window, "CaptTonerDensity2_radiobutton", TRUE);
		else if(strcmp(toner, "4") == 0)
			SetActiveCheckButton(window, "CaptTonerDensity4_radiobutton", TRUE);
		else if(strcmp(toner, "5") == 0)
			SetActiveCheckButton(window, "CaptTonerDensity5_radiobutton", TRUE);
		else
			SetActiveCheckButton(window, "CaptTonerDensity3_radiobutton", TRUE);
		memFree(toner);
	}
}

void SetColorModeSensitive(UIPropDlg *prop, int sense)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	SetWidgetSensitive(window, "ColorHalftone_hbox", sense);
	SetWidgetSensitive(window, "Cyan_label1", sense);
	SetWidgetSensitive(window, "Cyan_label2", sense);
	SetWidgetSensitive(window, "Cyan_label3", sense);
	SetWidgetSensitive(window, "Magenta_label1", sense);
	SetWidgetSensitive(window, "Magenta_label2", sense);
	SetWidgetSensitive(window, "Magenta_label3", sense);
	SetWidgetSensitive(window, "Yellow_label1", sense);
	SetWidgetSensitive(window, "Yellow_label2", sense);
	SetWidgetSensitive(window, "Yellow_label3", sense);
	SetWidgetSensitive(window, "Cyan_hscale", sense);
	SetWidgetSensitive(window, "Magenta_hscale", sense);
	SetWidgetSensitive(window, "Yellow_hscale", sense);
	SetWidgetSensitive(window, "KeepGray_frame", sense);

	SetWidgetSensitive(window, "Halftone_hbox", !sense);
}

void UpdateColorMode(UIPropDlg *prop)
{
	UpdateCpcaComboWidget(prop, ID_CNCOLORMODE, "ColorMode_combo");
	if((prop->ui_type == PRINTER_TYPE_CAPT_BIND) || (prop->ui_type == PRINTER_TYPE_UFR2)){
		char *mode = GetCurrOpt(ID_CNCOLORMODE, NULL);
		if(mode != NULL){
			if(strcasecmp(mode, "Auto") == 0
			|| strcasecmp(mode, "Color") == 0){
				SetColorModeSensitive(prop, TRUE);
			}else if(strcasecmp(mode, "Mono") == 0){
				SetColorModeSensitive(prop, FALSE);
			}
			memFree(mode);
		}
	}
}

void UpdatePureBlackText(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->profile)->window;
	GetOptToGList(window, ID_CNPUREBLACKTEXT, "ProfileDlgPBText_combo");
}

void UpdateBlackOverprint(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->profile)->window;
	GetOptToGList(window, ID_CNBLACKOVERPRINT, "ProfileDlgBOverprint_combo");
}

void UpdateRGBSourceProfile(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->profile)->window;
	GetOptToGList(window, ID_CNRGBSOURCEPROFILE, "ProfileDlgSourceProfile_combo");
}

void UpdateRGBMatchingMethod(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->profile)->window;
	GetOptToGList(window, ID_CNRGBMATCHINGMETHOD, "ProfileDlgMatchingMethod_combo");
}

void UpdateRGBPureBlackProcess(UIPropDlg *prop)
{
	char *list;
	int disable;
	GtkWidget *window;
	window = UI_DIALOG(prop->profile)->window;

	list = GetOptionList(ID_CNRGBPUREBLACKPROCESS);
	if(list == NULL)
		return;

	disable = GetCurrDisable(ID_CNRGBPUREBLACKPROCESS, list);

	if(disable != -1){
		int active;
		disable = disable > 0 ? 0 : 1;
		active = GetActive(ID_CNRGBPUREBLACKPROCESS, list);
		if(active > -1)
			SetActiveCheckButton(window, "ProfileDlgRGBPBProcess_checkbutton", active);
		SetWidgetSensitive(window, "ProfileDlgRGBPBProcess_checkbutton", disable);
	}
	memFree(list);
}

void UpdateCMYKSimulationProfile(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->profile)->window;
	GetOptToGList(window, ID_CNCMYKSIMULATIONPROFILE, "ProfileDlgCMYKSimulation_combo");
}

void UpdateCMYKPureBlackProcess(UIPropDlg *prop)
{
	char *list;
	int disable;
	GtkWidget *window;
	window = UI_DIALOG(prop->profile)->window;

	list = GetOptionList(ID_CNPUREBLACKPROCESS);
	if(list == NULL)
		return;

	disable = GetCurrDisable(ID_CNPUREBLACKPROCESS, list);

	if(disable != -1){
		int active;
		disable = disable > 0 ? 0 : 1;
		active = GetActive(ID_CNPUREBLACKPROCESS, list);
		if(active > -1)
			SetActiveCheckButton(window, "ProfileDlgCMYKPBProcess_checkbutton", active);
		SetWidgetSensitive(window, "ProfileDlgCMYKPBProcess_checkbutton", disable);
	}
	memFree(list);
}

void UpdateUseGrayscaleProfile(UIPropDlg *prop)
{
	char *list;
	int disable;
	GtkWidget *window;
	window = UI_DIALOG(prop->profile)->window;

	list = GetOptionList(ID_CNUSEGRAYSCALEPROFILE);
	if(list == NULL)
		return;

	disable = GetCurrDisable(ID_CNUSEGRAYSCALEPROFILE, list);

	if(disable != -1){
		int active;
		disable = disable > 0 ? 0 : 1;
		active = GetActive(ID_CNUSEGRAYSCALEPROFILE, list);
		if(active > -1)
			SetActiveCheckButton(window, "ProfileDlgUseGrayscale_checkbutton", active);
		SetWidgetSensitive(window, "ProfileDlgUseGrayscale_checkbutton", disable);
	}
	memFree(list);
}

void UpdateOutputProfile(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->profile)->window;
	GetOptToGList(window, ID_CNOUTPUTPROFILE, "ProfileDlgOutput_combo");
}

void UpdateFrontCvr(UIPropDlg *prop)
{
	char *list;
	int disable;

	GtkWidget *window;
	window = UI_DIALOG(prop->ftbkcvr)->window;

	GetOptToGList(window, ID_CNFRONT_PRINT_SHEET, "FrontCvrPrintSide_combo");
	GetOptToGList(window, ID_CNFRONT_PAPER_SRC, "FrontCvrPaperSrc_combo");

	list = GetOptionList(ID_CNSET_FRONT_COVER);
	if(list == NULL)
		return;

	disable = GetCurrDisable(ID_CNSET_FRONT_COVER, list);

	if(disable != -1){
		int active;
		disable = disable > 0 ? 0 : 1;
		active = GetActive(ID_CNSET_FRONT_COVER, list);
		if(active > -1)
		{
			SetActiveCheckButton(window, "FrontCvr_checkbutton", active);
			SetWidgetSensitive(window, "FrontCvrPrintSide_hbox", active);
			SetWidgetSensitive(window, "FrontCvrPaperSrc_hbox", active);
			if(active == 0){
				SetTextEntry(window, "FrontCvrPrintSide_combo_entry", "------------");
				SetTextEntry(window, "FrontCvrPaperSrc_combo_entry", "------------");
			}
		}
		SetWidgetSensitive(window, "FrontCvr_checkbutton", disable);
	}
	memFree(list);
}

void UpdateFrontCvrPrintSide(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->ftbkcvr)->window;
	GetOptToGList(window, ID_CNFRONT_PRINT_SHEET, "FrontCvrPrintSide_combo");
}

void UpdateFrontCvrPaperSrc(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->ftbkcvr)->window;
	GetOptToGList(window, ID_CNFRONT_PAPER_SRC, "FrontCvrPaperSrc_combo");
}

void UpdateBackCvr(UIPropDlg *prop)
{
	char *list;
	int disable;

	GtkWidget *window;
	window = UI_DIALOG(prop->ftbkcvr)->window;

	GetOptToGList(window, ID_CNBACK_PRINT_SHEET, "BackCvrPrintSide_combo");
	GetOptToGList(window, ID_CNBACK_PAPER_SRC, "BackCvrPaperSrc_combo");

	list = GetOptionList(ID_CNSET_BACK_COVER);
	if(list == NULL)
		return;

	disable = GetCurrDisable(ID_CNSET_BACK_COVER, list);

	if(disable != -1){
		int active;
		disable = disable > 0 ? 0 : 1;
		active = GetActive(ID_CNSET_BACK_COVER, list);
		if(active > -1)
		{
			SetActiveCheckButton(window, "BackCvr_checkbutton", active);
			SetWidgetSensitive(window, "BackCvrPrintSide_hbox", active);
			SetWidgetSensitive(window, "BackCvrPaperSrc_hbox", active);
			if(active == 0){
				SetTextEntry(window, "BackCvrPrintSide_combo_entry", "------------");
				SetTextEntry(window, "BackCvrPaperSrc_combo_entry", "------------");
			}
		}
		SetWidgetSensitive(window, "BackCvr_checkbutton", disable);
	}
	memFree(list);
}

void UpdateBackCvrPrintSide(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->ftbkcvr)->window;
	GetOptToGList(window, ID_CNBACK_PRINT_SHEET, "BackCvrPrintSide_combo");
}

void UpdateBackCvrPaperSrc(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop->ftbkcvr)->window;
	GetOptToGList(window, ID_CNBACK_PAPER_SRC, "BackCvrPaperSrc_combo");
}

void UpdateOHPPrintMode(UIPropDlg *prop)
{
	GtkWidget *window = UI_DIALOG(prop)->window;
	char *opt;
	gboolean enable = FALSE;
	UpdateCpcaComboWidget(prop, ID_CNOHPPRINTMODE, "OHPPrintMode_combo");
	opt = GetCurrOpt(ID_MEDIATYPE, NULL);
	if(opt != NULL){
		if(strcmp(opt, "OHP") == 0)
			enable = TRUE;
		else
			enable = FALSE;
		memFree(opt);
	}
	SetWidgetSensitive(window, "OHPPrintMode_frame", enable);
	if(enable == FALSE){
		SetTextEntry(window, "OHPPrintMode_combo_entry", "------------");
	}
}

UpdateFuncTable* FindFunc(int id, int type)
{
	UpdateFuncTable *table = NULL;

	if(type == PRINTER_TYPE_CAPT)
		table = capt_update_func_table;
	else
		table = update_func_table;

	while(table->id != -1){
		if(table->id == id)
			return table;
		table++;
	}
	return NULL;
}

void UpdatePropPPDWidgets(UIPropDlg *prop, int id)
{
	UpdateFuncTable *table = NULL;
	GtkWidget *window = UI_DIALOG(prop)->window;

	if(id == -1)
		return;

	table = FindFunc(id, prop->ui_type);

	if(table != NULL){
		switch(table->type){
		case COMBO_WIDGET:
			UpdateCpcaComboWidget(prop, table->id, table->widget_name);
			break;
		case CHECK_WIDGET:
			SetCpcaWidgetSensitive(prop, table->id, table->widget_name);
			break;
		case OTHER_WIDGET:
			table->func(prop);
			break;
		}
	}
	if(prop->special_func && (prop->user_id == 0)){
		gboolean active;
		active = GetToggleButtonActive(window, "JobAccouting_checkbutton");
		SetWidgetSensitive(window, "AllowPasswd_checkbutton", active);
		SetWidgetSensitive(window, "JobAccount_button", active);
		SetWidgetSensitive(window, "DisableJobAccountingBW_checkbutton", active);
		if(!active){
			SetActiveCheckButton(window, "AllowPasswd_checkbutton", active);
			SetActiveCheckButton(window, "DisableJobAccountingBW_checkbutton", active);
		}
	}
#if _UI_DEBUG
	DebugDisable(g_main_window->data, id);
#endif
}

void FindUpdateWidget(UIPropDlg *prop, char *str)
{
	char id[256], *ptr, *tmp;

	ptr = id;
	tmp = str;
	while(1){
		if(*tmp == '\0' || *tmp == '\n'){
			*ptr = '\0';
			UpdatePropPPDWidgets(prop, atoi(id));
			break;
		}
		if(*tmp == ','){
			*ptr = '\0';
			tmp++;
			UpdatePropPPDWidgets(prop, atoi(id));
			ptr = id;
		}
		*ptr = *tmp;
		ptr++;
		tmp++;
	}

	UpdateJobAccount(prop);
}

void UpdateAllPPDWidgets(UIPropDlg *prop)
{
	GtkWidget *window;

	window = UI_DIALOG(prop)->window;

	FindUpdateWidget(prop, prop->id_list);
}

void UpdateWidget(int id, char *str)
{
UI_DEBUG("UpdateWidget <<<%s>>>\n", str);
	if(str == NULL)
		return;
	if(strcmp(str, "NoChange") == 0)
		return;

	if(id < ID_COMMON_OPTION){
		if(id == ID_PAGESIZE){
			UpdatePropGeneralWidgets(g_main_window->prop, ID_PAGESIZE);
			UpdatePropFilterWidgets(g_main_window->prop, ID_PAGESIZE);
			UpdateAllPPDWidgets(g_main_window->prop);
		}else{
			if(str != NULL)
				FindUpdateWidget(g_main_window->prop, str);
		}
	}else if(id == ID_PRINTERNAME){
		if(str != NULL){
			DisposeDialogs();
			if(CreateDialogs(g_main_window))
				UpdateMainDlgWidgets(g_main_window, ID_PRINTERNAME);
			else
				gtk_main_quit();
		}
	}else if(id == ID_ORIENTATION_REQUESTED){
		UpdatePropPPDWidgets(g_main_window->prop, ID_BINDEDGE);
		UpdatePropPPDWidgets(g_main_window->prop, ID_STAPLELOCATION);
	}else if(id <= ID_IMAGE_OPTION){
		if(id == ID_FILTER){
			UpdatePropFilterWidgets(g_main_window->prop, id);
		}else if(id < ID_NUMBER_UP){
			UpdateMainDlgWidgets(g_main_window, id);
		}else{
			UpdatePropGeneralWidgets(g_main_window->prop, id);
		}
	}else if(id > ID_IMAGE_OPTION){
		UpdatePropFilterWidgets(g_main_window->prop, id);
	}
}


typedef struct{
	int	id;
	char	*widget_name1;
	char	*widget_name2;
	int	frame;
}ShowWidgetTable;

static ShowWidgetTable widget_table[] = {
	{ID_DUPLEX, "Sided2Print_radiobutton", NULL, 0},
	{ID_BOOKLET, "BookletPrint_radiobutton", "BookletSetting_button", 0},
	{ID_BINDEDGE, "BindingLocation_label", "BindingLocation_hbox", 0},
	{ID_COLLATE, "FinishingSel_hbox", NULL, 0},
	{ID_OUTPUTBIN, "PaperDisit_frame", NULL, 0},
	{ID_STAPLELOCATION, "FinStaple_hbox", NULL, 0},
	{ID_CNZFOLDING, "FinZfold_checkbutton", NULL, 0},
	{ID_CNPUNCH, "FinHolePanch_checkbutton", NULL, 0},
	{ID_CNSADDLESTITCH, "SaddleStich_checkbutton", NULL, 0},
	{ID_CNTRIMMING, "UseTrimmer_checkbutton", NULL, 0},
	{ID_CNINSERTER, "Inserter_checkbutton", NULL, 0},
	{ID_RESOLUTION, "QualityReso_frame", NULL, 0},
	{ID_CNCOLORMODE, "ColorMode_frame", NULL, 0},
	{ID_CNHALFTONE, "Halftone_hbox", NULL, 0},
	{ID_CNTONERSAVING, "TonerSave_hbox", "Capt21TonerSave_checkbutton", 0},
	{ID_CNBLACKSUBSTITUTE, "QualityGrayComp_frame", NULL, 0},
	{ID_CNJOBEXECMODE, "OutputMethod_frame", NULL, 1},
	{ID_CNFINISHER, "OutputOption_table", NULL, 1},
	{ID_CNPUNCHER, "PuncherUnit_checkbutton", NULL, 1},
	{ID_CNFOLDER, "PaperFolding_checkbutton", NULL, 1},
	{ID_CNINSERTUNIT, "DocInsertUnit_checkbutton", "hseparator21", 1},
	{ID_CNTRIMMER, "TrimmerUnit_checkbutton", NULL, 1},
	{ID_CNDUPLEXUNIT, "DuplexUnit_checkbutton", "hseparator21", 1},
	{ID_CNOUTPUTFACE, "FinOutputSide_frame", "hseparator20", 0},
	{ID_INPUTSLOT, "PaperSource_frame", NULL, 0},
	{ID_MEDIATYPE, "PaperType_frame", NULL, 0},
	{ID_CNIMAGESPECIALPROCESS, "ImgSpecialProc_frame", "hseparator20", 0},
	{ID_CNDRAFTMODE, "DraftMode_hbox", NULL, 0},
	{ID_CNINTERLEAFSHEET, "Interleaf_frame", NULL, 0},
	{ID_CNSRCOPTION, "SourceOption_vbox", NULL, 1},
	{ID_CNKEEPGRAY, "KeepGray_frame", NULL, 1},
	{ID_CNCOLORHALFTONE, "ColorHalftone_hbox", NULL, 0},
	{ID_CNPUREBLACKTEXT, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNBLACKOVERPRINT, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNRGBSOURCEPROFILE, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNRGBMATCHINGMETHOD, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNRGBPUREBLACKPROCESS, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNCMYKSIMULATIONPROFILE, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNPUREBLACKPROCESS, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNUSEGRAYSCALEPROFILE, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNOUTPUTPROFILE, "QualityProfileSettings_vbox", NULL, 0},
	{ID_CNGRADATIONSMOOTHING, "QualityGradeSmooth_frame", NULL, 0},
	{ID_CNIMAGEREFINEMENT, "QualityImgRefine_frame", NULL, 0},
	{ID_CNLINEREFINEMENT, "QualityLineRefine_frame", NULL, 0},
	{ID_CNSET_FRONT_COVER, "FrontBackCoverSettings_frame", NULL, 0},
	{ID_CNFRONT_PRINT_SHEET, "FrontBackCoverSettings_frame", NULL, 0},
	{ID_CNFRONT_PAPER_SRC, "FrontBackCoverSettings_frame", NULL, 0},
	{ID_CNSET_BACK_COVER, "FrontBackCoverSettings_frame", NULL, 0},
	{ID_CNBACK_PRINT_SHEET, "FrontBackCoverSettings_frame", NULL, 0},
	{ID_CNBACK_PAPER_SRC, "FrontBackCoverSettings_frame", NULL, 0},
	{ID_CNSPOTCOLORMATCHING, "SpotColorMatching_hbox", NULL, 0},
	{ID_CNOHPPRINTMODE, "OHPPrintMode_frame", NULL, 0},
	{ID_CNADVANCEDSMOOTHING, "AdvancedSmooth_hbox", NULL, 0},
	{ID_CNSPECIALPRINTINGMODE, "SpecialPrintMode_frame", NULL, 0},
	{ID_CNSMALLSIZECURL, "SmallSizeCurl_hbox", NULL, 0},
	{ID_CNOPTIONSTAPLE, "Staple_checkbutton", NULL, 1},
	{ID_CNSHIFTSTARTPRINTPOSITION, "AdvancedSettings_frame", NULL, 0},
	{ID_CNIMAGECOMPRESSION, "ImageCompression_frame", NULL, 0},
	{ID_CNENABLETRUSTPRINT, "EnableTrustPrint_checkbutton", NULL, 1},
	{ID_CNLINECONTROL, "LineControl_frame", NULL, 0},
	{ID_CNTONERVOLUMEADJUSTMENT, "TonerVolumeAdjustment_frame", NULL, 0},
	{ID_CNSPECIALPRINTADJUSTMENTA, "SpecialPrintAdjustmentA_frame", NULL, 0},
	{ID_CNSPECIALPRINTADJUSTMENTB, "SpecialPrintAdjustmentB_frame", NULL, 0},
	{-1, NULL, NULL, 0},
};

static ShowWidgetTable capt_widget_table[] = {
	{ID_INPUTSLOT, 		"CaptPaperSource_hbox", NULL, 0},
	{ID_MEDIATYPE, 		"CaptPaperType_hbox", NULL, 0},
	{ID_COLLATE, 		"CaptFin_frame", NULL, 0},
	{ID_CNHALFTONE,		"CaptQuality_frame", NULL, 0},
	{ID_CNTONERSAVING,	"CaptTonerSave_checkbutton", NULL, 1},
	{ID_CNTONERDENSITY,	"TonerDensity_frame", NULL, 1},
	{ID_CNSUPERSMOOTH,	"ImgRefinement_checkbutton", NULL, 1},
	{-1, NULL, NULL, 0},
};

ShowWidgetTable* FindWidgetName(int type, char *option)
{
	ShowWidgetTable *table = NULL;
	int id = atoi(option);

	if(type == PRINTER_TYPE_CAPT)
		table = capt_widget_table;
	else
		table = widget_table;

	while(table->id != -1){
		if(table->id == id)
			return table;
		table++;
	}
	return NULL;
}

int ShowPropDlgWidget(UIPropDlg *prop, char *option)
{
	GtkWidget *window;
	ShowWidgetTable *table = NULL;
	int ret = 0;
	int id = atoi(option);

	window = UI_DIALOG(prop)->window;
	table = FindWidgetName(prop->ui_type, option);

	if(table != NULL){
		ShowWidget(window, table->widget_name1);
		ShowWidget(window, table->widget_name2);
		ret = table->frame;
	}

	if(id == ID_CNTONERDENSITY){
		char *toner = GetOptionList(ID_CNTONERDENSITY);
		if(toner != NULL){
			if(strstr(toner, "16") != NULL)
				ShowWidget(window, "TonerDensity16_frame");
			else
				ShowWidget(window, "MonoDensity5_table");
			memFree(toner);
		}
		toner = GetOptionList(ID_CNCTONERDENSITY);
		if(toner == NULL)
			SetTextToLabel(window, "Black_label1", "     ");
		else
			memFree(toner);
	}else if(id == ID_CNCTONERDENSITY){
		char *toner = GetOptionList(ID_CNTONERDENSITY);
		if(toner != NULL){
			ShowWidget(window, "Cyan_label1");
			ShowWidget(window, "Cyan_label2");
			ShowWidget(window, "Cyan_label3");
			ShowWidget(window, "Cyan_hscale");
			ShowWidget(window, "Cyan_drawingarea");
		}
	}else if(id == ID_CNMTONERDENSITY){
		char *toner = GetOptionList(ID_CNTONERDENSITY);
		if(toner != NULL){
			ShowWidget(window, "Magenta_label1");
			ShowWidget(window, "Magenta_label2");
			ShowWidget(window, "Magenta_label3");
			ShowWidget(window, "Magenta_hscale");
			ShowWidget(window, "Magenta_drawingarea");
		}
	}else if(id == ID_CNYTONERDENSITY){
		char *toner = GetOptionList(ID_CNTONERDENSITY);
		if(toner != NULL){
			ShowWidget(window, "Yellow_label1");
			ShowWidget(window, "Yellow_label2");
			ShowWidget(window, "Yellow_label3");
			ShowWidget(window, "Yellow_hscale");
			ShowWidget(window, "Yellow_drawingarea");
		}
	}

	return ret;
}

void CreatePropDlgWidget(UIPropDlg *prop)
{
	GtkWidget *window;
	int frame = 0;
	char option[256], *ptr, *plist;
	int fin_details;
	char *enable_offset = NULL;
	char *adjust_trim = NULL;
	char *use_sharpness = NULL;
	char *sharpness= NULL;
	char *cfolding = NULL;
	char *cfold_setting = NULL;
	char *specialprintmode = NULL;

	window = UI_DIALOG(prop)->window;
	prop->id_list = cngplpGetData(UI_DLG_MAIN(prop)->data, ID_PPD_OPTION);
	if(prop->id_list == NULL)
		return;
	plist = prop->id_list;
	ptr = option;
	while(1){
		if(*plist == '\0'){
			*ptr = '\0';
			frame += ShowPropDlgWidget(prop, option);
			break;
		}
		if(*plist == ','){
			*ptr = '\0';
			frame += ShowPropDlgWidget(prop, option);
			plist++;
			ptr = option;
		}
		*ptr = *plist;
		ptr++;
		plist++;
	}
	enable_offset = GetCNUIValue("EnableCNOffsetNum");
	if(enable_offset != NULL){
		if(strcmp(enable_offset,"True") == 0){
			int max_offset_value;
			char *text, label[32];

			ShowWidget(window, "OffsetCopyNum_frame");

			max_offset_value = GetCurrOptInt(ID_CNUIOFFSETMAX, 9999);
			text = NameToText(ID_CNUIOFFSETMAX, "OffsetNum");
			snprintf(label, 31, text, max_offset_value);
			SetTextToLabel(window, "label221", label);
			SetSpinMaxValue(window, "OffsetCopyNum_spinbutton", max_offset_value);
		}else
			HideWidget(window, "OffsetCopyNum_frame");
		MemFree(enable_offset);
	}
	adjust_trim = GetOptionList(ID_CNADJUSTTRIM);
	ptr = GetOptionList(ID_CNTRIMMING);
	if(adjust_trim != NULL && ptr != NULL){
		double max_trim_value;
		double min_trim_value;
		char *text = NULL, label[32];

		ShowWidget(window, "TrimWidthAdjusting_frame");
		ShowWidget(window, "TrimWidth_hbox");

		max_trim_value = GetCurrOptDouble(ID_CNUITRIMVALMAX, 10.2);
		min_trim_value = GetCurrOptDouble(ID_CNUITRIMVALMIN, 2.0);
		text = NameToText(ID_CNUITRIMVALMAX, "TrimWidthRange");
		snprintf(label, 31, text, min_trim_value, max_trim_value);
		SetTextToLabel(window, "label257", label);
		SetSpinMaxMinValueDouble(window, "TrimWidth_spinbutton", max_trim_value, min_trim_value);
	}
	if(adjust_trim != NULL)
		memFree(adjust_trim);
	if(ptr != NULL)
		memFree(ptr);

	use_sharpness = GetOptionList(ID_CNUSESHARPNESS);
	if(use_sharpness != NULL){
		sharpness = GetOptionList(ID_CNSHARPNESS);
		if(sharpness != NULL){
			ShowWidget(window, "Sharpness_table");
			memFree(sharpness);
		}
		memFree(use_sharpness);
	}

	cfolding = GetOptionList(ID_CNCFOLDING);
	if(cfolding != NULL){
		cfold_setting = GetOptionList(ID_CNCFOLDSETTING);
		if(cfold_setting != NULL){
			ShowWidget(window, "FinCfold_checkbutton");
			ShowWidget(window, "FinCfoldSetting_combo");
			memFree(cfold_setting);
		}
		memFree(cfolding);
	}

	fin_details = GetCurrOptInt(ID_CNFINDETAILS, 0);
	if(prop->ui_type == PRINTER_TYPE_CAPT_BIND || fin_details == 1)
		ShowWidget(window, "FinDetails_frame");
	if(prop->ui_type == PRINTER_TYPE_CAPT_BIND){
		ShowWidget(window, "QualityCAPT_vbox");
		HideWidget(window, "TonerSave_hbox");
	}else
		HideWidget(window, "Capt21TonerSave_checkbutton");

	specialprintmode = GetCurrOpt(ID_CNSPECIALPRINTINGMODE, NULL);
	if(specialprintmode != NULL){
		char *label = ToChar(GetItemString(CNGPLP_DATA()->ppd_opt->items_list, kPPD_Items_CNSpecialPrintMode));
		char *text = NameToText(ID_CNSPECIALPRINTINGMODE, label);
		if(text != NULL)
			SetTextToLabel(window, "SpecialPrintMode_label", text);
		memFree(specialprintmode);
		memFree(label);
	}

	if(prop->ui_type == PRINTER_TYPE_CAPT){
		char *ptn = GetOptionList(ID_CNHALFTONE);
		if(ptn != NULL && strstr(ptn, "pattern3") != NULL){
			ShowWidget(window, "CaptHalftoneSmooth_radiobutton");
			memFree(ptn);
		}
		if(frame)
			ShowWidget(window, "CaptDetail_frame");
	}else{
		char *part = GetOptionList(ID_CNOUTPUTPARTITION);
		char *halftone = GetOptionList(ID_CNHALFTONE);
		int selectby = GetCurrOptInt(ID_SELECTBY, 0);
		if(part != NULL){
			if(strstr(part, "offset") != NULL)
				ShowWidget(window, "FinOffset_checkbutton");
			if(strstr(part, "rotation") != NULL)
				ShowWidget(window, "FinRotate_checkbutton");
			memFree(part);
		}

		 if(halftone != NULL){
			char *text;
			if(!prop->color_type){
				text = NameToText(ID_CNHALFTONE, "TitleHalfTone");
				SetTextToLabel(window, "label103", text);
			}
			memFree(halftone);
		}
		if(prop->special_func){
			char *job = GetOptionList(ID_SHOW_DISABLE_JOBACCOUNT_BW);
			ShowWidget(window, "JobAccounting_frame");
			if(job != NULL){
				if(strstr(job, "1") != NULL)
					ShowWidget(window, "DisableJobAccounting_box");
				memFree(job);
			}
			frame++;
		}

		if(selectby)
			ShowWidget(window, "SelectBy_frame");

		if(frame)
			ShowWidget(window, "DeviceSetting_frame");
		else
			SetShowPage(window, UI_PAGE_SPECIAL, FALSE);
	}

	SetFocus(window, "PropOK_button");
}


void ConnectTextViewSignal(GtkWidget *window, gchar *text_view_name, GtkSignalFunc sig_func)
{
	GtkTextView *text_view = NULL;
	GtkTextBuffer *text_buffer = NULL;

	text_view = GTK_TEXT_VIEW(getWidget(window, text_view_name));
	text_buffer = gtk_text_view_get_buffer(text_view);

	g_signal_connect(G_OBJECT(text_buffer), "changed", G_CALLBACK(sig_func), NULL);
}

gchar* GetTextofTextView(GtkWidget *widget, char *text_view_name)
{
	GtkTextView *text_view = NULL;
	GtkTextBuffer *text_buffer = NULL;
	GtkTextIter start_item, end_item;

	text_view = GTK_TEXT_VIEW(getWidget(widget, text_view_name));
	text_buffer = gtk_text_view_get_buffer(text_view);
	gtk_text_buffer_get_start_iter(text_buffer, &start_item);
	gtk_text_buffer_get_end_iter(text_buffer, &end_item);
	return gtk_text_buffer_get_text(text_buffer, &start_item, &end_item,  FALSE);
}

void SetTextofTextView(GtkWidget *widget, gchar *text_view_name, gchar *text, int length)
{
	GtkTextView *text_view = NULL;
	GtkTextBuffer *text_buffer = NULL;

	text_view = GTK_TEXT_VIEW(getWidget(widget, text_view_name));
	text_buffer = gtk_text_view_get_buffer(text_view);
	gtk_text_buffer_set_text(text_buffer, text, length);
}

