// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

/*
  names.cc
  Contains the functions for the starconvert utility which allow it to
  extract star names and designations from a text record.
*/

#include "convert.h"
#include "../classes/constellations.h"
#include "../classes/greek.h"

#include <iostream>
using std::cerr;
using std::cout;
using std::endl;

bool bayer_comm(StringList *, name, StringList *);
bool flam_comm (StringList *, name, StringList *);
bool cspec_comm(StringList *, name, StringList *);
bool dm_comm   (StringList *, name, StringList *) { return false; } // no-op
bool other_comm(StringList *, name, StringList *);

bool bayer(const string &, name, StringList *);
bool flam (const string &, name, StringList *);
bool cspec(const string &, name, StringList *);
bool dm   (const string &, name, StringList *);
bool other(const string &, name, StringList *);

// arrays of function pointers to the above for ease of calling in get_names()

bool (*commentfns[NUM_NAME_TYPES])(StringList *, name, StringList *) =
{ bayer_comm, flam_comm, cspec_comm, dm_comm, other_comm };

bool (*recordfns[NUM_NAME_TYPES])(const string &, name, StringList *) =
{ bayer, flam, cspec, dm, other };

static int get_constnum(const string &s)
{
  if (s.size() <= 3 /* too short */)
    return -1;

  string temp = s.substr(s.size() - 3);
  int constnum = 0;

  while (constnum < NUM_CONSTELLATIONS &&
	 ! starstrings::case_compare_n(temp, constellations[constnum], 3))
    constnum++;
  return (constnum < NUM_CONSTELLATIONS) ? constnum : -1;
}

static int get_greeknum(const string &s)
{
  int greeknum = 0;
  while (greeknum < NUM_GREEK_LETTERS
         && ! starstrings::case_compare_n(s, Greek[greeknum].abbrev,
                                          Greek[greeknum].abbrev.size()))
    greeknum++;
  return (greeknum < NUM_GREEK_LETTERS) ? greeknum : -1;
}


// get_names(): This is the top-level function which loops through the
//  name specifications and calls the appropriate function for each spec.
//  It then takes care of all the substitutions, if any.

void get_names(const string &record, namedata nmd, 
	       StringList *sn, StringList *sc)
               // sn: starnames; sc: starcomments
{
  *sn = StringList();

  // first find all the names
  for (unsigned int i = 0; i < nmd.names.size(); i++) {
    if (nmd.names[i].isNameCommented)
      commentfns[nmd.names[i].type](sc, nmd.names[i], sn);
    else
      recordfns[nmd.names[i].type](record, nmd.names[i], sn);
  }

  // then perform any substitutions
  citerate (std::vector<substitution>, nmd.substs, subst_ptr) {
    if ((nmd.isSubstCaseSensitive && (*subst_ptr).subst1 == (*sn)[0]) ||
        (!nmd.isSubstCaseSensitive &&
	 starstrings::case_compare((*subst_ptr).subst1, (*sn)[0])))
      sn->insert(sn->begin() + (*subst_ptr).insert_posn, (*subst_ptr).subst2);
  }
}


// The following functions do the actual work of extracting names from records
//  and comments.  They return true if successful at extracting a name, false
//  otherwise.

bool bayer(const string &record, name namespec, StringList *starnames)
{
  if (namespec.s.start >= record.size())
    return false;
  if (namespec.s.len < 6) // 3 chars for Greek letter, 3 for constellation
    return false;

  int constnum, greeknum;
  string bname = record.substr(namespec.s.start, namespec.s.len);

  starstrings::stripspace(bname);
  if (bname.size() < 5) return false;
  if ((greeknum = get_greeknum(bname)) < 0) return false;
  if ((constnum = get_constnum(bname)) < 0) return false;

  unsigned int posn;
  if ((posn = bname.find_first_of(DIGITS)) < bname.size()) {
    int supscript = starmath::atoi(bname.substr(posn));
    starnames->push_back(Greek[greeknum].name+"("+starstrings::itoa(supscript)
		         + ") " + constellations[constnum]);
  }
  else
    starnames->push_back(Greek[greeknum].name + " " + constellations[constnum]);
  
  return true;
}


bool flam(const string &record, name namespec, StringList *starnames)
{
  if (namespec.s.start >= record.size())
    return false;
  if (namespec.s.len < 6) // 3 chars for number, 3 for constellation
    return false;

  int constnum, flamnum;
  string fname = record.substr(namespec.s.start, namespec.s.len);
  starstrings::stripspace(fname);

  if (fname.size() < 4) return false;
  if (std::isalpha(fname[fname.size() - 4]))
    // purported "constellation abbreviation" is part of a longer word
    return false;
  if ((constnum = get_constnum(fname)) < 0) return false;
  
  flamnum = starmath::atoi(fname);
  if (flamnum <= 0 || flamnum > 140 /* what's the largest Flamsteed number? */)
    return false;

  starnames->push_back(starstrings::itoa(flamnum)+" "+constellations[constnum]);
  return true;
}


// "cspec" for other "constellation specific" names, e.g. Q Car, RR Lyr
bool cspec(const string &record, name namespec, StringList *starnames)
{
  if (namespec.s.start >= record.size())
    return false;
  if (namespec.s.len < 4) // 3 chars for constellation + at least 1 more
    return false;

  int constnum;
  string cname = record.substr(namespec.s.start, namespec.s.len);

  starstrings::stripspace(cname);
  if (cname.size() < 4)			    return false; // too short
  if ((constnum = get_constnum(cname)) < 0) return false;
  if (get_greeknum(cname) >= 0)             return false; // Bayer designation
  if (starmath::atoi(cname))                return false; // Flamsteed desig.

  cname = cname.substr(0, cname.size() - 3);
  starstrings::stripspace(cname);
  
  starnames->push_back(cname + " " + constellations[constnum]);
  return true;
}


bool dm(const string &record, name namespec, StringList *starnames)
{
  if (namespec.s.start >= record.size())
    return false;
  if (namespec.s.len < 9) return false;

  string dname = record.substr(namespec.s.start, namespec.s.len);
  starstrings::stripspace(dname);
  string catalog, degrees, number, sign;

  // which catalog: CP, SD, BD or CP
  catalog = dname.substr(0, 2);
  starstrings::toupper(catalog);
  if (catalog != "CP" && catalog != "BD" && catalog != "CD" && catalog != "SD")
    return false;

  // value of degree sign (+ or -)
  if (dname.find('-') < dname.size())
    sign = "-";
  else sign = "+";

  // degree part of identifier
  if (dname.find_first_of(DIGITS) >= dname.size() - 3)
    return false;
  degrees = dname.substr(dname.find_first_of(DIGITS), 2);
  starstrings::stripspace(degrees);
  if (degrees.size() == 1) // if `degrees' is one digit, pad with zero
    degrees = string("0") + degrees;

  // numerical part of identifier
  number = dname.substr(dname.find_first_of(DIGITS) + 2);
  starstrings::stripspace(number);
  if (starstrings::isempty(number))
    return false;

  starnames->push_back(catalog + " " + sign + degrees + DEGREE_UTF8 + number);
  return true;
}


bool other(const string &record, name namespec, StringList *starnames)
{
  if (namespec.s.start >= record.size())
    return false;
  if (namespec.s.len <= 0) return false;

  string sname = record.substr(namespec.s.start, namespec.s.len);
  starstrings::stripspace(sname);
  if (starstrings::isempty(sname))
    return false;
  
  if (!starstrings::isempty(namespec.name_prefixes[0]))
    sname = namespec.name_prefixes[0] + " " + sname;
  else if (sname.size() < 3)
    return false;

  // remove gratuitous internal spaces, e.g. "Gl   3" -> "Gl 3"
  StringList s = StringList(sname, ' ');
  s.eraseempty();
  starnames->push_back(s.flatten());
  return true;
}


// We'll use the existing Bayer / Flamsteed functions in the comment versions
//  via the miracle of function pointers.

bool var_comm(StringList *starcomments, name namespec, StringList *starnames,
	      bool (*star_fn)(const string &, name, StringList *))
{
  if (starcomments->size() < 2) return false;
  unsigned int oldlen = starnames->size();
  name testspec;
  testspec.s.start = 0;

  for (unsigned int i = 1; i < starcomments->size(); i++) {
    string teststring = (*starcomments)[i - 1] + " " + (*starcomments)[i];
    testspec.s.len = teststring.size();

    if ((*star_fn)(teststring, testspec, starnames)) {
      i--;
      starcomments->erase(starcomments->begin() + i + 1);
      starcomments->erase(starcomments->begin() + i);
    }
  }
  return (oldlen - starnames->size() > 0);
}


bool bayer_comm(StringList *starcomments, name namespec, StringList *starnames)
{ return var_comm(starcomments, namespec, starnames, bayer); }

bool flam_comm(StringList *starcomments, name namespec, StringList *starnames)
{ return var_comm(starcomments, namespec, starnames, flam); }

bool cspec_comm(StringList *starcomments, name namespec, StringList *starnames)
{ return var_comm(starcomments, namespec, starnames, cspec); }


bool other_comm(StringList *starcomments, name namespec, StringList *starnames)
{
  if (starcomments->size() < 2) return false;
  unsigned int oldlen = starnames->size();

  citerate (StringList, namespec.name_prefixes, prefix_ptr) {
    StringList prefix = StringList(*prefix_ptr, '=');
    prefix.stripspace();

    if (! starstrings::isempty(prefix[0])) {
      if (prefix.size() < 2) prefix.push_back(prefix[0]);

      for (unsigned int i = 1; i < starcomments->size(); i++) {
	if ((*starcomments)[i - 1] == prefix[0]) {
	  starnames->push_back(prefix[1] + " " + (*starcomments)[i]);
	  i--;
	  starcomments->erase(starcomments->begin() + i + 1);
	  starcomments->erase(starcomments->begin() + i);
	}
      }
    }
  }
  return (oldlen - starnames->size() > 0);
}

