/***
  * The contents of this file are subject to the Mozilla Public
  * License Version 1.1 (the "License"); you may not use this file
  * except in compliance with the License. You may obtain a copy of
  * the License at http://www.mozilla.org/MPL/
  * 
  * Software distributed under the License is distributed on an "AS
  * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
  * implied. See the License for the specific language governing
  * rights and limitations under the License.
  * 
  * The Original Code in this file as it was released publicly upon
  * October 30, 2004 after being tested for several months by a 
  * bunch of CSS co-workers. Thanks guys ;)
  * 
  * The Initial Developer of the Original Code is HJ van Rantwijk.
  * all Portions created by HJ van Rantwijk are Copyright (C) 2004-2008
  * by HJ van Rantwijk. All Rights Reserved.
  * 
  * Contributors:
  *   Michael Vincent van Rantwijk <mv_van_rantwijk@yahoo.com>
  *
  */

const NO_ROWS_AVAILABLE  = -1;
const NO_ROWS_SELECTED   = 0;
const SINGLE_ROW_SELECTED    = 1;
const MULTIPLE_ROWS_SELECTED = 2;

const CM_SELECT_ALL   = 0;
const CM_SELECT_NONE  = 1;

const CM_MARK_AS_READ     = 6;
const CM_MARK_ALL_AS_READ = 7;
const CM_MARK_AS_NEW  = 8;
const CM_MARK_ALL_AS_NEW  = 9;

var gFeedTree = null;
var gArticleTree = null;
var gFeedDataSource = null;
var gUpdateQueue = null;
var gPendingUpdates = null;
var gBundle = null;
var gInfoPopup = null;
var gArticleTreeBox = null;
var gContextMenu = null;
var gShowInfoPopup = true;

const Cc = Components.classes;
const Ci = Components.interfaces;
const nsIWebFeedService = new Components.Constructor("@multizilla.org/webfeed-service;1", "nsIWebFeedService");

var mzWebFeedService = new nsIWebFeedService();
  
function initFeedPanel()
{
  gFeedDataSource = mzGetNewOrCurrentDSByFilename("mzWebFeeds.rdf", false, null, false);
  gBundle = document.getElementById("multizilla-properties");

  gFeedTree = document.getElementById("FeedTree");
  gFeedTree.database.AddDataSource(gFeedDataSource);
  gFeedTree.builder.rebuild();
  gFeedTree.database.AddObserver(feedTreeRDFObserver);
  gFeedTree.builder.addObserver(feedTreeBuilderObserver);

  var categoryMenu = document.getElementById("categoryMenu");
  categoryMenu.database.AddDataSource(gFeedDataSource);
  categoryMenu.builder.rebuild();

  gArticleTree = document.getElementById("ArticleTree");
  gArticleTree.database.AddDataSource(gFeedDataSource);
  gArticleTree.builder.rebuild();
  gArticleTreeBox = document.getElementById("ArticleTreeBox");
  gInfoPopup = document.getElementById("InfoPopup");

  restoreLastActiveChannel();

  var observerService = Cc["@mozilla.org/observer-service;1"].getService(Ci.nsIObserverService);
  observerService.addObserver(onLinkVisited, "link-clicked", false);

  var navigatorWindow = top.document.defaultView;
  navigatorWindow.mzFeedHandler.infoHeaderLength = navigatorWindow.gMultiZilla.prefs.readInteger("multizilla.sidebar.feed-panel.info-header.length", 250);
  gShowInfoPopup = navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.sidebar.feed-panel.info-popup.show", true);
}

function initContextMenu(aEvent)
{
  gArticleTree.infoRow = -1;

  var tree = aEvent.originalTarget.parentNode;
  var treeState = getCurrentTreeState(tree);
  var menuitems = aEvent.originalTarget.childNodes;

  const NOTHING_AVAILABLE = (treeState == NO_ROWS_AVAILABLE || treeState == NO_ROWS_SELECTED);

  for (var i = 0; i < menuitems.length; i++) {
    menuitems[i].removeAttribute("disabled");
  }
  document.getElementById(tree.id + "-selectAll").setAttribute("disabled", NOTHING_AVAILABLE);
  document.getElementById(tree.id + "-undoSelection").setAttribute("disabled", (treeState != MULTIPLE_ROWS_SELECTED));

  if (tree.id == "FeedTree") {
    var resource = (tree.currentIndex >= 0) ? tree.builderView.getResourceAtIndex(tree.currentIndex).Value : null;
    var isFeedGroup = (resource && /^urn:feedgroup:/.test(resource));

    document.getElementById("show").setAttribute("disabled", (isFeedGroup || NOTHING_AVAILABLE));

    var menuItemAdd = document.getElementById("add")
    menuItemAdd.label = gBundle.getString("addFeed");
    // menuItemAdd.setAttribute("disabled", (treeState != SINGLE_ROW_SELECTED));
    menuItemAdd.removeAttribute("category");

    document.getElementById("renameCategory").setAttribute("disabled", !isFeedGroup);
    document.getElementById("removeCategory").setAttribute("disabled", !isFeedGroup);
    document.getElementById("refresh").setAttribute("disabled", NOTHING_AVAILABLE);
    document.getElementById("refreshAll").setAttribute("disabled", NOTHING_AVAILABLE);
    document.getElementById("delete").setAttribute("disabled", (isFeedGroup || NOTHING_AVAILABLE));
      
    if (tree.treeBoxObject.view.selection.count == 1) {
      var rowIndex = isFeedGroup ? tree.currentIndex : tree.treeBoxObject.view.getParentIndex(tree.currentIndex);

      if (rowIndex != -1) {
        var category = _getCellText(gFeedTree, "Name", rowIndex);
        menuItemAdd.setAttribute("category", resource);
        menuItemAdd.label = gBundle.getString("addFeedInCategory").replace("%S", category);
      }
    }
    var noCategories = true;
    var categoryMenu = document.getElementById("categoryMenu");

    if (!NOTHING_AVAILABLE) {
      categoryMenu.builder.rebuild();
      var feedCategories = mzWebFeedService.getFeedCategories();
      // dump("\nfeedCategories: " + feedCategories);
      noCategories = (feedCategories == null || feedCategories.length == 0);
    }
    categoryMenu.setAttribute("disabled", (NOTHING_AVAILABLE || noCategories || isFeedGroup));

    var disabled = resource ? /^urn:feedgroup:/.test(resource) : true;
    document.getElementById(tree.id + "-properties").setAttribute("disabled", (disabled || treeState != SINGLE_ROW_SELECTED));
  }
  else if (tree.id == "ArticleTree") {
    document.getElementById("openInTab").setAttribute("disabled", NOTHING_AVAILABLE);
    document.getElementById("openInWindow").setAttribute("disabled", NOTHING_AVAILABLE);

    var row = getRowFromEvent(tree, aEvent);
    var resource = (tree.currentIndex >= 0) ? tree.builderView.getResourceAtIndex(tree.currentIndex) : "";

    if (resource && mzWebFeedService.getVisitedState(resource)) {
      menuitems[CM_MARK_AS_READ].setAttribute("hidden", "true");
      menuitems[CM_MARK_ALL_AS_READ].setAttribute("hidden", "true");

      menuitems[CM_MARK_AS_NEW].removeAttribute("hidden");
      menuitems[CM_MARK_ALL_AS_NEW].removeAttribute("hidden");

      menuitems[CM_MARK_AS_NEW].setAttribute("disabled", (treeState != SINGLE_ROW_SELECTED));
      menuitems[CM_MARK_ALL_AS_NEW].setAttribute("disabled", (treeState == NO_ROWS_AVAILABLE));
    }
    else {
      menuitems[CM_MARK_AS_NEW].setAttribute("hidden", "true");
      menuitems[CM_MARK_ALL_AS_NEW].setAttribute("hidden", "true");

      menuitems[CM_MARK_AS_READ].removeAttribute("hidden");
      menuitems[CM_MARK_ALL_AS_READ].removeAttribute("hidden");

      menuitems[CM_MARK_AS_READ].setAttribute("disabled", (treeState != SINGLE_ROW_SELECTED));
      menuitems[CM_MARK_ALL_AS_READ].setAttribute("disabled", (treeState == NO_ROWS_AVAILABLE));
    }
    document.getElementById(tree.id + "-properties").setAttribute("disabled", "true");
  }
}

function onTreeSelect(aEvent)
{
  var tree = aEvent.target;

  if (tree.id == "FeedTree") {
    if (gFeedTree.currentIndex >= 0) {
      var ref = gFeedTree.builderView.getResourceAtIndex(gFeedTree.currentIndex);

      if (ref) {
        ref = ref.Value;

        if (!/^urn:feedgroup:/.test(ref)) {
          gArticleTree.ref = ref;
          // refreshVisitedStateFor(gArticleTree.ref); // XXX: removed (CPU spikes up)!
          var navigatorWindow = top.document.defaultView;
          navigatorWindow.gMultiZilla.prefs.writeString("multizilla.sidebar.feed-panel.last-channel", gArticleTree.ref);
        }
        else
          gArticleTree.ref = null;
      }
    }
  }
}

function onTreeClick(aEvent)
{
  if (aEvent.button == 2 || aEvent.ctrlKey || aEvent.originalTarget.localName != "treechildren")
    return;
 
  var tree = aEvent.target.parentNode;
  var row = getRowFromEvent(tree, aEvent);

  if (row >= 0) {
    if (tree.id == "ArticleTree") {
      var target = (aEvent.ctrlKey || aEvent.button == 1) ? "newtab" : "current";

      if (tree.currentIndex != row) {
        tree.currentIndex = row;
        tree.view.selection.clearSelection();
        tree.view.selection.select(tree.currentIndex);        
      }
      loadArticle(row, target);
    }
    else if (aEvent.button == 1) {
      executeCommand("show", aEvent);
    }
  }
}

function onTreeDBLClick(aEvent)
{
  aEvent.stopPropagation();

  if (aEvent.button != 0 || aEvent.originalTarget.localName != "treechildren")
    return;

  var tree = aEvent.target.parentNode;
  var rowIndex = getRowFromEvent(tree, aEvent);

  if (rowIndex >= 0) {
    if (/^urn:feedgroup:/.test(tree.treeBoxObject.view.getResourceAtIndex(rowIndex).Value))
      return;
    if (tree.id == "FeedTree") {
      tree.view.selection.clearSelection();
      tree.view.selection.select(rowIndex);
      setTimeout(executeCommand, 50, "show", aEvent);
    }
  }
}

function onTreeCommand(aEvent)
{
  var tree = aEvent.target;

  if (tree.id == "ArticleTree") {
    var target = (aEvent.ctrlKey) ? "newtab" : "current";
    loadArticle(gArticleTree.currentIndex, target);
  }
  else {
    var rowIndex = tree.currentIndex;

    if (rowIndex >= 0) {
      if (/^urn:feedgroup:/.test(tree.treeBoxObject.view.getResourceAtIndex(rowIndex).Value))
        return;
      executeCommand("show", aEvent);
    }
  }
}

function loadArticle(aRow, aTarget)
{
  var url = _getCellText(gArticleTree, "link", aRow);
  var navigatorWindow = top.document.defaultView;
  var useAboutBlankFirst = navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.tabs.blank-tab-first", true);
  var tabbrowser = navigatorWindow.getBrowser();

  if (aTarget == "newtab") {
    if (useAboutBlankFirst)
      var browser = navigatorWindow.gMultiZilla.tabs.checkForBlankTab(tabbrowser);

    if (useAboutBlankFirst && browser)
      browser.loadURI(url);
    else
      tabbrowser.addTab(url);
  }
  else if (aTarget == "window")
    window.open(url);
  else
    tabbrowser.loadURI(url);

  var resource = gArticleTree.builderView.getResourceAtIndex(aRow);
  mzWebFeedService.setVisitedState(resource, "visited");
  mzWebFeedService.hasNewArticles(resource, true, null);
  setTimeout("gContextMenu = null;", 50);
}

function doMenuCommand(aEvent)
{
  var menuID = aEvent.originalTarget.id;

  executeCommand(menuID, aEvent);
}

function executeCommand(aMenuID, aEvent)
{
  var tree, resourceList, resource, tabbrowser, browser, url;
  var navigatorWindow = top.document.defaultView;
  var useAboutBlankFirst = navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.tabs.blank-tab-first", true);

  if (aMenuID == "delete" && aEvent == null && 
      document.commandDispatcher.focusedElement == gFeedTree)
    tree = gFeedTree;
  else
    tree = (aMenuID.match(/portOPML$/)) ? gFeedTree : aEvent.target;

  while (tree.localName != "tree")
    tree = tree.parentNode;

  switch(aMenuID.replace(/^\w*-/,''))
  {
     case "select-all":
     case "selectAll":
          selectRows(tree, "select-all");
          break;
     case "select-none":
     case "undoSelection":
          selectRows(tree, "select-none");
          break;
     case "add":
          url = navigatorWindow.readFromClipboard();

          if (!url)
            url = "http://";
          else if(!url.match(/^https?:\/\/\w*.\w*/))
            url = "http://" + url.replace(/\s/g, "");

          var currentResource, category;

          if (gFeedTree.currentIndex != -1)
            currentResource = gFeedTree.builderView.getResourceAtIndex(gFeedTree.currentIndex);

          if (aEvent.originalTarget.hasAttribute("category"))
            category = aEvent.originalTarget.getAttribute("category")

          openAddFeedWindow(url, category, currentResource);
          break;
     case "show":
          tabbrowser = navigatorWindow.getBrowser();
          resourceList = getSelectedResources(tree);
          var isOffline = ('getOfflineStatus' in mzWebFeedService) ? mzWebFeedService.getOfflineStatus() : false;

          for (i in resourceList) {
            if (!/^urn:feedgroup:/.test(resourceList[i].resource)) {
              if (isOffline)
                url = mzWebFeedService.getFeedJarURI(resourceList[i].resource).spec;
              else
                url = _getCellText(gFeedTree, "url", resourceList[i].row);

              if (useAboutBlankFirst)
                browser = navigatorWindow.gMultiZilla.tabs.checkForBlankTab(tabbrowser);

              if (useAboutBlankFirst && browser)
                browser.loadURI(url, url);
              else
                tabbrowser.addTab(url, url);
            }
          }
          break;
     case "addCategory":
     case "renameCategory":
          var categoryResource = null;

          if (gFeedTree.currentIndex != -1)
            categoryResource = gFeedTree.builderView.getResourceAtIndex(gFeedTree.currentIndex);

          openFeedCategoryWindow(aMenuID, categoryResource);
          break;
     case "removeCategory":
          resource = gFeedTree.builderView.getResourceAtIndex(gFeedTree.currentIndex);

          if (resource instanceof Ci.nsIRDFResource && mzWebFeedService.removeFeedCategory(resource, true))
            gFeedTree.builder.rebuild();
          break;
     case "moveFeed":
          /* mzWebFeedService.moveFeed(RDF.GetResource("urn:dannyayers.com:feed-1"), 
                                       RDF.GetResource("urn:feedgroup:#$bPKcq1"), 
                                       RDF.GetResource("urn:feeds:root"), -1); */
          break;
     case "refresh":
          var feedResourceList = new Array();
          var selectedRows = getSelectedRows(tree);

          for (i in selectedRows) {
            var resource = getResourceForRow(tree, selectedRows[i]);

            if (resource && /^urn:feedgroup:/.test(resource)) {
              var resourceList = mzWebFeedService.getFeedsByCategory(RDF.GetResource(resource));

              if (resourceList) {
                resourceList = resourceList.split(',');

                for (i in resourceList) {
                  feedResourceList.push(resourceList[i]);
                }
              }
            }
            else {
              feedResourceList.push(resource);
            }
          }
          if (feedResourceList.length)
            mzWebFeedService.updateFeeds(String(feedResourceList), true);
          break;
     case "refreshAll":
          mzWebFeedService.updateAllFeeds(true);
          break;
     case "delete":
          try {

          resourceList = getSelectedResources(tree);

          if (resourceList.length) {
            var promptService = Cc["@mozilla.org/embedcomp/prompt-service;1"].getService(Ci.nsIPromptService);
            var dataElement = document.getElementById("confirmationText");
            var confirmationTitle = dataElement.getAttribute("title");
            var confirmationText = dataElement.getAttribute("text");

            if (promptService.confirm(window, confirmationTitle, confirmationText)) {
              for (var i = (resourceList.length-1); i >= 0; i--) {
                var row = resourceList[i].row;
                var parentIndex = gFeedTree.builderView.getParentIndex(row);
                var resourceString = resourceList[i].resource;
                var resource = this.RDF.GetResource(resourceString);
                mzWebFeedService.removeFeedFromFeedCache(resource);
                mzRemoveResources(gFeedDataSource, resourceString);
                mzRemoveDataResources(gFeedDataSource, resourceString);

                if (parentIndex == -1)
                  mzWebFeedService.removeFeed(this.RDF.GetResource("urn:feeds:root"), resource);
                else {
                  var parentResource = gFeedTree.builderView.getResourceAtIndex(parentIndex);
                  mzWebFeedService.removeFeed(parentResource, resource);
                }
              }
              tree.builder.rebuild();
            }
          }

          } catch(ex) { dump("\ndelete: " + ex); }
          break;
     case "openInTab":
          resourceList = getSelectedResources(tree);

          for (i in resourceList) {
            loadArticle(resourceList[i].row, "newtab");
          }
          break;
     case "openInWindow":
          resourceList = getSelectedResources(tree);

          if (resourceList.length == 1) {
            loadArticle(resourceList[0].row, "window");
          }
          else {
            tabbrowser = openWindowAndReturnBrowser();

            for (i in resourceList) {
              url = _getCellText(gArticleTree, "link", resourceList[i].row);

              if (useAboutBlankFirst)
                browser = navigatorWindow.gMultiZilla.tabs.checkForBlankTab(tabbrowser);

              if (useAboutBlankFirst && browser)
                browser.loadURI(url);
              else
                tabbrowser.addTab(url);
              mzWebFeedService.setVisitedState(RDF.GetResource(resourceList[i].resource), "visited");
            }
          }
          break;
     case "markAllAsRead":
     case "markAllAsNew":
     case "markAsRead":
     case "markAsNew":
          document.getElementById(aMenuID).setAttribute("batch", "true");

          function batch() {
            var startTime = new Date().getTime();
            var selectedCategory = gFeedTree.currentIndex;

            if (aMenuID.match(/^markAll/)) {
              if (getCurrentTreeState(tree) == SINGLE_ROW_SELECTED)
                selectRows(tree, "select-all");
            }
            var newState = (aMenuID.search(/AsNew/) > 0) ? "unread" : "visited";

            resourceList = getSelectedResources(tree);

            var globalHistoryCc, globalHistory, addURI = false;
            // Builds 20040210 and up (see also mozilla bug 224829) 
            if ("@mozilla.org/browser/global-history;2" in Cc) {
              globalHistoryCc = Cc["@mozilla.org/browser/global-history;2"];
              globalHistory = globalHistoryCc.getService(Ci.nsIGlobalHistory2);
              addURI = true;
            }
            else  {
              globalHistoryCc = Cc["@mozilla.org/browser/global-history;1"];
              globalHistory = globalHistoryCc.getService(Ci.nsIGlobalHistory);
            }
            var browserHistory = globalHistoryCc.getService(Ci.nsIBrowserHistory);
            var _nsIIOService = Cc["@mozilla.org/network/io-service;1"].getService(Ci.nsIIOService);
            gFeedDataSource.beginUpdateBatch();

            for (i in resourceList) {
              var rowState = _getCellText(gArticleTree, "state", resourceList[i].row);

              if (rowState != newState) {
                var url = _getCellText(gArticleTree, "link", resourceList[i].row);

                if (url) {
                  mzWebFeedService.setVisitedState(RDF.GetResource(resourceList[i].resource), newState);
                  var uri = _nsIIOService.newURI(url, null, null);
                  var isVisited = globalHistory.isVisited(uri);

                  if (newState == "visited" && isVisited == false) {
                    if (addURI)
                      globalHistory.addURI(uri, false, true, null); // nsIGlobalHistory2.idl
                    else
                      globalHistory.addPage(uri, null); // nsIGlobalHistory.idl
                  }
                  else if (newState == "unread" && isVisited)
                    browserHistory.removePage(uri);
                }
              }
            }
            resource = resourceList[i].resource.replace(/:article-\d*/, '');

            if (aMenuID.match(/^markAll/)) {
              newState = aMenuID.match(/AsNew$/) ? "new" : "";
              // dump("\ncalling setVisitedState() - newState: " + newState);
              mzWebFeedService.setVisitedState(RDF.GetResource(resource), newState);
            }
            else {
              // dump("\ncalling hasNewArticles() " + resource);
              mzWebFeedService.hasNewArticles(RDF.GetResource(resource), true, null);
            }
            gFeedDataSource.endUpdateBatch();
            selectRows(tree, "select-none");

            if (selectedCategory == -1)
              selectedCategory = getRowForResource(gFeedTree, gArticleTree.ref);

            gFeedTree.view.selection.select(selectedCategory);
            var endTime = new Date().getTime();
            dump("\n" + aMenuID + " done in: " + (endTime - startTime));
          }
          setTimeout(batch, 0);
          break;
     case "properties":
          resource = gFeedTree.builderView.getResourceAtIndex(gFeedTree.currentIndex);
          openPropertyWindow(resource.Value);
          break;
     case "exportOPML":
          mzWebFeedService.exportOPMLData(false);
          break;
     case "importOPML":
          mzWebFeedService.importOPMLData(false);
          break;
  }
}

function openAddFeedWindow(aURL, aCategory, aTargetResource)
{
  var navigatorWindow = top.document.defaultView;

  if (aURL && mzWebFeedService.isNewFeedURL(aURL) == false) {
    var promptService = Cc["@mozilla.org/embedcomp/prompt-service;1"]
                         .getService(Ci.nsIPromptService);
    var title = gBundle.getString("duplicatedFeedAlertTitle");
    var text = gBundle.getString("duplicatedFeedAlertText");
    promptService.alert(window, title, text);
  }
  else {
    var windowStyle = navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.classic-style", false);
    var isMac = /Mac/.test(navigator.platform) ||
                navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.mac-style", false);
    var isLinux = /Linux/.test(navigator.platform) ||
                navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.linux-style", false);

    window.openDialog("chrome://multiviews/content/feedpanel/miscellaneous/addFeedWindow.xul", 
                      "", "chrome, dependent, modal", window, navigatorWindow,
                      aURL, aCategory, aTargetResource, windowStyle, isMac, false);
  }
}

function openPropertyWindow(aResource)
{
  var navigatorWindow = top.document.defaultView;
  var propertyWindow = navigatorWindow.gMultiZilla.utility._getMostRecentWindow("multizilla:feedviewer-properties", false);

  if (propertyWindow) {
    propertyWindow.focus();
  }
  else {
    var windowStyling = navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.classic-style", false);
    var isMac = /Mac/.test(navigator.platform) ||
                navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.mac-style", false);
    var isLinux = /Linux/.test(navigator.platform) ||
                navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.linux-style", false);

    window.openDialog("chrome://multiviews/content/feedpanel/properties/propertiesWindow.xul", 
                      "", "chrome, dependent, modal", navigatorWindow,
                      gFeedDataSource, aResource, windowStyling, isMac);
  }
}

function openFeedCategoryWindow(aMenuID, aCategoryResource)
{
  var categoryName = (aMenuID == "addCategory") ? null :
                                                  mzWebFeedService.getFeedCategoryName(aCategoryResource);
  var navigatorWindow = top.document.defaultView;
  var windowStyle = navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.classic-style", false);
  var isMac = /Mac/.test(navigator.platform) ||
              navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.mac-style", false);
  var isLinux = /Linux/.test(navigator.platform) ||
              navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.windows.linux-style", false);

  window.openDialog("chrome://multiviews/content/feedpanel/miscellaneous/feedCategoryWindow.xul", 
                    "", "chrome, dependent, modal", aCategoryResource, categoryName,
                    { isClassicStyle: windowStyle, 
                      isMac: isMac ,
                      isMac: isLinux
                    } );
  gFeedTree.builder.rebuild();
}


function selectRows(aTree, aCommand)
{
  switch (aCommand) {
    case "select-all": // Called from the context menu, or Ctrl+A ?
      aTree.view.selection.selectAll();
      break;
    case "select-none": // Called from the context menu, or Ctrl+Z ?
      aTree.view.selection.clearSelection();
      aTree.view.selection.select(aTree.currentIndex);
      aTree.treeBoxObject.ensureRowIsVisible(aTree.currentIndex);
      break;
  }
  aTree.focus();
}

function refreshVisitedStateFor(aResource, aURL)
{
  // dump("\nrefreshVisitedStateFor - aResource: " + aResource)

  if (!aResource) // Prevent JS Exception from empty feed panel
    return;

  var resource = RDF.GetResource(aResource);
  var articles = mzGetResourcesFromContainer(gFeedDataSource, resource);

  if (articles.length) {
    var linkProperty = RDF.GetResource("http://multizilla.mozdev.org/rdf#link");
    var stateProperty = RDF.GetResource("http://multizilla.mozdev.org/rdf#state");
    var checkAgainstHistory = (aURL == undefined);

    if (checkAgainstHistory) {
      var navigatorWindow = top.document.defaultView;
      var _nsIIOService = Cc["@mozilla.org/network/io-service;1"].getService(Ci.nsIIOService);
      var _nsIGlobalHistory;
      // Builds 20040210 and up (see also mozilla bug 224829)
      if ("@mozilla.org/browser/global-history;2" in Cc) {
        _nsIGlobalHistory = Cc["@mozilla.org/browser/global-history;2"].getService(Ci.nsIGlobalHistory2);
      }
      else {
        _nsIGlobalHistory = Cc["@mozilla.org/browser/global-history;1"].getService(Ci.nsIGlobalHistory);
      }
    }
    for (i in articles) {
      if (checkAgainstHistory) {
        /* var stateResource = gFeedDataSource.GetTarget(RDF.GetResource(articles[i]), stateProperty, true);

        if (stateResource instanceof Ci.nsIRDFNode) {
          if (stateResource.QueryInterface(Ci.nsIRDFLiteral).Value == "unread")
            continue;
        } */
        if (gFeedDataSource.HasAssertion(RDF.GetResource(articles[i]), stateProperty, this.RDF.GetLiteral("unread"), true))
          continue;
      }
      var linkResource = gFeedDataSource.GetTarget(RDF.GetResource(articles[i]), linkProperty, true);
      
      if (linkResource instanceof Ci.nsIRDFNode) {
        var url = linkResource.QueryInterface(Ci.nsIRDFLiteral).Value;

        if (url) {
          if (checkAgainstHistory) {
            var uri = _nsIIOService.newURI(url, null, null);
            aURL = _nsIGlobalHistory.isVisited(uri) ? url : "";
          }
          if (url == aURL) {
            // mzAddOrChangeProperty(gFeedDataSource, articles[i], "state", "visited");
            mzWebFeedService.setVisitedState(RDF.GetResource(articles[i]), "visited");
            mzWebFeedService.hasNewArticles(RDF.GetResource(articles[i]), false, null);

            if (!checkAgainstHistory)
              break;
          }
        }
      }
    }
    // mzWebFeedService.hasNewArticles(aResource, true, null);
  }
}

function onXMLProgress(aEvent)
{
  // dump("\nposition : " + aEvent.position +
       // "\ntotalSize: " + aEvent.totalSize);
}

function restoreLastActiveChannel()
{
  if (getCurrentTreeState(gFeedTree) != NO_ROWS_AVAILABLE) {
    var navigatorWindow = top.document.defaultView;
    var lastChannel = navigatorWindow.gMultiZilla.prefs.readString("multizilla.sidebar.feed-panel.last-channel", "");

    if (lastChannel) {
      gArticleTree.ref = lastChannel;

      for (var i = 0; i < gFeedTree.treeBoxObject.view.rowCount; i++) {
        var resource = getResourceForRow(gFeedTree, i);

        if (resource == lastChannel) {
          gFeedTree.view.selection.select(i);
          gFeedTree.treeBoxObject.ensureRowIsVisible(i);
          // refreshVisitedStateFor(resource);
          break;
        }
      }
    }
  }
}

function isTreeSorted(aTree)
{
  // this will fail in older Mozilla versions/builds!
  var column = aTree.columns.getSortedColumn();

  if (column) {
    var treecol = document.getElementById(column.id);

    if (treecol.hasAttribute("sortActive") && treecol.getAttribute("sortDirection") != "natural")
      return true;
  }
  return false;
}

function cleanupMoveFeedToCategory(aMenuPopup)
{
  // menuitem, menuseparator, template
  for (var i = (aMenuPopup.childNodes.length - 1); i > 0; i--) {
    if (aMenuPopup.childNodes[i].hasAttribute("injected"))
      aMenuPopup.removeChild(aMenuPopup.childNodes[i]);
  }
}

function initMoveFeedToCategory(aMenuPopup)
{
  var row = gFeedTree.currentIndex;
  var disabled = (row == -1);
  var parentIndex = disabled ? -1 : gFeedTree.builderView.getParentIndex(row);

  cleanupMoveFeedToCategory(aMenuPopup);

  if (parentIndex == -1) {
    if (!disabled)
      disabled = true;
  }
  else {
    var parentResource = gFeedTree.builderView.getResourceAtIndex(parentIndex).Value;

    for (var i = 1; i < aMenuPopup.childNodes.length; i++) {
      var menuitem = aMenuPopup.childNodes[i];

      if (menuitem.id == parentResource) {
        menuitem.setAttribute("disabled", "true");
        menuitem.setAttribute("checked", "true");
      }
      else {
        menuitem.removeAttribute("disabled");
        menuitem.removeAttribute("checked");
      }
    }
  }
  aMenuPopup.firstChild.setAttribute("disabled", disabled);
  aMenuPopup.firstChild.setAttribute("checked", disabled);

  var menuseparator = document.createElementNS("http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul", "menuseparator");
  menuseparator.setAttribute("injected", "true");
  aMenuPopup.appendChild(menuseparator);
  var menuitem = document.createElementNS("http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul", "menuitem");
  menuitem.setAttribute("injected", "true");
  menuitem.setAttribute("id", "MoveToNewCategory");
  menuitem.setAttribute("label", gBundle.getString("moveToNewCategory.label")); // settings .label breaks!
  menuitem.setAttribute("accesskey", gBundle.getString("moveToNewCategory.accesskey"));
  aMenuPopup.appendChild(menuitem);
}

function moveFeedToCategory(aTargetCategory)
{
  var resourceList = getSelectedResources(gFeedTree);

  if (aTargetCategory == "MoveToNewCategory") {
    var newCategoryResource = mzWebFeedService.addFeedCategory("", null); // appends category "New Feed Group"
    openFeedCategoryWindow("renameCategory", RDF.GetResource(newCategoryResource));// gives the user the opportunity to rename the category
    aTargetCategory = newCategoryResource;
  }
  for (i in resourceList) {
    var resource = resourceList[i].resource;

    if (/^urn:feedgroup:/.test(resource)) // prevent categories from being moved!
      continue;

    var row = resourceList[i].row;
    var parentIndex = gFeedTree.builderView.getParentIndex(row);
    var currentCategory = (parentIndex == -1) ? "urn:feeds:root" : gFeedTree.builderView.getResourceAtIndex(parentIndex).Value;

    if (currentCategory == aTargetCategory) // prevent moves from/to the same category!
      continue;

    /* dump("\n\nmoveFeedToCategory: " + resource +
         "\nfrom category     : " + currentCategory +
         "\nto category       : " + aCategory); */
    mzWebFeedService.moveFeed(RDF.GetResource(resource), 
                              RDF.GetResource(currentCategory), 
                              RDF.GetResource(aTargetCategory), -1);
  }
}


function getResourceForRow(aTree, aRowNumber)
{
  return (aRowNumber > -1) ? aTree.builderView.getResourceAtIndex(aRowNumber).Value : null;
}

function getRowForResource(aTree, aResourceString)
{
  var rowCount = aTree.treeBoxObject.view.rowCount;

  for (var row = 0; row < rowCount; row++) {
    if (aTree.builderView.getResourceAtIndex(row).Value == aResourceString)
      return row;
  }
  return -1;
}

function getSelectedResources(aTree)
{
  // dump("\ngetSelectedResources - tree: " + aTree);
  var resources = new Array();
  var rows = getSelectedRows(aTree);

  if (aTree.id == "FeedTree" ) { // channel/url and description
    for (i in rows) {
      var resource = getResourceForRow(aTree, rows[i]);
      dump("\nresource: " + resource);
      if (resource && /^urn:feedgroup:/.test(resource)) {
        var resourceList = mzWebFeedService.getFeedsByCategory(RDF.GetResource(resource));
        dump("\nresourceList: " + resourceList);

        if (resourceList)
          resourceList = resourceList.split(',');

        for (i in resourceList) {
          resources[i] = { row: getRowForResource(aTree, resourceList[i]), 
                           resource: resourceList[i],
                           parentResource: resource
                         };
        }
      }
      else {
        resources[i] = { row: rows[i], 
                         resource: getResourceForRow(aTree, rows[i])
                       };
      }
    }
  }
  else { // title/link and prettydate
    for (i in rows) {
      resources[i] = { row: rows[i], 
                       resource: getResourceForRow(aTree, rows[i]),
                       url: _getCellText(aTree, "link", rows[i])
                     };
    }
  }
  if (resources)
    return resources;
  return null;
}

function getSelectedRows(aTree)
{
  var rows = -1;
  var selectedRows = new Array();
  // Get and return all selection ranges
  var rangeCount = aTree.builderView.selection.getRangeCount();
  // Iterate over all ranges to collect the selected rows
  for (var range = 0; range < rangeCount; range++) {
    var startSelection = {};
    var endSelection = {};
    aTree.builderView.selection.getRangeAt(range, startSelection, endSelection);
    // Collect all rows and store them in our array
    for (var row = startSelection.value; row <= endSelection.value; row++) {
      // Check for invalid row numbers
      if (startSelection.value != -1 || endSelection.value != -1)
        selectedRows[++rows] = row;
    }
  }
  if (selectedRows)
    return selectedRows;
  return null;
}

function getCurrentTreeState(aTree)
{
  if (aTree.treeBoxObject.view.rowCount < 1)
    return NO_ROWS_AVAILABLE;

  var selectedRows = getSelectedRows(aTree).length;

  if (selectedRows > 1)
    return MULTIPLE_ROWS_SELECTED;
  else if (selectedRows == 1)
    return SINGLE_ROW_SELECTED;
  return NO_ROWS_SELECTED;
}

function _getCellText(aTree, aColumnID, aRow)
{
  aColumnID = ('columns' in aTree) ? aTree.columns[aColumnID] : aColumnID;
  return aTree.view.getCellText(aRow, aColumnID);
}

function getRowFromEvent(aTree, aEvent)
{
  var row = new Object();
  var col = new Object();
  var obj = new Object();

  aTree.treeBoxObject.getCellAt(aEvent.clientX, aEvent.clientY, row, col, obj);
  return row.value;
}

function shutdownFeedPanel()
{
  var observerService = Cc["@mozilla.org/observer-service;1"].getService(Ci.nsIObserverService);
  observerService.removeObserver(onLinkVisited, "link-clicked");
  gFeedTree.database.RemoveObserver(feedTreeRDFObserver);
  gFeedTree.builder.removeObserver(feedTreeBuilderObserver);
}

var onLinkVisited =
{
  observe: function(subject, topic, state)
  {
    if (topic == "link-clicked") {
      dump("\nlink-clicked: " + state);
      refreshVisitedStateFor(gArticleTree.ref, state); // 'state' represents the url
    }
  }
};

function selectFeedAfterAdd(aResource)
{
  var row = gFeedTree.builderView.getIndexOfResource(aResource);

  if (row >= 0) {
    gFeedTree.currentIndex = row;
    gFeedTree.view.selection.clearSelection();
    gArticleTree.view.selection.clearSelection();
    gFeedTree.view.selection.select(row);
    gFeedTree.treeBoxObject.ensureRowIsVisible(row);
  }
}

function initTreeTooltip(aEvent)
{
  if (gShowInfoPopup && gContextMenu) {
    aEvent.preventDefault();
    return;
  }
  var node = document.tooltipNode;
  var nodeName = node.localName;
  var tooltipNode = aEvent.originalTarget;

  if (nodeName == 'tree' || nodeName.match('menu') || 
      (nodeName == 'treechildren' && tooltipNode.id == "ArticleTreeTooltip" && gShowInfoPopup))
  {
    aEvent.preventDefault();
    return;
  }
  tooltipNode.firstChild.hidden = (!node.hasAttribute('sort') || nodeName != 'treecol');
  tooltipNode.childNodes[1].hidden = (nodeName != 'treecols');
  tooltipNode.childNodes[2].hidden = (nodeName != 'splitter');
  tooltipNode.childNodes[3].hidden = (nodeName != 'treechildren');
}

function showInfoPopup(aEvent)
{
  if (!gShowInfoPopup || (gShowInfoPopup && gContextMenu))
    return;

  var tree, row, targetNode = aEvent.target;

  if (aEvent.type == 'mousemove') {
    if (targetNode.nodeName != 'treechildren')
      return;
    tree = targetNode.parentNode;
    row = getRowFromEvent(tree, aEvent)
  }
  else {
    if (targetNode.nodeName != 'tree')
      return;
    tree = targetNode;
    row = tree.currentIndex;
  }
  if (row >= 0 && row != tree.infoRow) {
    tree.infoRow = row;
    var resource = tree.builderView.getResourceAtIndex(row);
    var mzRDF = "http://multizilla.mozdev.org/rdf#";
    var property = mzRDF + "published";
    var dateResource = gFeedDataSource.GetTarget(resource, RDF.GetResource(property), true);

    if (dateResource instanceof Ci.nsIRDFNode)
      var date = dateResource.QueryInterface(Ci.nsIRDFLiteral).Value;
    date = new Date(Number(date)).toUTCString();

    property = mzRDF + "title";
    var titleResource = gFeedDataSource.GetTarget(resource, RDF.GetResource(property), true);

    if (titleResource instanceof Ci.nsIRDFNode)
      var title = titleResource.QueryInterface(Ci.nsIRDFLiteral).Value;

    property = mzRDF + "infoHeader";
    var infoHeaderResource = gFeedDataSource.GetTarget(resource, RDF.GetResource(property), true);

    if (infoHeaderResource instanceof Ci.nsIRDFNode)
      var infoHeader = infoHeaderResource.QueryInterface(Ci.nsIRDFLiteral).Value;

    var navigatorWindow = top.document.defaultView;
    var showLink = navigatorWindow.gMultiZilla.prefs.readBoolean("multizilla.sidebar.feed-panel.info-header.show-link", true);

    if (showLink) {
      property = mzRDF + "link";
      var linkResource = gFeedDataSource.GetTarget(resource, RDF.GetResource(property), true);

      if (linkResource instanceof Ci.nsIRDFNode)
        var url = linkResource.QueryInterface(Ci.nsIRDFLiteral).Value;
    }
    var popupNode = gInfoPopup.firstChild;
    var dateNode = popupNode.childNodes[0];
    var titleNode = popupNode.childNodes[1];
    var linkNode = popupNode.childNodes[3];
    var descriptionBoxNode = popupNode.childNodes[2];

    if (date)
      dateNode.value = date;
    if (title)
      titleNode.value = title;
    if (url)
      linkNode.value = url;

    if (descriptionBoxNode.hasChildNodes())
      descriptionBoxNode.removeChild(descriptionBoxNode.lastChild);

    if (infoHeader) {
      var descriptionElement = document.createElement("description");
      descriptionElement.textContent = infoHeader;
      descriptionElement.className = "infoPopupDescription";
      descriptionBoxNode.appendChild(descriptionElement);

      if (showLink)
        linkNode.className = "info-popup-url-indent";
    }
    else if (showLink) {
      linkNode.className = "info-popup-url";
    }
    dateNode.hidden = (!date || date == "Invalid Date");
    titleNode.hidden = (!title);
    linkNode.hidden = (!url || !showLink);
    /***
      * The following lines are a hack to prevent the info popup from jumping to the 
      * wrong position. Note that it will still fail the very first time only!
      */
    var contextMenu = document.getElementById("ArticleTreeContextMenu");

    if (document.popupNode != contextMenu)
      document.popupNode = contextMenu;

    // Don't check with if (Components.interfaces.nsIPopupBoxObject.number == "{8714441F-0E24-4EB5-BE58-905F2854B4EB}") {
    // to prevent (near) future errors introduced by API changes!
    if ('openPopupAtScreen' in gInfoPopup) {
      gInfoPopup.openPopupAtScreen((gFeedTree.parentNode.boxObject.screenX - 2), 
                                   (gFeedTree.parentNode.boxObject.screenY - 2), 
                                   false);
    }
    else {
      gInfoPopup.showPopup(gInfoPopup,
                           (gFeedTree.boxObject.x - 2), 
                           (gFeedTree.boxObject.y - 2), 
                           "popup");
    }
    var arrowscrollbox = document.getAnonymousNodes(gInfoPopup)[0];

    if (popupNode.boxObject.height > 0) {
      arrowscrollbox.height = popupNode.boxObject.height;
      return;
    }
    window.setTimeout(function _inner(){arrowscrollbox.height = popupNode.boxObject.height;}, 0);
  }
}

function hideInfoPopup(aEvent)
{
  var nodeName = aEvent.originalTarget.localName;
  var scrolling = (nodeName == 'thumb' || nodeName == 'slider' || nodeName.match(/scroll/) != null);
  var sorting = (nodeName == 'treecol');
  var columnPicker = (nodeName == 'treecolpicker' || nodeName == 'menuitem');

  gArticleTree.infoRow = -1;

  if (aEvent.type == 'blur' || (scrolling && aEvent.type != 'mouseout'))
    gContextMenu = null;
  else if (aEvent.type == 'menupopup')
    gContextMenu = aEvent.originalTarget;
  else if (aEvent.type == 'mousedown' && !sorting && !columnPicker || aEvent.type == 'keydown')
    gContextMenu = -1;

  gInfoPopup.hidePopup();
}

var feedTreeRDFObserver =
{
  onAssert: function(aDataSource, aSource, aProperty, aTarget)
  {
    /* if (aSource.Value == "urn:feeds:root") {
      if (aProperty.Value.match(/#_\d* /)) {
        setTimeout(selectFeedAfterAdd, 0, aTarget);
      }
    } */
  },

  onEndUpdateBatch: function()
  {
    // setTimeout(gFeedTree.builder.rebuild(), 0);
    // gFeedTree.builder.refresh();
  },

  onUnassert: function(aDataSource, aSource, aProperty, aTarget) {},
  onMove: function(aDataSource, aOldTarget, aNewTarget, aProperty, aTarget) {},
  onChange: function(aDataSource, aSource, aProperty, aOldTarget, aNewTarget) {},
  onBeginUpdateBatch: function() {}
}

var feedTreeDNDObserver =
{
  onDragStart: function(aEvent, aXferData, aDragSession)
  {
    try {

    if (gFeedTree.treeBoxObject.view.rowCount > 0) {
      aDragSession.action = Ci.nsIDragService.DRAGDROP_ACTION_MOVE;
      var row = gFeedTree.treeBoxObject.getRowAt(aEvent.pageX, aEvent.pageY);

      if (row == -1)
        return;

      var treeHasSortOrder = isTreeSorted(gFeedTree);
      var resource = getResourceForRow(gFeedTree, row);
      var isFeedGroup = /^urn:feedgroup:/.test(resource);
      var parentIndex = gFeedTree.treeBoxObject.view.getParentIndex(row);

      var draggedChannel = new Object();
      draggedChannel.row = row;
      draggedChannel.resource = resource;

      if (isFeedGroup) {
        var container = mzConstructRDFContainer(gFeedDataSource, "seq", resource, false);
        draggedChannel.numberOfFeeds = container.GetCount();
        container = null;
      }
      else {
        draggedChannel.numberOfFeeds = 0;
      }
      draggedChannel.parentIndex = parentIndex;
      draggedChannel.isFeedGroup = isFeedGroup;
      draggedChannel.parentResource = (parentIndex == -1) ? "urn:feeds:root" : getResourceForRow(gFeedTree, parentIndex);

      feedTreeBuilderObserver.draggedObject = draggedChannel;
      feedTreeBuilderObserver.isTreeSorted = treeHasSortOrder;
      feedTreeBuilderObserver.dragSession = aDragSession;

      aXferData.data = new TransferData();
      aXferData.data.addDataForFlavour("text/x-moz-url", _getCellText(gFeedTree, "url", row));
    }

    } catch(ex) { dump("\nex(onDragStart): " + ex); }
  },

  onDragOver: function(aEvent, aXferData, aDragSession)
  {
    // set 'canDrop' when the user drags over an empty area of the tree
    var row = { }, col = { }, child = { };
    gFeedTree.treeBoxObject.getCellAt(aEvent.clientX, aEvent.clientY, row, col, child);

    if (row.value == -1) {
      var dragService = Cc["@mozilla.org/widget/dragservice;1"].getService(Ci.nsIDragService);
      var dragSession = dragService.getCurrentSession();
      dragSession.canDrop = feedTreeBuilderObserver.canDrop();
    }
  },

  getSupportedFlavours: function ()
  {
    // this.onDragOver won't be called when you remove this!
    var flavourSet = new FlavourSet();

    flavourSet.appendFlavour("text/x-moz-url");
    flavourSet.appendFlavour("text/unicode");
    flavourSet.appendFlavour("text/html");
    return flavourSet;
  },

  canDrop: function (aEvent, aDragSession)              {},
  onDragExit: function(aEvent, aXferData, aDragSession) {},
  onDrop: function(aEvent, aXferData, aDragSession)     {}
};

var feedTreeBuilderObserver =
{
  DROP_BEFORE : -1, // Ci.nsITreeView.DROP_BEFORE, // XXX: do we still want this?
  DROP_ON     : 0,  // Ci.nsITreeView.DROP_ON, // XXX: do we still want this?
  DROP_AFTER  : 1,  // Ci.nsITreeView.DROP_AFTER, // XXX: do we still want this?

  draggedChannel: null,
  isTreeSorted: false,

  canDrop: function(aRowIndex, orientation)
  {
    dump("\n\ncanDrop - aRowIndex  : " + aRowIndex +
         "\ncanDrop - orientation: " + orientation +
         "\ncanDrop - target row : " + (aRowIndex + orientation));

    if (this.isTreeSorted) { // Block DND when the tree is sorted
      
      return false;
    }
    if (!this.draggedObject) // && aRowIndex >= 0)
      return true;

    if (aRowIndex == this.draggedObject.row || // block drops on the same row, and previous/next siblings
        orientation == 1 && (this.draggedObject.row - 1) == aRowIndex ||
        orientation != 1 &&  (this.draggedObject.row + 1) == aRowIndex)
      return false;

    var isDraggedFeedGroup = /^urn:feedgroup:/.test(this.draggedObject.resource);
    // block useless container dragging
    if (isDraggedFeedGroup && aRowIndex == (this.draggedObject.row + this.draggedObject.numberOfFeeds + 1))
      return false;

    var parentIndex = (aRowIndex == -1) ? -1 : gFeedTree.treeBoxObject.view.getParentIndex(aRowIndex);
    var parentResource = (parentIndex == -1) ? "" : gFeedTree.view.getResourceAtIndex(parentIndex).Value;
    var dropInFeedGroup = /^urn:feedgroup:/.test(parentResource);

    if (/* aRowIndex == this.draggedObject.parentIndex && orientation != -1 || // block dragged row from being dropped on parent container */
        isDraggedFeedGroup && dropInFeedGroup) // block dragged container from being dropped in other containers
      return false;

    var resource = getResourceForRow(gFeedTree, aRowIndex);
    var isFeedGroup = /^urn:feedgroup:/.test(resource);

    if (this.draggedObject.isFeedGroup && isFeedGroup && orientation != -1) // block drop of container on/or under other containers
       return false;

    return true;
  },

  onDrop: function(aRowIndex, orientation)
  {
    /* dump("\n\nonDrop - aRowIndex  : " + aRowIndex +
         "\n\nonDrop - orientation: " + orientation); */

    var parentIndex = /* (aRowIndex == -1) ? -1 : */ gFeedTree.treeBoxObject.view.getParentIndex(aRowIndex);
    var targetIndex = (parentIndex == -1) ? aRowIndex : parentIndex;
    var parentResource = /* (parentIndex == -1) ? "urn:feeds:root" : */ gFeedTree.view.getResourceAtIndex(targetIndex).Value;
    var droppedInFeedGroup = (/^urn:feedgroup:/.test(parentResource) && parentResource != "urn:feeds:root");
    var targetResource = (droppedInFeedGroup) ? parentResource : "urn:feeds:root";

    const RDFC = Cc["@mozilla.org/rdf/container;1"].createInstance(Ci.nsIRDFContainer);	
    RDFC.Init(gFeedDataSource, RDF.GetResource(this.draggedObject.parentResource));
    var sourceIndex = RDFC.IndexOf(RDF.GetResource(this.draggedObject.resource));
    RDFC.Init(gFeedDataSource, RDF.GetResource(targetResource));
    var targetIndex = /* (aRowIndex == -1) ? -1 :*/ RDFC.IndexOf(gFeedTree.view.getResourceAtIndex(aRowIndex));

    if (droppedInFeedGroup && !gFeedTree.treeBoxObject.view.isContainerOpen(aRowIndex))
      gFeedTree.treeBoxObject.view.toggleOpenState(aRowIndex);

    if (orientation == 1) {
      if (targetIndex > 0)
        targetIndex++;
      else if (targetIndex == -1 ||
               gFeedTree.treeBoxObject.view.isContainer(aRowIndex) &&
               gFeedTree.treeBoxObject.view.isContainerOpen(aRowIndex) && 
              !gFeedTree.treeBoxObject.view.isContainerEmpty(aRowIndex))
        targetIndex = 1;
      aRowIndex++;
    }
    if (aRowIndex < targetIndex) { // sourceIndex < targetIndex) {
      targetIndex--;
      aRowIndex--;
    }
    dump("\nFeed       : " + this.draggedObject.resource +
         "\nFrom       : " + this.draggedObject.parentResource +
         "\nTo         : " + targetResource +
         "\ntargetIndex: " + targetIndex);

    if (mzWebFeedService.moveFeed(RDF.GetResource(this.draggedObject.resource), 
                                  RDF.GetResource(this.draggedObject.parentResource), 
                                  RDF.GetResource(targetResource), 
                                  targetIndex)) {
      var row = (droppedInFeedGroup) ? getRowForResource(gFeedTree, this.draggedObject.resource) : aRowIndex;
      gFeedTree.view.selection.select(row);
      gFeedTree.treeBoxObject.ensureRowIsVisible(row);
    }
    this.draggedObject = null;
    // gFeedTree.builder.refresh();
  },

  // Backward compatibility with older Mozilla versions/builds
  canDropOn: function(aRow) // XXX: do we still want this?
  {
    var orientation = this.DROP_ON;
    return this.canDrop(aRow, orientation);
  },
  // Backward compatibility with older Mozilla versions/builds
  canDropBeforeAfter: function(aRow, before) // XXX: do we still want this?
  {
    var orientation = (before) ? this.DROP_BEFORE : this.DROP_AFTER;
    return this.canDrop(aRow, orientation);
  },

  onToggleOpenState    : function (aRow)                           {},
  onCycleHeader        : function (aColumnID, aHeaderElement)      {},
  onSelectionChanged   : function ()                               {},
  onCycleCell          : function (aItemIndex, aColumnID)          {},
  isEditable           : function (aItemIndex, aColumnID)          {},
  onSetCellText        : function (aItemIndex, aColumnID, aValue)  {},
  onPerformAction      : function (aAction)                        {},
  onPerformActionOnRow : function (aAction, aItemIndex)            {},
  onPerformActionOnCell: function (aAction, aItemIndex, aColumnID) {}
};



