/*****************************************************************************
 * Authors: Louis Bavoil <louis@bavoil.net>
 *          Fredrik Hbinette <hubbe@hubbe.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
*****************************************************************************/

#include "mozplugger.h"

#define WINDOW ((Window) windata.window)

#ifndef MIN
#define MIN(X,Y) ((X)<(Y)?(X):(Y))
#endif

Display *display;
char *displayname;
NPWindow windata;
int flags;
int repeats;
char *winname;
char *command;
char *file;
char *mimetype;

static XWindowAttributes wattr;
static Window victim;

#define MAX_IGNORED 1000
static int windows_found;
static Window ignored[MAX_IGNORED];

static int got_sigwinch;

/*****************************************************************************
 * Functions below are called by the child process.
 *****************************************************************************/
void sigwinch(int tmp)
{
     D("GOT SIGWINCH\n");
     got_sigwinch = 1;
}

/*****************************************************************************
 * Called on X11 errors.
 *****************************************************************************/
int error_handler(Display *dpy, XErrorEvent *err)
{
     char s[1024];
     XGetErrorText(dpy, err->error_code, (char*)s, sizeof(s));
     D("!!!ERROR_HANDLER!!! - %s\n", s);
     return 0;
}

/*****************************************************************************
 * Make the window swallowed by the browser.
 *****************************************************************************/
static void place_window()
{
     int x;
     Window parent = WINDOW;
     int w = windata.width;
     int h = windata.height;
     XWMHints *leader_change;

     D("Parent Window: %x\n", parent);
     D("Victim Window: %x\n", victim);

     XWithdrawWindow(display, victim, XScreenNumberOfScreen (wattr.screen));
     XSync(display, FALSE);

     D("Changing leader of window %x\n", victim);
     if ((leader_change = XGetWMHints(display, victim)))
     {
	  leader_change->flags = (leader_change->flags | WindowGroupHint);
	  leader_change->window_group = wattr.root;
	  XSetWMHints(display, victim, leader_change);
	  XFree(leader_change);
     }

     D("Reparenting window %x into %x\n", victim, parent);
     for (x = 0; x < 25; x++)
     {
	  XReparentWindow(display, victim, parent, 0, 0);
	  XSync(display, FALSE);
     }

     XMapWindow(display, victim);
     XSync(display, FALSE);

     if (flags & H_FILL) {
	  fill_window(display, victim, w, h);
     } else if (flags & H_MAXASPECT) {
	  max_aspect_window(display, victim, w, h);
     }

     D("placewindow done\n");
}

/*****************************************************************************
 * Helper functions for winrecur().
 *****************************************************************************/
static int inital_win_cb(Window w, int depth)
{
     if (windows_found < MAX_IGNORED)
	  ignored[windows_found++] = w;
     return 0;
}

static int win_cb(Window w, int depth)
{
     int q;
     for (q = 0; q < windows_found; q++)
	  if (ignored[q] == w)
	       return 0;
     return 1;
}

static int is_viewable(Display *display, Window w)
{
     XWindowAttributes a;

     if (!XGetWindowAttributes(display, w, &a))
     {
          D("Could not get Window Attributes");
	  return 0;
     }
     if (a.map_state != IsViewable)
     {
	  D("Invisible window\n");
	  return 0;
     }
     return 1;
}

/*****************************************************************************
 * Find the window to swallow by scanning a window tree recursively.
 *****************************************************************************/
static Window winrecur(Window from,
		       int depth,
		       char *name,
		       int (*cb)(Window,int))
{
     Window root, parent;
     Window *children = NULL;
     unsigned int e, num_children;
     int found = -1;
  
     D("Winrecur, checking window %x (depth=%d)\n", from, depth);

     if (!XQueryTree(display, from, &root, &parent, &children, &num_children))
     {
	  D("Querytree failed!!!\n");
	  return (Window)0;
     }

     D("Num children = %d\n", num_children);

     for (e = 0; e < num_children; e++)
     {
	  char *windowname;
	  unsigned long nitems=0, bytes=0; 
	  unsigned char *property=0;
	  int fmt;
	  Atom type;

	  if (!is_viewable(display, children[e]))
	       continue;

	  D("XFetchName\n");
	  if (XFetchName(display, children[e], &windowname))
	  {
	       int yes=0;
	       D("Winrecur, checking window NAME %x (%s != %s)\n",children[e],windowname,name);
   
	       if (!name || !strcmp(windowname, name))
		    yes = cb(children[e],depth);
	       XFree(windowname);
	       if (yes) found = children[e];
	  }

	  /* Return the last window found */
	  if (found != -1) goto success;

	  D("XGetTextProperty\n");
	  property = 0;
	  XGetWindowProperty(display, children[e], XA_WM_CLASS,
			     0, 10000,
			     0,
			     XA_STRING,
			     &type, &fmt, &nitems, &bytes,
			     &property);

	  if (property)
	  {
	       if (type == XA_STRING && fmt==8)
	       {
		    int len = nitems;
		    char *n = (char *)property;
		    while (len)
		    {
			 D("Winrecur, checking window CLASS %x (%s != %s)\n", children[e], n,name);
			 if (!name || !strcmp(n, name))
			      if (cb(children[e],depth))
				   found = children[e];
	  
			 len -= strlen(n)+1;
			 n += strlen(n)+1;
		    }
	       }
	       XFree(property);
	  }
     }
     if (found != -1) goto success;
     if (depth > 0)
     {
	  for (e = 0;  e < num_children; e++)
	  {
	       Window ret = winrecur(children[e], depth-1, name, cb);
	       if (ret) found = ret;
	  }
     }
     if (found != -1) goto success;
     XFree(children);
     return (Window)0;

 success:
     XFree(children);
     return (Window)found;
}

/*****************************************************************************
 * Swallow using winrecur()
 *****************************************************************************/
static int rec_swallow()
{
     if ((victim = winrecur(wattr.root, 3, winname, win_cb)))
     {
	  D("Found it!! Victim=%x\n", victim);
	  place_window();
	  return 1;
     }
     return 0;
}

/*****************************************************************************
 * Swallow using XCheckMaskEvent()
 *****************************************************************************/
static int event_swallow()
{
     XEvent evt;
     XClassHint windowclass;
     char *windowname;
     int found = 0;
     int len;
     char *victimname = winname;

     D("Looking for victim... (%s)\n", victimname);
     /* XCheckMaskEvent is a non-blocking call, go through all
	waiting SubStructureEvents, untill we're out of them. */
     while (XCheckMaskEvent(display,SubstructureNotifyMask,&evt))
     {
	  /* got a structure event */
	  D("Got Structure Event of type %x\n", evt.type);
	  if (evt.type == CreateNotify)
	  {
	       /* looks promising, let's see if it's the one we're looking for */
	       D("Found CreateNotify Event %x\n", evt.type);

	       if (!XGetClassHint(display, evt.xcreatewindow.window, &windowclass))
	       {
		    D("Could Not get Class Hint, trying the next one");
		    continue;
	       }

	       if (!XFetchName(display, evt.xcreatewindow.window, &windowname))
	       {
		    D("Could Not Get Window Name, trying the next one");
		    continue;
	       }

	       if (windowname)
	       {
		    D("Victim Name is :%s\n", victimname);
		    D("Window Name is :%s\n", windowname);
		    len = strlen(victimname);
		    if (strncmp(victimname, windowname, len) == 0 ||
			strncmp(victimname, windowclass.res_class, len) == 0 ||
			strncmp(victimname, windowclass.res_name, len) == 0)
		    {
			 victim = evt.xcreatewindow.window;
			 found = 1;
			 D("Found it!! Victim=%x\n", victim);
			 D("Flags %x\n", flags);
			 place_window();
		    }
		    XFree(windowname);
	       }
	       if (found) break;
	  }
     }

     return found;
}


/*****************************************************************************
 * Prepare the swallowing.
 *****************************************************************************/
static int init_swallow()
{
     D("Setting up X data\n");

     if (!display)
	  return 1;

     D("display=%x\n", display);

     if (!XGetWindowAttributes(display, WINDOW, &wattr))
          return 1;

     D("rootwin=%x\n", wattr.root);

     victim = 0;

     if (flags & H_EVENT_SWALLOW)
     {
          if (!XSelectInput(display, DefaultRootWindow(display), SubstructureNotifyMask))
	       return 1;
     }
     else
     {
	  windows_found = 0;
	  winrecur(wattr.root, 3, winname, inital_win_cb);
     }

     D("Done setting up X data\n");

     return 0;
}

/*****************************************************************************
 * Terminate the swallowing.
 *****************************************************************************/
static void end_swallow()
{
     if (flags & H_EVENT_SWALLOW)
     {
	  XSelectInput(display, DefaultRootWindow(display), 0);
     }
}

/*****************************************************************************
 * Wrapper for execvp().
 *****************************************************************************/
static void run_app(char **argv)
{
     int nl;

     if (flags & H_NOISY)
     {
	  D("Redirecting stdout and stderr\n");
	  if ((nl = open("/dev/null", O_RDONLY)) == -1)
	       exit(EX_UNAVAILABLE);
	  dup2(nl, 1);
	  dup2(nl, 2);
	  close(nl);
     }
     
     execvp(argv[0],argv);
     D("Execvp failed..%d\n", errno);
     exit(EX_UNAVAILABLE);
}

/*****************************************************************************
 * Handle the swallowing, the SIGWINCH signal, and wait.
 *****************************************************************************/
static void handle_app(int pid)
{
     int waitpid_done = 0;
     int swallow_done = 0;
     int status;
     int tmp;

     if ((flags & H_SWALLOW) || (flags & H_EVENT_SWALLOW))
     {
	  for (tmp = 0; tmp < SWALLOW_LOOPS; tmp++)
	  {
	       if (!(flags & H_DAEMON))
	       {
		    if (waitpid(pid, &status, WNOHANG) > 0)
		    {
			 waitpid_done = 1;
			 break;
		    } 
	       }

	       if (flags & H_SWALLOW)
		    swallow_done = rec_swallow();
	       else
		    swallow_done = event_swallow();

	       if (swallow_done) break;

	       usleep(SWALLOW_WAIT_TIME);
	  }
	  end_swallow();
     }
     
     while (!waitpid_done)
     {
	  if (got_sigwinch)
	  {
	       got_sigwinch = 0;
	       if (swallow_done)
		    place_window();
	  }

	  if (flags & H_DAEMON)
	       exit(0);

	  D("Waiting for godot... %d\n",pid);
	  if (waitpid(pid, &status, 0) > 0)
	       waitpid_done++;
	  D("Wait done?? (%d)\n",waitpid_done);
     }
  
     if (!WIFEXITED(status))
     {
	  D("Process dumped core or something...\n");
	  exit(10);
     }
     if (WEXITSTATUS(status) && !(flags & H_IGNORE_ERRORS))
     {
	  D("Process exited with error code: %d\n", WEXITSTATUS(status));
	  exit(WEXITSTATUS(status));
     }
     D("exited ok!\n");
}

int main(int argc, char **argv)
{
     D("Helper started.....\n");

     if (argc < 2)
     {
	  fprintf(stderr,"MozPlugger version " VERSION " helper application.\n"
		  "Written by Fredrik Hubinette and Louis Bavoil.\n" 
		  "Please see 'man mozplugger' for details.\n");
	  exit(1);
     }

     signal(SIGWINCH, sigwinch);

     sscanf(argv[1],"%d,%d,%lu,%d,%d,%d,%d",
	    &flags,
	    &repeats,
	    (long unsigned int *) &windata.window,
	    (int *)&windata.x,
	    (int *)&windata.y,
	    (int *)&windata.width,
	    (int *)&windata.height);

     command = argv[2];
     winname = getenv("winname");
     file = getenv("file");

     D("HELPER: %s %s %s %s\n",
       argv[0],
       argv[1],
       file,
       command);

     while (repeats > 0)
     {
	  char *argv[10];
	  int loops = 1;
	  int pid;

	  /* This application will use the $repeat variable */
	  if (flags & H_REPEATCOUNT)
	       loops = repeats;

	  /* Expecting the application to loop */
	  if (flags & H_LOOP)
	       loops = 0x7fffffff;

	  /* Create command line */
	  if (flags & H_CONTROLS)
	  {
	       argv[0] = getenv("controller");
	       argv[1] = command;
	       argv[2] = NULL;
	  }
	  else
	  {
	       argv[0] = "/bin/sh";
	       argv[1] = "-c";
	       argv[2] = command;
	       argv[3] = NULL;
	  }

	  D("Execing %s (repeats=%d loops=%d)\n",command,repeats,loops);

	  if ((flags & H_SWALLOW) || (flags & H_EVENT_SWALLOW))
	  {
	       XSetErrorHandler(error_handler);

	       if (!display)
	       {
		    display = XOpenDisplay(displayname);
	       }

	       if (init_swallow())
	       {
		    D("setup_swallow() failed!!!!\n");
		    flags &=~ H_SWALLOW;
		    flags &=~ H_EVENT_SWALLOW;
	       }
	  }
    
	  D("Running %s\n", command);

	  if ((pid = fork()) == -1)
	       exit(10);
      
	  if (!pid)
	  {
	       run_app(argv);
	  } else {
	       D("waiting for (%d)\n", pid);
	       handle_app(pid);
	       D("wait done repeats=%d loops=%d\n", repeats, loops);
	       if (repeats < MAXINT)
		    repeats -= loops;
	  }
     }
     _exit(0);
}
