/*$Id: Test.cpp,v 1.9 2006/08/20 16:39:04 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "Common.h"
#include "RulesParser.h"
#include "ActiveRuleSet.h"
#include "Wrap.h"
#include "Error.h"

#define TESTDIR "tests/"
#define BUFSIZE 16000

Rules *testRules;


struct Test
{
	int fid;
	char *url;
	int urllen;
	char *headers;
	int headerslen;
	char *body;
	int bodylen;
};


void readRules(char *rules_path)
{
	char *error;
	char buf[BUFSIZE];
	int len;
	
	RulesParser p(&error);
	if (error){
		printf(error);
	}
	
	FILE *input = fopen(rules_path, "r");
	if (!input)
		err_sys("can't open rules file %s", rules_path);
	len = fread(buf, 1, BUFSIZE, input);
	buf[len] = 0;
	testRules = new Rules();
	testRules->init();
	p.parseRules(testRules, buf, &error);
	if (error){
		printf(error);		
	}
}

bool emptyLine(char *p)
{
	for(; *p; p++)
		if (!isspace(*p))
			return false;
	return true;
}

Test readTest(char *test_path)
{
	Test r;
	char buf[BUFSIZE];
	
	FILE *test_file = fopen(test_path, "r");
	if (!test_file)
		err_sys("fopen");
	fgets(buf, BUFSIZE, test_file);
	if (sscanf(buf, "%d", &r.fid) != 1)
		err_quit("incorrect test syntax %s: fid missing", test_path);
	
	do{
		fgets(buf, BUFSIZE, test_file);
	}while(emptyLine(buf));	
	if (sscanf(buf, "%d", &r.urllen) != 1)
		err_quit("incorrect test syntax %s: url len missing", test_path);
	
	r.url = (char *)Malloc(r.urllen + 1);
	if (fread(r.url, sizeof(char), r.urllen, test_file) != (unsigned int)r.urllen)
		err_quit("incorrect test syntax %s: url too short", test_path);	
	r.url[r.urllen] = 0;
	
	do{
		fgets(buf, BUFSIZE, test_file);
	}while(emptyLine(buf));	
	if (sscanf(buf, "%d", &r.headerslen) != 1)
		err_quit("incorrect test syntax %s: headers len missing", test_path);
	
	r.headers = (char *)Malloc(r.headerslen + 1);
	if (fread(r.headers, sizeof(char), r.headerslen, test_file) 
	    != (unsigned int)r.headerslen)
		err_quit("incorrect test syntax %s: headers too short", test_path);
	r.headers[r.headerslen] = 0;

	do{
		fgets(buf, BUFSIZE, test_file);
	}while(emptyLine(buf));
	if (sscanf(buf, "%d", &r.bodylen) != 1)
		err_quit("incorrect test syntax %s: body len missing", test_path);
	
	r.body = (char *)Malloc(r.bodylen);
	if (fread(r.body, sizeof(char), r.bodylen, test_file) != (unsigned int)r.bodylen)
		err_quit("incorrect test syntax %s: body too short", test_path);
	
	return r;
}

void test(int argc, char **argv)
{
	for(int i = 2; i < argc; i++){
		Test t = readTest(argv[i]);
		ActiveRuleSet ruleSet(testRules);
		list<const Rule*> result, tmp;
		result = ruleSet.checkURL(t.url, t.urllen);
		TRACE("SIZE %d", result.size());
		tmp = ruleSet.checkHeaders(t.headers, t.headerslen);
		result.insert(result.begin(), tmp.begin(), tmp.end());
		TRACE("SIZE %d", result.size());
		tmp = ruleSet.checkBody(t.body, t.bodylen);
		result.insert(result.begin(), tmp.begin(), tmp.end());
		if (result.size() == 0){
			if (t.fid == 0)
				printf("Test %s ok\n", argv[i]);
			else
				printf("Test %s ERROR: test data didn't match any rule\n", argv[i]);
		}
		else if (result.size() > 1){
			printf("Test %s ERROR: test data matched %d rules\n", 
			       argv[i], result.size());
		}
		else if ((*result.begin())->fid == t.fid)
			printf("Test %s OK\n", argv[i]);
		else
			printf("Test %s ERROR: test data matched rule %d instead of %d\n", 
			       argv[i], (*result.begin())->fid, t.fid);
		free(t.url);
		free(t.headers);
		free(t.body);
	}
}
int
main(int argc, char **argv)	
{
	if (argc < 2){
		fprintf(stderr, "Test rule_file [test_files]\n");
		return 1;
	}
	readRules(argv[1]);
	test(argc, argv);
	return 0;
}
