/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */


namespace Hultmann.Zip
{

internal sealed class ZipFile : System.Collections.Generic.IEnumerable<ZipEntry>, System.IDisposable
{

private readonly System.IO.Stream                           m_archive;
private readonly System.Collections.Generic.List<ZipEntry>  m_entries = new System.Collections.Generic.List<ZipEntry>();
private readonly string                                     m_filename = null;
//private event Errors

public ZipFile(string name)
{
    m_filename = name;
    m_archive = System.IO.File.OpenRead(name); // FileNotFoundException
    this.ReadAllEntries(m_archive);
}


public ZipFile(System.IO.Stream archive)
{
    m_archive = archive;
    this.ReadAllEntries(m_archive);
}


void System.IDisposable.Dispose()
{
    this.Close();
}


public void Close()
{
    m_archive.Close();
    m_entries.Clear();
}


private void ReadAllEntries(System.IO.Stream archive)
{
    archive.Position = 0L;
    ZipEntry zip;
    do {
        if (CheckSignature(archive) == false) {
            if (m_entries.Count > 0) {
                return; // no more files
            } else {
                throw new System.IO.InvalidDataException();
            }
        }

        zip = NextEntry(archive);
        if (zip == null) {
            throw new System.IO.InvalidDataException();
        }

        zip.ZipFileIndex = m_entries.Count;
        m_entries.Add(zip);

    } while (zip != null);

    // todo: read central directory
}


private static ZipEntry NextEntry(System.IO.Stream archive)
{
    ZipEntry zip = ParseEntry(archive);
    if (zip != null) {
        zip.Offset = (int) archive.Position;
        archive.Seek(zip.CompressedSize, System.IO.SeekOrigin.Current);
    }
    return zip;
}


private static bool CheckSignature(System.IO.Stream archive)
{
    const int n = 4;
    byte[] bin = new byte[n];
    if (archive.Read(bin, 0, n) != n) {
        return false;
    }
    return bin[0] == 0x50
        && bin[1] == 0x4b
        && bin[2] == 0x03
        && bin[3] == 0x04;
}


private static ZipEntry ParseEntry(System.IO.Stream archive)
{
    const int HeaderSize = 26;
    byte[] header= new byte[HeaderSize];
    if (archive.Read(header, 0, HeaderSize) != HeaderSize) {
        return null;
    }

    int  version    = System.BitConverter.ToUInt16(header,  0);
    int  flags      = System.BitConverter.ToUInt16(header,  2);
    int  method     = System.BitConverter.ToUInt16(header,  4);
    long datetime   = System.BitConverter.ToUInt32(header,  6);
    long crc        = System.BitConverter.ToUInt32(header, 10);
    long compressed = System.BitConverter.ToUInt32(header, 14);
    long size       = System.BitConverter.ToUInt32(header, 18);
    int  lenName    = System.BitConverter.ToUInt16(header, 22);
    int  lenExtra   = System.BitConverter.ToUInt16(header, 24);


    byte[] binName = new byte[lenName];
    if (archive.Read(binName, 0, lenName) != lenName) {
        return null;
    }
    System.Text.Encoding enc = System.Text.Encoding.GetEncoding(850);
    string name = enc.GetString(binName, 0, lenName);

    byte[] extra = new byte[lenExtra];
    if (archive.Read(extra, 0, lenExtra) != lenExtra) {
        return null;
    }

    ZipEntry zip = new ZipEntry(name, version);
    zip.DosTime           = datetime;
    zip.Flags             = flags;
    zip.Crc               = crc;
    zip.CompressedSize    = compressed;
    zip.Size              = size;
    zip.CompressionMethod = (CompressionMethod) method;
    zip.ExtraData         = extra;

    return zip;
}


public ZipEntry this[int idx]
{
    get {
        return m_entries[idx].Clone();
    }
}


public System.IO.Stream GetInputStream(int idxEntry)
{
    ZipEntry entry = m_entries[idxEntry];
    System.IO.MemoryStream entryStream = ReadEntryStream(entry, m_archive);
    if (entryStream == null) {
        return null;
    }

    entryStream.Position = 0L;

    switch (entry.CompressionMethod) {

        case CompressionMethod.Deflated:
            System.IO.Compression.DeflateStream def
                = new System.IO.Compression.DeflateStream(entryStream,
                                                          System.IO.Compression.CompressionMode.Decompress);
            // workaround: DeflateStream doesn't support Length property
            System.IO.MemoryStream mem = new System.IO.MemoryStream((int) entry.Size);
            WriteToStream(def, mem);
            def.Close();
            mem.Position = 0L;
            return mem;

        case CompressionMethod.Stored:
            return entryStream;

        default:
            return null;
    }
}


private static void WriteToStream(System.IO.Stream input, System.IO.Stream output)
{
    const int LenBuffer = 128;
    int count;
    byte[] buffer = new byte[LenBuffer];
    do {
        count = input.Read(buffer, 0, LenBuffer);
        output.Write(buffer, 0, count);
    } while (count > 0);
}


private static System.IO.MemoryStream ReadEntryStream(ZipEntry entry, System.IO.Stream archive)
{
    long total = entry.CompressedSize;
    long remaining = total;
    System.IO.MemoryStream entryStream = new System.IO.MemoryStream((int) total);

    const int LenBuffer = 256;
    byte[] buffer = new byte[LenBuffer];
    archive.Position = entry.Offset;

    do {
        int sizeBlock = remaining > LenBuffer ? LenBuffer : (int) remaining;
        int filledBlock = archive.Read(buffer, 0, sizeBlock);
        entryStream.Write(buffer, 0, filledBlock);
        remaining -= filledBlock;
    } while (remaining > 0);


    if (entryStream.Length == total) {
        return entryStream;
    } else {
        return null;
    }
}


public bool TestArchive(bool testData)
{
    int qty = this.Size;
    for (int idx = 0; idx < qty; idx++) {
        System.IO.Stream bin = this.GetInputStream(idx);
        MyCrc myCrc = new MyCrc();
        const int LenBuffer = 128;
        byte[] buffer = new byte[LenBuffer];
        int count;
        do {
            count = bin.Read(buffer, 0, LenBuffer);
            myCrc.Update(buffer, count);
        } while (count > 0);

        if (this[idx].Crc != myCrc.Value) {
            return false;
            // event
        }
    }

    return true;
}


public System.Collections.Generic.IEnumerator<ZipEntry> GetEnumerator()
{
    foreach (ZipEntry zip in m_entries) {
        yield return zip.Clone();
    }
}


System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
{
    return this.GetEnumerator();
}


public string Name
{
    get {
        return m_filename;
    }
}


public int Size
{
    get {
        return m_entries.Count;
    }
}


public int FindEntry(string name, bool ignoreCase)
{
    System.StringComparison comp = ignoreCase ? System.StringComparison.OrdinalIgnoreCase
                                              : System.StringComparison.Ordinal;

    for (int idx = 0; idx < m_entries.Count; idx++) {
        if (string.Equals(m_entries[idx].Name, name, comp)) {
            return idx;
        }
    }

    return -1;
}


// ==================================================================================================================================
}//class
}//ns
