/* sim.h

   originally written by Don Maszle

   Copyright (c) 1993.  Don Maszle, Frederic Bois.  All rights reserved.

   -- Revisions -----
     Logfile:  SCCS/s.sim.h
    Revision:  1.19
        Date:  22 Jan 1996
     Modtime:  15:36:32
      Author:  @a
   -- SCCS  ---------

   Header file for simulation
*/

#ifndef _SIM_H_
#define _SIM_H_

/* -----------------------------------------------------------------------------
   Inclusions
*/

#include "modiface.h"
#include "list.h"
#include "matutil.h"
#include "random.h"


/* -----------------------------------------------------------------------------
   Constants
*/

/* These are potential array size problems.
   Other maximum sizes are MAX_EQN and MAX_LEX in lex.h */

#define DIM1MAX               100  /* should be > number of states + outputs */
#define MAX_EXPERIMENTS       35
#define LSODES_IWORKSIZE      300  /* Lsodes tells if this is not big enough */
#define LSODES_RWORKSIZE      1000 /* Idem */
#define ARGS_MAX              8    /* Maximum number of args to lex */
                                   /* Used in allocating argument strings */

/* Keyword Map constants */

#define KM_INTEGRATE       1
#define KM_SIMULATE        2
#define KM_PRINT           5
#define KM_PRINTSTEP       6
#define KM_DATA            7

#define KM_SIMTYPE         8  /* For simulation type specification */
#define KM_DEFAULTSIM      9  /* 'Normal' keyword: normal sims */

#define KM_EXPERIMENT      10
#define KM_MONTECARLO      11
#define KM_MCVARY          12
#define KM_SETPOINTS       13
#define KM_OUTPUTFILE      14
#define KM_GIBBS           15

#define KM_END             100

/* Function argument keywords */

#define KM_YES             200
#define KM_NO              201

#define KM_UNIFORM         210
#define KM_LOGUNIFORM      211
#define KM_BETA            212
#define KM_NORMAL          213
#define KM_LOGNORMAL       214
#define KM_TRUNCNORMAL     215
#define KM_TRUNCLOGNORMAL  216
#define KM_CHI2            217
#define KM_SINE            218
#define KM_PIECEWISE       219

#define KM_PREDICTION      300

#define KM_LSODES          600

/* Context bit flags */

#define CN_END             0x0000    /* No context */

#define CN_GLOBAL          0x0001
#define CN_EXPERIMENT      0x0002
#define CN_FUNCARG         0x0100

#define CN_ALL             0xFFFF    /* All contexts */

/* Analysis Types */

#define AT_NOTSPECD     0    /* Not yet specified */
#define AT_DEFAULTSIM   1    /* Normal simulation */
#define AT_MONTECARLO   2    /* Monte Carlo variations */
#define AT_SETPOINTS    3    /* Set points simulation */
#define AT_GIBBS        4    /* Gibbs estimation */

/* Options Flags -- bit flags */

#define OF_PARMLIST    0x0001   /* Print parameter list */
#define OF_MCOUTPUTS   0x0002   /* Print MCOutData array -- output values    */
#define OF_MCRESULT    0x0004   /* Print result of MC or SetPoints analysis  */
#define OF_PASSFAIL    0x0008   /* Print result of MC as Pass/Fail [1 or 0]  */
#define OF_MONTEBITS   0x000F

#define OF_COMMENT     0x0010   /* Print a comment (no comment) to outfile */
#define OF_VARNAMES    0x0020   /* Print the variable names (no names)     */
#define OF_VERT_OUTPUT 0x0040   /* Print variables in columns (rows) */
#define OF_NORMALBITS  0x00F0

#define OF_CMDLSEED    0x1000    /* Seed given on command-line */
#define OF_CMDLOUTFILE 0x2000    /* Output filename given on command-line */
#define OF_CMDLBITS    0xF000

/* Monte Carlo Variation types */

#define MCV_SETPOINTS       (-1) /* Not really Monte Carlo */
#define MCV_UNIFORM         0
#define MCV_LOGUNIFORM      1
#define MCV_BETA            2
#define MCV_NORMAL          3
#define MCV_LOGNORMAL       4
#define MCV_TRUNCNORMAL     5
#define MCV_TRUNCLOGNORMAL  6
#define MCV_CHI2            7
#define MCV_SINE            8
#define MCV_PIECEWISE       9

/* MonteCarlo dependency levels */

#define MC_DEPLEVELS 2

/* Integration Method types */

#define IAL_LSODES 3  /* lsodes algorithm */

/* Integrator spec defaults */

#define IAL_DEFAULT     IAL_LSODES
#define IOPT_DEFAULT    (0)
#define ITOL_DEFAULT    (1)
#define ITASK_DEFAULT   (1)
#define RTOL_DEFAULT    (1.0e-5)
#define ATOL_DEFAULT    (1.0e-7)
#define IMF_DEFAULT     (0)  /* non-stiff */

/* Simulation specification defaults */

#define SEED_DEFAULT 3141592.653589793

#define T0_DEFAULT            0.0
#define TFINAL_DEFAULT        0.0
#define NSIMULATIONS_DEFAULT  0

/* Defs for Distrib statement */

#define MCVP_FIXED    0
#define MCVP_VARIABLE 1


/* ----------------------------------------------------------------------------
   Typedefs
*/

typedef double matType1[DIM1MAX];

/* Union of two types of variables: constants and input fns */

typedef union tagUVAR {
  double dVal;
  PIFN pifn;
} UVAR; /* tagUVAR */

/* Modification specification for one variable */

typedef struct tagVARMODIFICATION {
  HVAR hvar; /* Handle to the variable */
  UVAR uvar; /* Union of variable value or input function spec */
} VARMODIFICATION, *PVARMOD; /* tagVARMODIFICATION */

/* Specification of integrator settings */

typedef struct tagINTSPEC {
  int     iAlgo;          /* one of IM_ types */
  long    iopt;           /* optional inputs flag */
  long    itask;          /* type of work */
  long    itol;           /* type of error checking */
  double  dRtol;          /* relative error tolerance */
  double  dAtol;          /* aboslute error tolerance */
  long    iMf;            /* 0 = nonstiff, 1 = stiff */
  long    iDSFlag;        /* lsodes return flag */
  long    liw;            /* length of iwork array */
  long    lrw;            /* length of rwork array */
  PLONG   iwork;          /* working array pointer */
  PDOUBLE rwork;          /* working array pointer */
} INTSPEC, *PINTSPEC; /* tagINTSPEC */


/* Print Record: for info from a Print() statement */

typedef struct tagPRINTREC {
  PSTR szOutputName;
  HVAR hvar;
  int  cTimes;
  PDOUBLE pdTimes;
} PRINTREC, *PPRINTREC; /* tagPRINTREC */


/* Data record: for info from a Data() statement */

typedef struct tagDATAREC {
  PSTR szDataName;
  HVAR hvar;
  int  cData;
  PDOUBLE pdData;
} DATAREC, *PDATAREC; /* tagDATAREC */


/* Output specification */

typedef struct tagOUTSPEC {
  PSTR  szOutfilename;      /* Name of file for regular output */
  PFILE pfileOut;           /* Pointer to file */
  BOOL  bCommandLineSpec;   /* Output file specified on command line */

  int   nOutputs;           /* Number of outputs */
  PSTR  *pszOutputNames;    /* Array of output names */
  HVAR  *phvar;             /* Array of handles to outputs */

  PLIST plistPrintRecs;     /* List of records from Print()'s */
  PLIST plistDataRecs;      /* List of records from Data()'s */

  /* The lists are converted into the following */

  PINT    pcOutputTimes;    /* Count of output times for each var */
  PINT    piCurrentOut;     /* Index to current output for each var */
  PINT    piSigmas;         /* Array of SDs for each var */
  PDOUBLE *prgdOutputTimes; /* Array of output times for each var */
  PDOUBLE *prgdOutputVals;  /* Array of output values for each var */
  PDOUBLE *prgdDataVals;    /* Array of data values for each var */

  int cDistinctTimes;       /* Count of distinct output times */
  PDOUBLE rgdDistinctTimes; /* Array of distinct output times */

} OUTSPEC, *POUTSPEC; /* tagOUTSPEC */

/* Union of value, parameter handle for Distrib statement */

typedef union tagMCVU {
  double dval;
  HVAR hvar;
} MCVU, *PMCVU; /* tagMCVU */


/* Monte Carlo Variation for one parameter */

typedef struct tagMCVAR {
  HVAR hvar;              /* Handle to variable to be modified */
  int  iType;             /* One of MCV_ types */
  char cLevel;            /* Level of dependency */
  char cVarParm;          /* Flags for variable parameters */
  /*BOOL bDependsOnDistrib; True if param vars are in other Distribs */
  MCVU uParm1, uParm2;    /* Shape parms */
  MCVU uMin, uMax;        /* Range */
  double dVal;            /* Value for this run */
} MCVAR, *PMCVAR;   /* tagMCVAR */


typedef struct tagGIBBSDATA {
  long nMaxIter;        /* Number of iterations */
  long nInitIter;       /* Number of iterations prior to vector sampling */
  long nPrintFreq;      /* To request output every nPrintFreq iterations */
  long nPrintIter;      /* Number of final iterations to print */

  PSTR szGout;          /* Filename for Gibbs output */
  PFILE pfileGibOut;    /* File for Gibbs output */

  PSTR szGrestart;      /* Parm file  for restarting Gibbs */
  PFILE pfileRestart;   /* File for Gibbs restart */

  PSTR szGdata;         /* Filename for Gibbs input data */

} GIBBSDATA, *PGIBBSDATA; /* tagGIBBSDATA */


/* Specification for Monte Carlo type experiment */

typedef struct tagMONTECARLO {
  int    nRuns;             /* Number of Monte Carlo runs */
  double dRandTemp;         /* Temporary random number draw */
  BOOL   bIndependent;      /* Variables independent from run to run */

  long   lRun;              /* Number of current Run */

  long   nPasses, nFails;   /* Number of passes and fails */

  PSTR szMCOutfilename;     /* File name for Monte Carlo output */
  PFILE pfileMCOut;         /* File for Monte Carlo output */

  PSTR szMCPassFilename;    /* File name for Monte Carlo output */
  PFILE pfileMCPass;        /* File for Monte Carlo output */

  PSTR szMCFailFilename;    /* File name for Monte Carlo output */
  PFILE pfileMCFail;        /* File for Monte Carlo output */

  PSTR szSetPointsFilename; /* File name for set points */
  PFILE pfileSetPoints;     /* File of set points */

  PLIST plistMCVars;        /* List of MCVAR record, variation specs */

  /* The list is converted to the following */
  long nParms;              /* Count of parameters */
  double *rgdParms;         /* The actually used parameter vector */
  HVAR *rghvar;             /* Array of handles to the parameters */
  MCVAR **rgpMCVar;         /* A priori distributions for each */

  long nSetParms;           /* Count of setpoint parameters */

} MONTECARLO, *PMONTECARLO; /* tagMONTECARLO */


  /* Record of info about the model */

typedef struct tagMODELINFO {
  long        nStates;
  long        nModelVars;

  PDOUBLE     pdModelVars;
  matType1    rgStatesMax;
  matType1    rgError;

} MODELINFO, *PMODELINFO; /* tagMODELINFO */


  /* Record of information for one experiment.
     An experiment specifies a set of experimental
     conditions, parameter settings, input levels, etc.
   */

typedef struct tagEXPERIMENT {
  int iExp;                 /* Number of this experiment */

  double dT0;               /* Time limits */
  double dTfinal;

  double dTime;             /* Current Time -- not used for global sim */

  int iSubject;             /* Subject #, eventually */

  PMODELINFO pmodelinfo;    /* Pointer to the model information */
  PLIST plistParmMods;      /* List of parameter mods (elt = PVARMOD) */
  INTSPEC is;               /* Integrator spec, this experiment */
  OUTSPEC os;               /* Output spec, this experiment */

} EXPERIMENT, *PEXPERIMENT; /* tagEXPERIMENT */


/* Defines an analysis for an input file */

typedef struct tagANALYSIS {

  int iType;            /* Type of analysis. One of AT_ types */
  int fOptions;         /* Options flags -- combination of OF_ above */

  int fCmdOptions;      /* Command line opts, overwrite defaults */
  int fNotOptions;      /* Command line opts, inhibits these options */

  WORD wContext;        /* Context flag used during input processing */
  double dSeed;         /* Random seed used for all analyses */

  MODELINFO   modelinfo;/* The model we are using */

  EXPERIMENT  expGlobal;/* Global experiment settings */

  PEXPERIMENT rgpExps[MAX_EXPERIMENTS];/* List of pointer to experiments */
  PEXPERIMENT pexpCurrent;             /* Experiment being currently defined */

  MONTECARLO    mc;     /* Monte Carlo specification */
  GIBBSDATA     gd;     /* Gibbs estimate specification */

} ANALYSIS, *PANALYSIS; /* tagANALYSIS */


/* -----------------------------------------------------------------------------
   Globals/Externals
*/

extern PSTRLEX vrgszlexArgs[];


/* ----------------------------------------------------------------------------
   Prototypes
*/

void AnnounceProgram (void);
void CheckDependencies (PANALYSIS panal);
void CorrectInputToTransition (PEXPERIMENT, PDOUBLE);

int  DoOneExperiment (PEXPERIMENT pexp);

void DoAnalysis (PANALYSIS panal);
void DoMonteCarlo (PANALYSIS panal);
void DoNormal (PANALYSIS panal);
void DoOneMCExp (PANALYSIS panal, PEXPERIMENT pexp);
void DoOneNormalExp (PANALYSIS panal, PEXPERIMENT pexp);

void FreeVarMod (PVOID pData);

BOOL GetCmdLineArgs (int cArg, char *const *rgszArg, PSTR *pszFileIn,
                     PSTR *pszFileOut, PANALYSIS panal);
void GetOutputFlagOption (PANALYSIS panal, char *optarg);

int  MCVarListToArray (PVOID pv_pMCVar, PVOID pv_Null);
int  ModifyOneParm (PVOID pData, PVOID pNullInfo);
void ModifyParms (PLIST plistParmMods);

void PrepAnalysis (PANALYSIS panal);
int  ProcessMonteCarlo (PINPUTBUF, PANALYSIS, PSTR, int);
void PromptFilenames (PSTR *pszFileIn, PSTR *pszFileOut);

char *SansPath (char *szFullPathname);

void WriteArray (FILE *pfile, long cElems, double *rg);
void WriteArrayLog (FILE *pfile, long cElems, double *rg);

#endif /* _SIM_H_ */

/* End */

