# LXNavigation Eos80 Vario by Benedikt Wolf (D-ECHO) based on

# A3XX Lower ECAM Canvas
# Joshua Davidson (it0uchpods)

# THANKS TO Colin Geniet (SoundPitchController), original developers of the ilec-sc7 (WooT)

# Information based on manual https://downloads.lxnavigation.com/downloads/manuals/Eos/LX-Eos-80-user-manual-0.9H.pdf
#######################################

## REQUIRES:
##	* second airspeed indicator module enabled from <sim><instrumentation> ("eos-airspeed-indicator")
##	* second altimeter module enabled from <sim><instrumentation> ("eos-altimeter")
##	* power supply from systems/electrical/outputs/eos
##	(opt) * vario sound set up in the aircraft's sound file (see eos-sound.txt as an example)

var state = 0;		#	0 = off, 1 = init/firmware, 2 = select pilot, 3 = set elevation, 4 = set QNH, 5 = ON

var zeus_coupled = 1; # used with Zeus navigation display

var Eos_start = nil;
var Eos_main = [ nil ];
var Eos_display = nil;

var eos	=	props.globals.initNode("/instrumentation/eos");

var te_rdg	=	eos.initNode("te-reading-mps", 0.0, "DOUBLE");
var te_avg	=	eos.initNode("te-average-mps", 0.0, "DOUBLE");
var volume	=	eos.initNode("volume", 0.5, "DOUBLE");

setprop("instrumentation/eos-altimeter/serviceable", 1 );
setprop("instrumentation/eos-airspeed-indicator/serviceable", 1 );

var alt		=	props.globals.getNode("instrumentation/eos-altimeter/indicated-altitude-ft", 1);
var press_alt	=	props.globals.getNode("instrumentation/eos-altimeter/pressure-alt-ft", 1);
var qnh		=	props.globals.getNode("instrumentation/eos-altimeter/setting-hpa", 1);
var ias_kt		=	props.globals.getNode("instrumentation/eos-airspeed-indicator/indicated-speed-kt", 1);
var tas_kt		=	props.globals.getNode("instrumentation/eos-airspeed-indicator/true-speed-kt", 1);

var g_accel =	props.globals.getNode("accelerations/pilot-g", 1);
var g_max = [ 1.0, 1.0 ];

var oat		=	props.globals.getNode("environment/temperature-degc", 1);

var gps = {
	lat:	props.globals.getNode("position/latitude-string"),
	lon:	props.globals.getNode("position/longitude-string"),
	trk:	props.globals.getNode("/orientation/track-deg"),
};

var wind = {
	hdg: props.globals.getNode("/environment/wind-from-heading-deg", 1),
	spd: props.globals.getNode("/environment/wind-speed-kt", 1),
};

var hdg_true	=	props.globals.getNode("/orientation/heading-deg",	1);


var volt_prop	=	props.globals.initNode("/systems/electrical/outputs/eos", 0.0, "DOUBLE");

var mc		=	eos.initNode( "mc", 1.5, "DOUBLE" );

var needle		=	eos.initNode( "needle-deg", 0.0, "DOUBLE" );

var polar_def = [ -0.000997169190256748, 0.19882060566162, -13.4837656352864 ]; # see src/performance_calc.ods

var pilot_list = [ "SAMPLE", "GLIDER", "EOS", "ZEUS", "PHOENIX" ];
var pilot_x = 0;


var instrument_dir	=	"Aircraft/Phoenix/Models/Instruments/Eos/";

var canvas_Eos_base = {
	init: func(canvas_group, file) {
		var font_mapper = func(family, weight) {
			return "LiberationFonts/LiberationSans-Bold.ttf";
		};

		
		canvas.parsesvg(canvas_group, file, {'font-mapper': font_mapper});

		var svg_keys = me.getKeys();
		 
		foreach(var key; svg_keys) {
			me[key] = canvas_group.getElementById(key);
		}

		me.page = canvas_group;

		return me;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
		var volts = volt_prop.getDoubleValue();
		if( volts > 9 ){
			if( state >= 5 ){
				Eos_main[0].page.show();
				Eos_main[0].update();

				Eos_start.page.hide();

				check_g_force();
			} elsif( state >= 1 ) {
				Eos_start.page.show();
				foreach( var el; Eos_main ){
					el.page.hide();
				}
			} else {
				foreach( var el; Eos_main ){
					el.page.hide();
				}
				Eos_start.page.hide();
			}
		} else {
			foreach( var el; Eos_main ){
				el.page.hide();
			}
			Eos_start.page.hide();
		}
	},
};

var check_g_force = func{
	var current_val = g_accel.getDoubleValue();

	if( current_val < g_max[0] ) {
		g_max[0] = current_val;
	} elsif( current_val > g_max[1] ) {
		g_max[1] = current_val;
	}
}
	
var canvas_Eos_vario = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Eos_vario , canvas_Eos_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ "tas.digits", "trk.digits", "gforce.digits", "oat.digits","wind.needle", "wind.hdg", "wind.spd_kph","ias.digits", "asi.tape" ];
	},
	update: func() {
		# Various air speeds
		var ias_kph = ias_kt.getDoubleValue() * KT2MPS * 3.6;
		me["asi.tape"].setTranslation( 0, ias_kph * 2.5 );
		me["ias.digits"].setText( sprintf("%3d", math.round( ias_kph ) ) );
		me["tas.digits"].setText( sprintf("%3d", math.round( tas_kt.getDoubleValue() * KT2MPS * 3.6 ) ) );

		# Wind indication
		var wind_heading = wind.hdg.getDoubleValue();
		var ac_heading = hdg_true.getDoubleValue();
		me["wind.needle"].setRotation( ( ac_heading - wind_heading ) * D2R );
		me["wind.hdg"].setText( sprintf("%3d", math.round( wind_heading ) ) ~ "°" );
		me["wind.spd_kph"].setText( sprintf("%3d", math.round( wind.spd.getDoubleValue() * KT2MPS * 3.6 ) ) );

		me["trk.digits"].setText( sprintf("%3d", math.round( gps.trk.getDoubleValue() ) ) );
		me["gforce.digits"].setText( sprintf("%02.1f", g_accel.getDoubleValue() ) );
	}

};

var canvas_Eos_start = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Eos_start , canvas_Eos_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ "group.logo_firmware", "group.input", "group.qnh", "group.elev", "group.pilot", "heading", "pilot_number", "pilot_name", "elev.digits", "qnh.digits"];
	},
	on_state_change: func {
		if( state == 1 ){
			me["group.logo_firmware"].show();
			me["group.input"].hide();
		} elsif( state == 2 ){
			me["group.logo_firmware"].hide();
			me["group.input"].show();
			me["group.pilot"].show();
			me["heading"].setText( "Select pilot" );
			me["group.elev"].hide();
			me["group.qnh"].hide();
		} elsif( state == 3 ){
			me["group.logo_firmware"].hide();
			me["group.input"].show();
			me["group.pilot"].hide();
			me["group.elev"].show();
			me["heading"].setText( "Set elevation" );
			me["group.qnh"].hide();
		} elsif( state == 4 ){
			me["group.logo_firmware"].hide();
			me["group.input"].show();
			me["group.pilot"].hide();
			me["group.elev"].hide();
			me["group.qnh"].show();
			me["heading"].setText( "Set QNH" );
			me["qnh.digits"].setText( sprintf("%4d", math.round( qnh.getDoubleValue() ) ) );
		}
	},
	on_pilot_change: func {
		if( state == 2 ){
			me["pilot_number"].setText( sprintf("%1d", pilot_x ) );
			me["pilot_name"].setText( pilot_list[ pilot_x ] );
		}
	},
	on_elevation_change: func {
		if( state == 3 ){
		}
	},
	on_qnh_change: func {
		if( state == 4 ){
			me["qnh.digits"].setText( sprintf("%4d", math.round( qnh.getDoubleValue() ) ) );
		}
	},
};

var eos_update = maketimer(0.05, func() { canvas_Eos_base.update() } );
eos_update.simulatedTime = 1;


var switch_state = func {
	if( state == 0 ){
		state = 1;
		Eos_start.on_state_change();
		settimer( switch_state, 0.8 + rand() * 0.5 );
	} elsif( state == 1 ){
		if( zeus_coupled ) state = 5;
		else state = 2;
		Eos_start.on_state_change();
	} elsif( state == 2 ){
		state = 3;
		Eos_start.on_state_change();
	} elsif( state == 3 ){
		state = 4;
		Eos_start.on_state_change();
	} elsif( state == 4 ){
		state = 5;
		Eos_start.on_state_change();
	}
}

var ls = setlistener("sim/signals/fdm-initialized", func {
	Eos_display = canvas.new({
		"name": "Eos",
		"size": [320, 240],
		"view": [320, 240],
		"mipmapping": 1
	});
	Eos_display.addPlacement({"node": "eos80.screen"});

	Eos_start = canvas_Eos_start.new(Eos_display.createGroup(), instrument_dir~"eos80_start.svg");
	Eos_main[0] = canvas_Eos_vario.new(Eos_display.createGroup(), instrument_dir~"eos80_vario.svg");
	
	eos_update.start();
	
	removelistener(ls);
});

var check_electric_off = func () {
	if( volt_prop.getDoubleValue() < 9.0 and state != 0 ){
		state = 0;
	}
}


var power_btn = func {
	if( volt_prop.getDoubleValue() >= 9.0 and state == 0 ){
		switch_state();
	} else {
		check_electric_off();
	}
}

var btn = func{
	if( state == 0 ) power_btn();
	elsif( state <  5 ) switch_state();
}

var knob = func( dir ){
	if( state == 2 ){
		pilot_x += dir;
		if( pilot_x < 0 ) pilot_x = size( pilot_list ) - 1;
		Eos_start.on_pilot_change();
	} elsif( state == 3 ){
		Eos_start.on_elevation_change();
	} elsif( state == 4 ){
		qnh.setDoubleValue( qnh.getDoubleValue() + dir );
		if( qnh.getDoubleValue() < 950 ) qnh.setDoubleValue( 950.0 ) ;
		elsif( qnh.getDoubleValue() > 1050 ) qnh.setDoubleValue( 1050.0 );
		Eos_start.on_qnh_change();
	}
}


setlistener(volt_prop, func {
	check_electric_off();
});



#The following code is based on the ILEC SC7 e-vario and computes the different values shown by the display
io.include("Aircraft/Generic/soaring-instrumentation-sdk.nas");

####################################
####	INSTRUMENT SETUP	####
####################################

# Vario sound pitch controller by Colin Geniet (for ASK21), thanks!
#
# var vario_sound = SoundPitchController.new(
#   input: Object connected to the pitch controller input, e.g. a variometer reading.
#   max_pitch: (optional) Maximum sound frequency factor, the output will be
#              in the range [1/max_pitch, max_pitch], default 2.
#   max_input: Value of input for which max_pitch is reached.
#	on_update: (optional) function to call whenever a new output is available

var SoundPitchController = {
	parents: [InstrumentComponent],
	
	new: func(input, max_input, max_pitch = 2, on_update = nil) {
		return {
			parents: [me],
			input: input,
			max_pitch: max_pitch,
			max_input: max_input,
			on_update: on_update,
		};
	},
	
	update: func {
		var input = math.clamp(me.input.output, -me.max_input, me.max_input);
		me.output = math.pow(me.max_pitch, input / me.max_input);
		
		if (me.on_update != nil) me.on_update(me.output);
	},
};

var probe = TotalEnergyProbe.new();

var eos_needle = Dampener.new(
	input: probe,
	dampening: 1.5, 
	on_update: update_prop("/instrumentation/eos/te-reading-mps"));#1.5 is default dampening value according to POH

var averager = Averager.new(
	input: probe,
	buffer_size: 20,
	on_update: update_prop("/instrumentation/eos/te-average-mps")); #20s is default time according to POH
	
var eos_sound = SoundPitchController.new(
	input: eos_needle,
	max_input: 5,
	on_update: update_prop("/instrumentation/eos/sound-pitch"));

# Wrap everything together into an instrument
var fast_instruments = UpdateLoop.new(
	update_period: 0,
	components: [probe, eos_needle, eos_sound],
	enable: 1);

var slow_instruments = UpdateLoop.new(
	update_period: 1,
	components: [averager],
	enable: 1);
	
