/***************************************************************************
*   Copyright (C) 2010 by Peter Hatina                                    *
*   email: phatina (at) gmail.com                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU Lesser General Public License           *
*   version 2.1 as published by the Free Software Foundation              *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU Lesser General Public License for more details.                   *
*   http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.               *
***************************************************************************/

#include "tabwidgetitem.h"
#include <QFileDialog>
#include <QDirModel>

const int DEFAULT_LEVEL = 50;

TabWidgetItem::TabWidgetItem(QWidget *parent):
    QWidget(parent),
    m_btn_browse(new QPushButton("&Browse", this)),
    m_txt_command(new QLineEdit(this)),
    m_cmb_type(new QComboBox(this)),
    m_spn_level(new QSpinBox(this)),
    m_lbl_action(new QLabel("Action to execute", this)),
    m_lbl_level(new QLabel("Execute the action at battery level of", this)),
    m_completer(new QCompleter())
{
    // set completer model
    m_completer->setModel(new QDirModel(m_completer.data()));

    // set browse button tooltip
    m_btn_browse->setToolTip("Click to add a new action");

    // fill initial values of the gui components
    m_spn_level->setRange(0, 100);
    m_cmb_type->addItem("Show traybar message", Action::MESSAGE);
    m_cmb_type->addItem("Execute a program", Action::ACTION);
#ifdef Q_OS_WIN32
    m_cmb_type->addItem("Suspend the computer", Action::SUSPEND);
    m_cmb_type->addItem("Hibernate the computer", Action::HIBERNATE);
    m_cmb_type->addItem("Poweroff the computer", Action::POWEROFF);
#endif
    m_cmb_type->addItem("Do nothing", Action::NOTHING);
#if defined(Q_OS_WIN32)
    m_cmb_type->setCurrentIndex(5);
#elif defined(Q_OS_LINUX)
    m_cmb_type->setCurrentIndex(2);
#endif
    m_btn_browse->setMaximumWidth(70);
    m_btn_browse->setEnabled(false);
    m_spn_level->setValue(DEFAULT_LEVEL);
    m_txt_command->setEnabled(false);

    m_cmb_type->setGeometry(QRect(168, 89, 171, 20));
    m_spn_level->setGeometry(QRect(290, 56, 50, 20));
    m_spn_level->setMaximum(100);
    m_spn_level->setValue(50);
    m_txt_command->setGeometry(QRect(90, 120, 250, 20));
    m_btn_browse->setGeometry(QRect(271, 144, 75, 20));
#if (defined(Q_OS_WIN32))
    m_lbl_action->setGeometry(QRect(60, 90, 91, 16));
    m_lbl_level->setGeometry(QRect(59, 57, 180, 16));
#elif (defined(Q_OS_LINUX))
    m_lbl_action->setGeometry(QRect(50, 90, 100, 16));
    m_lbl_level->setGeometry(QRect(50, 57, 220, 16));
#endif

    connect(m_cmb_type.data(), SIGNAL(currentIndexChanged(int)), this, SLOT(onCmbChanged(int)));
    connect(m_btn_browse.data(), SIGNAL(clicked()), this, SLOT(onBrowseClicked()));
    connect(m_txt_command.data(), SIGNAL(textChanged(const QString &)),
            this, SLOT(onCommandChanged(const QString &)));
}

TabWidgetItem::~TabWidgetItem()
{
}

void TabWidgetItem::clear()
{
    m_txt_command->clear();
    m_cmb_type->setCurrentIndex(m_cmb_type->count() - 1);
    m_spn_level->setValue(DEFAULT_LEVEL);
}

int TabWidgetItem::getLevel() const
{
    return m_spn_level->value();
}

Action::type_t TabWidgetItem::getType() const
{
    const int ind = m_cmb_type->currentIndex();
    return static_cast <Action::type_t> (m_cmb_type->itemData(ind).toInt());
}

QString TabWidgetItem::getText() const
{
    return m_txt_command->text();
}

void TabWidgetItem::setLevel(int level)
{
    m_spn_level->setValue(level);
}

void TabWidgetItem::setType(Action::type_t type)
{
    m_cmb_type->setCurrentIndex(static_cast <int> (type));
}

void TabWidgetItem::setText(const QString &text)
{
    m_txt_command->setText(text);
}

void TabWidgetItem::onCmbChanged(int i)
{
    // execute a program
    m_btn_browse->setEnabled(i == 1);
    m_txt_command->setEnabled(i < 2);

    if (m_cmb_type->itemData(i).toInt() != Action::ACTION)
    {
        m_txt_command->setStyleSheet("color: black;");
        m_txt_command->setCompleter(NULL);
    }
    else
    {
        m_txt_command->setCompleter(m_completer.data());
        onCommandChanged("");
    }
}

void TabWidgetItem::onBrowseClicked()
{
    m_txt_command->setText(QFileDialog::getOpenFileName(this, "Select a file...", "."));
}

void TabWidgetItem::onCommandChanged(const QString &text)
{
    if (m_cmb_type->itemData(m_cmb_type->currentIndex()).toInt() == Action::ACTION)
    {
        QFileInfo file_info(text);
        bool valid = file_info.isFile();
        m_txt_command->setStyleSheet(valid ? "color: green;" : "color: red;");
        m_txt_command->setToolTip(text);
    }
}
