/**
 * Copyright (c) 2014,2017 Contributors to the Eclipse Foundation
 *
 * See the NOTICE file(s) distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.smarthome.config.dispatch.internal;

import static java.nio.file.StandardWatchEventKinds.*;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.WatchEvent;
import java.nio.file.WatchEvent.Kind;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.stream.Collectors;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.eclipse.smarthome.config.core.ConfigConstants;
import org.eclipse.smarthome.core.service.AbstractWatchService;
import org.osgi.framework.BundleContext;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.cm.ManagedService;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.JsonIOException;
import com.google.gson.JsonSyntaxException;

/**
 * This class provides a mean to read any kind of configuration data from
 * config folder files and dispatch it to the different bundles using the {@link ConfigurationAdmin} service.
 *
 * <p>
 * The name of the configuration folder can be provided as a program argument "smarthome.configdir" (default is "conf").
 * Configurations for OSGi services are kept in a subfolder that can be provided as a program argument
 * "smarthome.servicedir" (default is "services"). Any file in this folder with the extension .cfg will be processed.
 *
 * <p>
 * The format of the configuration file is similar to a standard property file, with the exception that the property
 * name can be prefixed by the service pid of the {@link ManagedService}:
 *
 * <p>
 * &lt;service-pid&gt;:&lt;property&gt;=&lt;value&gt;
 *
 * <p>
 * In case the pid does not contain any ".", the default service pid namespace is prefixed, which can be defined by the
 * program argument "smarthome.servicepid" (default is "org.eclipse.smarthome").
 *
 * <p>
 * If no pid is defined in the property line, the default pid namespace will be used together with the filename. E.g. if
 * you have a file "security.cfg", the pid that will be used is "org.eclipse.smarthome.security".
 *
 * <p>
 * Last but not least, a pid can be defined in the first line of a cfg file by prefixing it with "pid:", e.g.
 * "pid: com.acme.smarthome.security".
 *
 * @author Kai Kreuzer - Initial contribution and API
 * @author Petar Valchev - Added sort by modification time, when configuration files are read
 * @author Ana Dimova - Reduce to a single watch thread for all class instances
 * @author Henning Treu - Delete orphan exclusive configuration from configAdmin
 */
@Component(immediate = true)
public class ConfigDispatcher extends AbstractWatchService {

    private final Logger logger = LoggerFactory.getLogger(ConfigDispatcher.class);

    private final Gson gson = new Gson();

    /** The program argument name for setting the service config directory path */
    final static public String SERVICEDIR_PROG_ARGUMENT = "smarthome.servicedir";

    /** The program argument name for setting the service pid namespace */
    final static public String SERVICEPID_PROG_ARGUMENT = "smarthome.servicepid";

    /**
     * The program argument name for setting the default services config file
     * name
     */
    final static public String SERVICECFG_PROG_ARGUMENT = "smarthome.servicecfg";

    /** The default folder name of the configuration folder of services */
    final static public String SERVICES_FOLDER = "services";

    /** The default namespace for service pids */
    final static public String SERVICE_PID_NAMESPACE = "org.eclipse.smarthome";

    /** The default services configuration filename */
    final static public String SERVICE_CFG_FILE = "smarthome.cfg";

    private static final String PID_MARKER = "pid:";

    private static final String EXCLUSIVE_PID_STORE_FILE = "configdispatcher_pid_list.json";

    private ExclusivePIDMap exclusivePIDMap;

    private ConfigurationAdmin configAdmin;

    private File exclusivePIDStore;

    public ConfigDispatcher() {
        super(getPathToWatch());
    }

    @Activate
    public void activate(BundleContext bundleContext) {
        super.activate();
        exclusivePIDStore = bundleContext.getDataFile(EXCLUSIVE_PID_STORE_FILE);
        loadExclusivePIDList();
        readDefaultConfig();
        readConfigFiles();
        processOrphanExclusivePIDs();
        storeCurrentExclusivePIDList();
    }

    @Deactivate
    @Override
    public void deactivate() {
        super.deactivate();

    }

    @Reference(cardinality = ReferenceCardinality.MANDATORY, policy = ReferencePolicy.STATIC)
    protected void setConfigurationAdmin(ConfigurationAdmin configAdmin) {
        this.configAdmin = configAdmin;
    }

    protected void unsetConfigurationAdmin(ConfigurationAdmin configAdmin) {
        this.configAdmin = null;
    }

    @Override
    protected boolean watchSubDirectories() {
        return false;
    }

    @Override
    protected Kind<?>[] getWatchEventKinds(Path subDir) {
        return new Kind<?>[] { ENTRY_CREATE, ENTRY_DELETE, ENTRY_MODIFY };
    }

    @Override
    protected void processWatchEvent(WatchEvent<?> event, Kind<?> kind, Path path) {
        if (kind == ENTRY_CREATE || kind == ENTRY_MODIFY) {
            try {
                File f = path.toFile();
                if (!f.isHidden()) {
                    processConfigFile(f);
                }
            } catch (IOException e) {
                logger.warn("Could not process config file '{}': {}", path, e);
            }
        } else if (kind == ENTRY_DELETE) {
            // Detect if a service specific configuration file was removed. We want to
            // notify the service in this case with an updated empty configuration.
            File configFile = path.toFile();
            if (configFile.isHidden() || configFile.isDirectory() || !configFile.getName().endsWith(".cfg")) {
                return;
            }

            exclusivePIDMap.setFileRemoved(configFile.getAbsolutePath());
            processOrphanExclusivePIDs();
        }

        storeCurrentExclusivePIDList();
    }

    private void loadExclusivePIDList() {
        try (FileReader reader = new FileReader(exclusivePIDStore)) {
            exclusivePIDMap = gson.fromJson(reader, ExclusivePIDMap.class);
            if (exclusivePIDMap != null) {
                exclusivePIDMap.initializeProcessPIDMapping();
            }
        } catch (JsonSyntaxException | JsonIOException e) {
            logger.error("Error parsing exclusive pids from '{}': {}", exclusivePIDStore.getAbsolutePath(),
                    e.getMessage());
        } catch (IOException e) {
            logger.debug("Error loading exclusive pids from '{}': {}", exclusivePIDStore.getAbsolutePath(),
                    e.getMessage());
        } finally {
            if (exclusivePIDMap == null) {
                exclusivePIDMap = new ExclusivePIDMap();
            }
        }
    }

    private void storeCurrentExclusivePIDList() {
        try (FileWriter writer = new FileWriter(exclusivePIDStore)) {
            exclusivePIDMap.setCurrentExclusivePIDList();
            gson.toJson(exclusivePIDMap, writer);
        } catch (JsonIOException | IOException e) {
            logger.error("Error storing exclusive PID list in bundle data file: {}", e.getMessage());
        }
    }

    private void processOrphanExclusivePIDs() {
        for (String orphanPID : exclusivePIDMap.getOrphanPIDs()) {
            try {
                Configuration configuration = configAdmin.getConfiguration(orphanPID, null);
                configuration.delete();
                logger.debug("Deleting configuration for orphan pid {}", orphanPID);
            } catch (IOException e) {
                logger.error("Error deleting configuration for orphan pid {}.", orphanPID);
            }
        }
    }

    private static String getPathToWatch() {
        String progArg = System.getProperty(SERVICEDIR_PROG_ARGUMENT);
        if (progArg != null) {
            return ConfigConstants.getConfigFolder() + File.separator + progArg;
        } else {
            return ConfigConstants.getConfigFolder() + File.separator + SERVICES_FOLDER;
        }
    }

    private String getDefaultServiceConfigPath() {
        String progArg = System.getProperty(SERVICECFG_PROG_ARGUMENT);
        if (progArg != null) {
            return progArg;
        } else {
            return ConfigConstants.getConfigFolder() + File.separator + SERVICE_CFG_FILE;
        }
    }

    private void readDefaultConfig() {
        String defaultCfgPath = getDefaultServiceConfigPath();
        try {
            processConfigFile(new File(defaultCfgPath));
        } catch (IOException e) {
            logger.warn("Could not process default config file '{}': {}", defaultCfgPath, e.getMessage());
        }
    }

    private void readConfigFiles() {
        File dir = getSourcePath().toFile();
        if (dir.exists()) {
            File[] files = dir.listFiles();
            // Sort the files by modification time,
            // so that the last modified file is processed last.
            Arrays.sort(files, new Comparator<File>() {
                @Override
                public int compare(File left, File right) {
                    return Long.valueOf(left.lastModified()).compareTo(right.lastModified());
                }
            });
            for (File file : files) {
                try {
                    processConfigFile(file);
                } catch (IOException e) {
                    logger.warn("Could not process config file '{}': {}", file.getName(), e.getMessage());
                }
            }
        } else {
            logger.debug("Configuration folder '{}' does not exist.", dir.toString());
        }
    }

    private static String getServicePidNamespace() {
        String progArg = System.getProperty(SERVICEPID_PROG_ARGUMENT);
        if (progArg != null) {
            return progArg;
        } else {
            return SERVICE_PID_NAMESPACE;
        }
    }

    /**
     * The filename of a given configuration file is assumed to be the service PID. If the filename
     * without extension contains ".", we assume it is the fully qualified name.
     *
     * @param configFile The configuration file
     * @return The PID
     */
    private String pidFromFilename(File configFile) {
        String filenameWithoutExt = StringUtils.substringBeforeLast(configFile.getName(), ".");
        if (filenameWithoutExt.contains(".")) {
            // it is a fully qualified namespace
            return filenameWithoutExt;
        } else {
            return getServicePidNamespace() + "." + filenameWithoutExt;
        }
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    private void processConfigFile(File configFile) throws IOException, FileNotFoundException {
        if (configFile.isDirectory() || !configFile.getName().endsWith(".cfg")) {
            logger.debug("Ignoring file '{}'", configFile.getName());
            return;
        }
        logger.debug("Processing config file '{}'", configFile.getName());

        // we need to remember which configuration needs to be updated
        // because values have changed.
        Map<Configuration, Dictionary> configsToUpdate = new HashMap<Configuration, Dictionary>();

        // also cache the already retrieved configurations for each pid
        Map<Configuration, Dictionary> configMap = new HashMap<Configuration, Dictionary>();

        String pid = pidFromFilename(configFile);

        // configuration file contains a PID Marker
        List<String> lines = IOUtils.readLines(new FileInputStream(configFile));
        String exclusivePID = lines.size() > 0 ? getPIDFromLine(lines.get(0)) : null;
        if (exclusivePID != null) {
            if (exclusivePIDMap.contains(exclusivePID)) {
                logger.warn("The file {} subsequently defines the exclusive PID '{}'.", configFile.getAbsolutePath(),
                        exclusivePID);
            }
            pid = exclusivePID;
            lines = lines.subList(1, lines.size());
            exclusivePIDMap.setProcessedPID(pid, configFile.getAbsolutePath());
        } else if (exclusivePIDMap.contains(pid)) {
            // the pid was once from an exclusive file but there is either a second non-exclusive-file with config
            // entries or the `pid:` marker was removed.
            exclusivePIDMap.removeExclusivePID(pid);
        }

        Configuration configuration = configAdmin.getConfiguration(pid, null);

        // this file does only contain entries for this PID and no other files do contain further entries for this PID.
        if (exclusivePIDMap.contains(pid)) {
            configMap.put(configuration, new Properties());
        }

        for (String line : lines) {
            ParseLineResult parsedLine = parseLine(configFile.getPath(), line);
            // no valid configuration line, so continue
            if (parsedLine.isEmpty()) {
                continue;
            }

            if (exclusivePIDMap.contains(pid) && parsedLine.pid != null && !pid.equals(parsedLine.pid)) {
                logger.error("Error parsing config file {}. Exclusive PID {} found but line starts with {}.",
                        configFile.getName(), pid, parsedLine.pid);
                configuration.update((Dictionary) new Properties()); // update with empty properties
                return;
            }

            if (!exclusivePIDMap.contains(pid) && parsedLine.pid != null && exclusivePIDMap.contains(parsedLine.pid)) {
                logger.error(
                        "Error parsing config file {}. The PID {} is exclusive but defined in another file, skipping the line.",
                        configFile.getName(), parsedLine.pid);
                continue;
            }

            if (parsedLine.pid != null) {
                pid = parsedLine.pid;
                configuration = configAdmin.getConfiguration(pid, null);
            }

            Dictionary configProperties = configMap.get(configuration);
            if (configProperties == null) {
                configProperties = configuration.getProperties() != null ? configuration.getProperties()
                        : new Properties();
                configMap.put(configuration, configProperties);
            }
            if (!parsedLine.value.equals(configProperties.get(parsedLine.property))) {
                configProperties.put(parsedLine.property, parsedLine.value);
                configsToUpdate.put(configuration, configProperties);
            }
        }

        for (Entry<Configuration, Dictionary> entry : configsToUpdate.entrySet()) {
            entry.getKey().update(entry.getValue());
        }
    }

    private String getPIDFromLine(String line) {
        if (line.startsWith(PID_MARKER)) {
            return line.substring(PID_MARKER.length()).trim();
        }

        return null;
    }

    private ParseLineResult parseLine(final String filePath, final String line) {
        String trimmedLine = line.trim();
        if (trimmedLine.startsWith("#") || trimmedLine.isEmpty()) {
            return new ParseLineResult();
        }

        String pid = null; // no override of the pid is default
        String key = StringUtils.substringBefore(trimmedLine, "=");
        if (key.contains(":")) {
            pid = StringUtils.substringBefore(key, ":");
            trimmedLine = trimmedLine.substring(pid.length() + 1);
            pid = pid.trim();
            // PID is not fully qualified, so prefix with namespace
            if (!pid.contains(".")) {
                pid = getServicePidNamespace() + "." + pid;
            }
        }
        if (!trimmedLine.isEmpty() && trimmedLine.substring(1).contains("=")) {
            String property = StringUtils.substringBefore(trimmedLine, "=");
            String value = trimmedLine.substring(property.length() + 1);
            return new ParseLineResult(pid, property.trim(), value.trim());
        } else {
            logger.warn("Could not parse line '{}'", line);
            return new ParseLineResult();
        }
    }

    /**
     * Represents a result of parseLine().
     */
    private class ParseLineResult {
        public String pid;
        public String property;
        public String value;

        public ParseLineResult() {
            this(null, null, null);
        }

        public ParseLineResult(String pid, String property, String value) {
            this.pid = pid;
            this.property = property;
            this.value = value;
        }

        public boolean isEmpty() {
            return pid == null && property == null && value == null;
        }
    }

    /**
     * The {@link ExclusivePIDMap} serves two purposes:
     * 1. Store the exclusive PIDs which where processed by the {@link ConfigDispatcher} in the bundle data file in JSON
     * format.
     * 2. Map the processed PIDs to the absolute file paths of their config files. This way orphan PIDs from the bundle
     * data file will be recognised and their corresponding configuration will be deleted from configAdmin.
     */
    private class ExclusivePIDMap {

        /**
         * The list will be stored in the bundle cache and loaded on bundle start.
         * This way we can sync the processed files and delete all orphan configurations from the configAdmin.
         */
        private List<String> exclusivePIDs = new ArrayList<>();

        /**
         * The internal Map of PIDs to filenames will only be used during runtime to determine exclusively used
         * service config files.
         * The map will hold a 1:1 relation mapping from an exclusive PID to its absolute path in the file system.
         */
        private transient Map<String, String> processedPIDMapping = new HashMap<>();

        public void setProcessedPID(String pid, String pathToFile) {
            processedPIDMapping.put(pid, pathToFile);
        }

        public void removeExclusivePID(String pid) {
            processedPIDMapping.remove(pid);
        }

        public void setFileRemoved(String absolutePath) {
            for (Entry<String, String> entry : processedPIDMapping.entrySet()) {
                if (entry.getValue().equals(absolutePath)) {
                    entry.setValue(null);
                    return; // we expect a 1:1 relation between PID and path
                }
            }
        }

        public void initializeProcessPIDMapping() {
            processedPIDMapping = new HashMap<>();
            for (String pid : exclusivePIDs) {
                processedPIDMapping.put(pid, null);
            }
        }

        /**
         * Collect PIDs which where not processed (mapped path is null).
         *
         * @return the list of PIDs which where not processed either during #activate or on file deleted event.
         */
        public List<String> getOrphanPIDs() {
            return processedPIDMapping.entrySet().stream().filter(e -> e.getValue() == null).map(e -> e.getKey())
                    .collect(Collectors.toList());
        }

        /**
         * Set the exclusivePID list to the processed PIDs (mapped path is not null).
         */
        public void setCurrentExclusivePIDList() {
            exclusivePIDs = processedPIDMapping.entrySet().stream().filter(e -> e.getValue() != null)
                    .map(e -> e.getKey()).collect(Collectors.toList());
        }

        public boolean contains(String pid) {
            return processedPIDMapping.containsKey(pid);
        }
    }

}
