/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLParser.java,v 1.3 2007/03/18 09:07:06 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource.parser;

import java.util.List;

import org.eclipse.eodm.owl.resource.parser.exception.OWLParserException;
import org.eclipse.eodm.owl.resource.parser.impl.OWLOntologyGraph;

/**
 * OWLParser - OWL Semantics and Abstract Syntax Parser.
 * <p>
 * Parsing OWL documents in RDF/XML into ODM (Ontology Definition Meta-Model).
 * Semantics and Abstract Syntax is described at <a
 * href="http://www.w3.org/TR/owl-semantics/">OWL S&AS </a>.
 * 
 */
public interface OWLParser {
    /**
     * Add an OWL Document to the OWL Parser, which will be parsed into an OWL
     * Ontology by calling {@link #parseOWLDocument(OWLDocument) parse}method.
     * 
     * @param document
     *            the OWL Document to be added
     * @throws OWLParserException
     *             if the OWL Document already exists or the parsing fails
     */
    void addOWLDocument(OWLDocument document) throws OWLParserException;

    /**
     * Add a local file to the OWL Parser.
     * 
     * @see #addOWLDocument(OWLDocument)
     * @param localFileName
     *            the local file name
     * @throws OWLParserException
     *             if the OWL Document already exists or the parsing fails
     */
    void addOWLDocument(String localFileName) throws OWLParserException;

    /**
     * Parse OWL Document in RDF/XML into an OWL Ontology Definition Meta-Model.
     * If the document is not added prevously, then add it.
     * 
     * @param document
     *            the OWL Document as the input
     * @return the OWL Ontology Graph
     * @throws OWLParserException
     *             if the OWL parsing fails
     */
    OWLOntologyGraph[] parseOWLDocument(OWLDocument document)
            throws OWLParserException;

    /**
     * Parse a local file into an OWL Ontology.
     * 
     * @see #parseOWLDocument(OWLDocument)
     * @param localFileName
     *            the local file name
     * @return the OWL Ontology object
     * @throws OWLParserException
     *             if the OWL parsing fails
     */
    OWLOntologyGraph[] parseOWLDocument(String localFileName)
            throws OWLParserException;

    /**
     * Clear all OWL Documents and OWL Ontologies existing in the parser.
     * 
     */
    void clear();

    /**
     * Clear an OWL document from the parser
     * 
     * @param localFileName
     *            the name of the owl document
     */
    void clearDocument(String localFileName);

    /**
     * Get all OWL documents existing in the parser
     * 
     * @return a list of all documents
     */
    List getOWLDocuments();

    /**
     * Check if an OWL document with a specific URI exists in the parser
     * 
     * @param document
     *            the OWL document
     * @return true if the document exists, or false
     */
    OWLDocument getOWLDocument(OWLDocument document);
    
    /**
     * Addd a mapping between an ontology URL and local file name
     * @param url the url of the ontology
     * @param filename the name of the local file
     */
    void addURL2FileMapping(String url, String filename);
}