/*******************************************************************************
 * Copyright (c)2010 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *******************************************************************************/

package org.eclipse.birt.report.engine.emitter;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.birt.report.engine.content.IImageContent;
import org.eclipse.birt.report.engine.util.FlashFile;
import org.eclipse.birt.report.engine.util.SvgFile;
import org.eclipse.birt.report.model.api.IResourceLocator;
import org.eclipse.birt.report.model.api.ReportDesignHandle;

public class ImageReader
{

	public static final int TYPE_IMAGE_OBJECT = 0;
	public static final int TYPE_FLASH_OBJECT = 1;
	public static final int TYPE_SVG_OBJECT = 2;
	public static final int TYPE_CONVERTED_SVG_OBJECT = 3;

	public static final int OBJECT_UNLOADED = -1;
	public static final int RESOURCE_UNREACHABLE = 0;
	public static final int UNSUPPORTED_OBJECTS = 1;
	public static final int OBJECT_LOADED_SUCCESSFULLY = 2;

	private int objectType = TYPE_IMAGE_OBJECT;
	private int status = OBJECT_UNLOADED;

	private IImageContent content;
	private String supportedImageFormats = null;
	private byte[] buffer;

	protected static Logger logger = Logger.getLogger( ImageReader.class
			.getName( ) );

	public ImageReader( IImageContent content, String supportedImageFormats )
	{
		this.content = content;
		this.supportedImageFormats = supportedImageFormats;
	}

	public int read( )
	{
		buffer = null;
		checkObjectType( content );
		String uri = content.getURI( );
		try
		{
			switch ( content.getImageSource( ) )
			{
				case IImageContent.IMAGE_FILE :
					ReportDesignHandle design = content.getReportContent( )
							.getDesign( ).getReportDesign( );
					URL url = design
							.findResource( uri, IResourceLocator.IMAGE,
									content.getReportContent( )
											.getReportContext( ) == null
											? null
											: content.getReportContent( )
													.getReportContext( )
													.getAppContext( ) );
					readImage( url );
					break;
				case IImageContent.IMAGE_URL :
					readImage( uri );
					break;
				case IImageContent.IMAGE_NAME :
				case IImageContent.IMAGE_EXPRESSION :
					readImage( content.getData( ) );
					break;
				default :
					assert ( false );
			}
		}
		catch ( IOException e )
		{
			buffer = null;
			status = RESOURCE_UNREACHABLE;
			logger.log( Level.WARNING, e.getMessage( ), e );
		}
		return status;
	}

	public byte[] getByteArray( )
	{
		return buffer;
	}
	
	public int getType( )
	{
		return objectType;
	}

	private byte[] getImageByteArray( InputStream in ) throws IOException
	{
		ByteArrayOutputStream out = new ByteArrayOutputStream( );
		int data = in.read( );
		while ( data != -1 )
		{
			out.write( data );
			data = in.read( );
		}
		byte[] buffer = out.toByteArray( );
		out.close( );
		return buffer;
	}

	private void checkObjectType( IImageContent content )
	{
		String uri = content.getURI( );
		String mimeType = content.getMIMEType( );
		String extension = content.getExtension( );
		if ( FlashFile.isFlash( mimeType, uri, extension ) )
		{
			objectType = TYPE_FLASH_OBJECT;
		}
		else if ( SvgFile.isSvg( mimeType, uri, extension ) )
		{
			objectType = TYPE_SVG_OBJECT;
		}
		else
		{
			objectType = TYPE_IMAGE_OBJECT;
		}
	}

	/**
	 * Check if the target output emitter supports the object type.
	 */
	private boolean isOutputSupported( )
	{
		if ( objectType == TYPE_IMAGE_OBJECT )
		{
			if ( -1 != supportedImageFormats.indexOf( "PNG" )
					|| -1 != supportedImageFormats.indexOf( "GIF" )
					|| -1 != supportedImageFormats.indexOf( "BMP" )
					|| -1 != supportedImageFormats.indexOf( "JPG" ) )
				return true;
		}
		else if ( objectType == TYPE_FLASH_OBJECT )
		{
			if ( -1 != supportedImageFormats.indexOf( "SWF" ) )
			{
				return true;
			}
		}
		else if ( objectType == TYPE_SVG_OBJECT )
		{
			if ( -1 != supportedImageFormats.indexOf( "SVG" ) )
			{
				return true;
			}
		}
		return false;
	}

	private void readImage( String uri ) throws IOException
	{
		if( uri == null )
		{
			status = RESOURCE_UNREACHABLE;
			return;
		}
		readImage( new URL( uri ) );
	}

	private void readImage( URL url ) throws IOException
	{
		if( url == null )
		{
			status = RESOURCE_UNREACHABLE;
			return;
		}
		InputStream in = url.openStream( );
		readImage( in );
	}

	private void readImage( InputStream in ) throws IOException
	{
		if ( isOutputSupported( ) )
		{
			buffer = getImageByteArray( in );
			status = OBJECT_LOADED_SUCCESSFULLY;
		}
		else
		{
			if ( objectType == TYPE_SVG_OBJECT )
			{
				buffer = SvgFile.transSvgToArray( in );
				objectType = TYPE_CONVERTED_SVG_OBJECT;
				status = OBJECT_LOADED_SUCCESSFULLY;
			}
			else
			{
				buffer = null;
				status = UNSUPPORTED_OBJECTS;
			}
		}
	}

	private void readImage( byte[] data ) throws IOException
	{
		if ( data == null || data.length == 0 )
		{
			buffer = null;
			status = RESOURCE_UNREACHABLE;
			return;
		}
		if ( isOutputSupported( ) )
		{
			buffer = data;
			status = OBJECT_LOADED_SUCCESSFULLY;
		}
		else
		{
			if ( objectType == TYPE_SVG_OBJECT )
			{
				InputStream in = new ByteArrayInputStream( data );
				buffer = SvgFile.transSvgToArray( in );
				in.close( );
				objectType = TYPE_CONVERTED_SVG_OBJECT;
				status = OBJECT_LOADED_SUCCESSFULLY;
			}
			else
			{
				buffer = null;
				status = UNSUPPORTED_OBJECTS;
			}
		}
	}
}
