"use strict";
/*******************************************************************************
* Copyright (c) 2019 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v2.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v20.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const log = require("./Logger");
const VSCWatchedPath_1 = require("./VSCWatchedPath");
/**
 * This class works with the VSCode resource workbench monitoring functionality
 * to identify file/folder changes, and pass them to the FW core code.
 *
 * Specifically, a WatchEventEntry is created for each file/folder change, and the passes it to this class.
 * Next, this class passes it to the appropriate VSCWatchedPath, which then
 * processes it and forwards it to the core FW code.
 *
 * See IWatchService for more information on watch services.
 */
class VSCodeResourceWatchService {
    constructor() {
        this._projIdToWatchedPaths = new Map();
        this._disposed = false;
        /* empty */
    }
    addPath(fileToMonitor, ptw) {
        if (this._disposed) {
            return;
        }
        log.info("Path '" + fileToMonitor + "' added to watcher for " + ptw.projectId);
        const key = ptw.projectId;
        const value = this._projIdToWatchedPaths.get(key);
        if (value) {
            value.dispose();
        }
        this._projIdToWatchedPaths.set(key, new VSCWatchedPath_1.VSCWatchedPath(fileToMonitor, ptw, this));
    }
    removePath(fileToMonitor, oldProjectToWatch) {
        if (this._disposed) {
            return;
        }
        const key = oldProjectToWatch.projectId;
        const value = this._projIdToWatchedPaths.get(key);
        if (value) {
            log.info("Path '" + fileToMonitor + "' removed from " + key);
            value.dispose();
        }
        else {
            log.error("Path '" + fileToMonitor + "' attempted to be removed, but could not be found in " + key);
        }
    }
    receiveWatchEntries(cwProjectId, entries) {
        const wp = this._projIdToWatchedPaths.get(cwProjectId);
        if (wp) {
            wp.receiveFileChanges(entries);
        }
        else {
            // TODO: If this is printed for projects that are not managed by Codewind, then just comment this out.
            log.error("Could not find project with ID '" + cwProjectId + "' in list.");
        }
    }
    internal_handleEvent(event) {
        if (this._disposed) {
            return;
        }
        try {
            log.info("Received event " + event.toString());
            const timeReceived = Math.round(new Date().getTime());
            this._parent.receiveNewWatchEventEntry(event, timeReceived);
        }
        catch (e) {
            log.severe("handleEvent caught an uncaught exception.", e);
        }
    }
    setParent(parent) {
        this._parent = parent;
    }
    get parent() {
        return this._parent;
    }
    dispose() {
        this._disposed = true;
        this._projIdToWatchedPaths.clear();
        log.info("Dispose called.");
        for (const e of this._projIdToWatchedPaths.values()) {
            e.dispose();
        }
    }
    generateDebugState() {
        if (this._disposed) {
            return "[disposed]";
        }
        let result = "";
        for (const [key, value] of this._projIdToWatchedPaths) {
            result += "- " + key + " | " + value.pathRoot + "\n";
        }
        return result;
    }
}
exports.VSCodeResourceWatchService = VSCodeResourceWatchService;
