"use strict";
/*! Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.XhrRequest = void 0;
const mona_dish_1 = require("mona-dish");
const AjaxImpl_1 = require("../AjaxImpl");
const XhrFormData_1 = require("./XhrFormData");
const ErrorData_1 = require("./ErrorData");
const EventData_1 = require("./EventData");
const Lang_1 = require("../util/Lang");
const Const_1 = require("../core/Const");
const RequestDataResolver_1 = require("./RequestDataResolver");
var failSaveExecute = Lang_1.ExtLang.failSaveExecute;
/**
 * Faces XHR Request Wrapper
 * as AsyncRunnable for our Asynchronous queue
 *
 * The idea is that we basically just enqueue
 * a single ajax request into our queue
 * and let the queue do the processing.
 *
 *
 */
class XhrRequest {
    /**
     * Required Parameters
     *
     * @param source the issuing element
     * @param sourceForm the form which is related to the issuing element
     * @param requestContext the request context with all pass through values
     *
     * Optional Parameters
     *
     * @param internalContext internal context with internal info which is passed through, not used by the user
     * @param partialIdsArray an optional restricting partial ids array for encoding
     * @param timeout optional xhr timeout
     * @param ajaxType optional request type, default "POST"
     * @param contentType optional content type, default "application/x-www-form-urlencoded"
     * @param xhrObject optional xhr object which must fulfill the XMLHTTPRequest api, default XMLHttpRequest
     */
    constructor(source, sourceForm, requestContext, internalContext, partialIdsArray = [], timeout = Const_1.NO_TIMEOUT, ajaxType = Const_1.REQ_TYPE_POST, contentType = Const_1.URL_ENCODED, xhrObject = new XMLHttpRequest()) {
        this.source = source;
        this.sourceForm = sourceForm;
        this.requestContext = requestContext;
        this.internalContext = internalContext;
        this.partialIdsArray = partialIdsArray;
        this.timeout = timeout;
        this.ajaxType = ajaxType;
        this.contentType = contentType;
        this.xhrObject = xhrObject;
        this.stopProgress = false;
        /**
         * helper support so that we do not have to drag in Promise shims
         */
        this.catchFunctions = [];
        this.thenFunctions = [];
        // we omit promises here because we have to deal with cancel functionality,
        // and promises to not provide that (yet) instead we have our async queue
        // which uses an api internally, which is very close to promises
        this.registerXhrCallbacks((data) => {
            this.resolve(data);
        }, (data) => {
            this.reject(data);
        });
    }
    start() {
        let ignoreErr = failSaveExecute;
        let xhrObject = this.xhrObject;
        let executesArr = () => {
            return this.requestContext.getIf(Const_1.CTX_PARAM_REQ_PASS_THR, Const_1.P_EXECUTE).get(Const_1.IDENT_NONE).value.split(/\s+/gi);
        };
        try {
            // encoded we need to decode
            // We generated a base representation of the current form
            // in case someone has overloaded the viewState with additional decorators we merge
            // that in, there is no way around it, the spec allows it and getViewState
            // must be called, so whatever getViewState delivers has higher priority then
            // whatever the formData object delivers
            // the partialIdsArray arr is almost deprecated legacy code where we allowed to send a separate list of partial
            // ids for reduced load and server processing, this will be removed soon, we can handle the same via execute
            // anyway TODO reimplement the partial ids array, we still do not have it in jsf the way we need it
            let formData = new XhrFormData_1.XhrFormData(this.sourceForm, (0, RequestDataResolver_1.resoveNamingContainerMapper)(this.internalContext), executesArr(), this.partialIdsArray);
            this.contentType = formData.isMultipartRequest ? "undefined" : this.contentType;
            // next step the pass through parameters are merged in for post params
            this.requestContext.$nspEnabled = false;
            let requestContext = this.requestContext;
            let requestPassThroughParams = requestContext.getIf(Const_1.CTX_PARAM_REQ_PASS_THR);
            // we are turning off here the jsf, faces remapping because we are now dealing with
            // pass-through parameters
            requestPassThroughParams.$nspEnabled = false;
            // this is an extension where we allow pass through parameters to be sent down additionally
            // this can be used and is used in the impl to enrich the post request parameters with additional
            // information
            try {
                formData.shallowMerge(requestPassThroughParams, true, true);
            }
            finally {
                this.requestContext.$nspEnabled = true;
                requestPassThroughParams.$nspEnabled = true;
            }
            this.responseContext = requestPassThroughParams.deepCopy;
            // we have to shift the internal passthroughs around to build up our response context
            let responseContext = this.responseContext;
            responseContext.assign(Const_1.CTX_PARAM_MF_INTERNAL).value = this.internalContext.value;
            // per spec the onevent and onerror handlers must be passed through to the response
            responseContext.assign(Const_1.ON_EVENT).value = requestContext.getIf(Const_1.ON_EVENT).value;
            responseContext.assign(Const_1.ON_ERROR).value = requestContext.getIf(Const_1.ON_ERROR).value;
            xhrObject.open(this.ajaxType, (0, RequestDataResolver_1.resolveFinalUrl)(this.sourceForm, formData, this.ajaxType), true);
            // adding timeout
            this.timeout ? xhrObject.timeout = this.timeout : null;
            // a bug in the xhr stub library prevents the setRequestHeader to be properly executed on fake xhr objects
            // normal browsers should resolve this
            // tests can quietly fail on this one
            if (this.contentType != "undefined") {
                ignoreErr(() => xhrObject.setRequestHeader(Const_1.CONTENT_TYPE, `${this.contentType}; charset=utf-8`));
            }
            ignoreErr(() => xhrObject.setRequestHeader(Const_1.HEAD_FACES_REQ, Const_1.VAL_AJAX));
            // probably not needed anymore, will test this
            // some webkit based mobile browsers do not follow the w3c spec of
            // setting, they accept headers automatically
            ignoreErr(() => xhrObject.setRequestHeader(Const_1.REQ_ACCEPT, Const_1.STD_ACCEPT));
            this.sendEvent(Const_1.BEGIN);
            this.sendRequest(formData);
        }
        catch (e) {
            // _onError
            this.handleError(e);
        }
        return this;
    }
    cancel() {
        try {
            this.xhrObject.abort();
        }
        catch (e) {
            this.handleError(e);
        }
    }
    resolve(data) {
        mona_dish_1.Stream.of(...this.thenFunctions).reduce((inputVal, thenFunc) => {
            return thenFunc(inputVal);
        }, data);
    }
    reject(data) {
        mona_dish_1.Stream.of(...this.catchFunctions).reduce((inputVal, catchFunc) => {
            return catchFunc(inputVal);
        }, data);
    }
    catch(func) {
        this.catchFunctions.push(func);
        return this;
    }
    finally(func) {
        // no ie11 support we probably are going to revert to shims for that one
        this.catchFunctions.push(func);
        this.thenFunctions.push(func);
        return this;
    }
    then(func) {
        this.thenFunctions.push(func);
        return this;
    }
    /**
     * attaches the internal event and processing
     * callback within the promise to our xhr object
     *
     * @param resolve
     * @param reject
     */
    registerXhrCallbacks(resolve, reject) {
        let xhrObject = this.xhrObject;
        xhrObject.onabort = () => {
            this.onAbort(reject);
        };
        xhrObject.ontimeout = () => {
            this.onTimeout(reject);
        };
        xhrObject.onload = () => {
            this.onSuccess(resolve);
        };
        xhrObject.onloadend = () => {
            this.onDone(this.xhrObject, resolve);
        };
        xhrObject.onerror = (errorData) => {
            // some browsers trigger an error when cancelling a request internally
            // in this case we simply ignore the request and clear up the queue, because
            // it is not safe anymore to proceed with the current queue
            // This bypasses a Safari issue where it keeps requests hanging after page unload
            // and then triggers a cancel error on then instead of just stopping
            // and clearing the code
            if (this.isCancelledResponse(this.xhrObject)) {
                reject();
                this.stopProgress = true;
                return;
            }
            this.onError(errorData, reject);
        };
    }
    isCancelledResponse(currentTarget) {
        return (currentTarget === null || currentTarget === void 0 ? void 0 : currentTarget.status) === 0 && // cancelled by browser
            (currentTarget === null || currentTarget === void 0 ? void 0 : currentTarget.readyState) === 4 &&
            (currentTarget === null || currentTarget === void 0 ? void 0 : currentTarget.responseText) === '' &&
            (currentTarget === null || currentTarget === void 0 ? void 0 : currentTarget.responseXML) === null;
    }
    /*
         * xhr processing callbacks
         *
         * Those methods are the callbacks called by
         * the xhr object depending on its own state
         */
    onAbort(reject) {
        reject();
    }
    onTimeout(reject) {
        this.sendEvent(Const_1.STATE_EVT_TIMEOUT);
        reject();
    }
    onSuccess(resolve) {
        var _a, _b;
        this.sendEvent(Const_1.COMPLETE);
        // malformed responses always result in empty response xml
        // per spec a valid response cannot be empty
        if (!((_a = this === null || this === void 0 ? void 0 : this.xhrObject) === null || _a === void 0 ? void 0 : _a.responseXML)) {
            this.handleMalFormedXML(resolve);
            return;
        }
        (0, Const_1.$faces)().ajax.response(this.xhrObject, (_b = this.responseContext.value) !== null && _b !== void 0 ? _b : {});
    }
    handleMalFormedXML(resolve) {
        var _a;
        this.stopProgress = true;
        let errorData = {
            type: Const_1.ERROR,
            status: Const_1.MALFORMEDXML,
            responseCode: 200,
            responseText: (_a = this.xhrObject) === null || _a === void 0 ? void 0 : _a.responseText,
            // we remap the element just in case it gets replaced
            // it will be unremapped
            source: this.source.id.value
        };
        try {
            this.handleError(errorData, true);
        }
        finally {
            // we issue a resolve in this case to allow the system to recover
            // reject would clean up the queue
            resolve(errorData);
        }
        // non blocking non clearing
    }
    onDone(data, resolve) {
        // if stop progress a special handling including resolve is already performed
        if (this.stopProgress) {
            return;
        }
        resolve(data);
    }
    onError(errorData, reject) {
        this.handleError(errorData);
        reject();
    }
    sendRequest(formData) {
        let isPost = this.ajaxType != Const_1.REQ_TYPE_GET;
        if (formData.isMultipartRequest) {
            // in case of a multipart request we send in a formData object as body
            this.xhrObject.send((isPost) ? formData.toFormData() : null);
        }
        else {
            // in case of a normal request we send it normally
            this.xhrObject.send((isPost) ? formData.toString() : null);
        }
    }
    /*
     * other helpers
     */
    sendEvent(evtType) {
        var _a;
        let eventData = EventData_1.EventData.createFromRequest(this.xhrObject, this.requestContext, evtType);
        try {
            // User code error, we might cover
            // this in onError, but also we cannot swallow it.
            // We need to resolve the local handlers lazily,
            // because some frameworks might decorate them over the context in the response
            let eventHandler = (0, RequestDataResolver_1.resolveHandlerFunc)(this.requestContext, this.responseContext, Const_1.ON_EVENT);
            AjaxImpl_1.Implementation.sendEvent(eventData, eventHandler);
        }
        catch (e) {
            e.source = (_a = e === null || e === void 0 ? void 0 : e.source) !== null && _a !== void 0 ? _a : this.requestContext.getIf(Const_1.SOURCE).value;
            this.handleError(e);
            throw e;
        }
    }
    handleError(exception, responseFormatError = false) {
        let errorData = (responseFormatError) ? ErrorData_1.ErrorData.fromHttpConnection(exception.source, exception.type, exception.status, exception.responseText, exception.responseCode, exception.status) : ErrorData_1.ErrorData.fromClient(exception);
        let eventHandler = (0, RequestDataResolver_1.resolveHandlerFunc)(this.requestContext, this.responseContext, Const_1.ON_ERROR);
        AjaxImpl_1.Implementation.sendError(errorData, eventHandler);
    }
}
exports.XhrRequest = XhrRequest;
//# sourceMappingURL=XhrRequest.js.map