"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const entry_1 = require("./entry");
const virtual_1 = require("./virtual");
class FileSystemDirEntry extends virtual_1.VirtualDirEntry {
    constructor(_host, tree, _systemPath = '', path = core_1.normalize('/')) {
        super(tree, path);
        this._host = _host;
        this._systemPath = _systemPath;
    }
    _createDir(name) {
        return new FileSystemDirEntry(this._host, this._tree, this._host.join(this._systemPath, name), core_1.join(this._path, name));
    }
    get parent() {
        return this._path == '/' ? null : this._tree.getDir(core_1.dirname(this._path));
    }
    get subdirs() {
        const result = new Set();
        try {
            this._host.listDirectory(this._systemPath)
                .filter(name => this._host.isDirectory(this._host.join(this._systemPath, name)))
                .forEach(name => result.add(core_1.fragment(name)));
        }
        catch (e) {
            if (e.code != 'ENOENT' && e.code != 'ENOTDIR') {
                throw e;
            }
        }
        for (const path of this._tree.staging.keys()) {
            if (path.startsWith(this._path) && core_1.dirname(path) != this._path) {
                result.add(core_1.basename(path));
            }
        }
        return [...result];
    }
    get subfiles() {
        const result = new Set();
        try {
            this._host.listDirectory(this._systemPath)
                .filter(name => !this._host.isDirectory(this._host.join(this._systemPath, name)))
                .forEach(name => result.add(core_1.fragment(name)));
        }
        catch (e) {
            if (e.code != 'ENOENT' && e.code != 'ENOTDIR') {
                throw e;
            }
        }
        for (const path of this._tree.staging.keys()) {
            if (path.startsWith(this._path) && core_1.dirname(path) == this._path) {
                result.add(core_1.basename(path));
            }
        }
        return [...result];
    }
    file(name) {
        return this._tree.get(core_1.join(this._path, name));
    }
}
exports.FileSystemDirEntry = FileSystemDirEntry;
class FileSystemTree extends virtual_1.VirtualTree {
    constructor(_host) {
        super();
        this._host = _host;
        this._initialized = false;
        this._root = new FileSystemDirEntry(_host, this);
    }
    get tree() {
        const host = this._host;
        if (!this._initialized) {
            this._initialized = true;
            this._recursiveFileList().forEach(([system, schematic]) => {
                this._tree.set(schematic, new entry_1.LazyFileEntry(schematic, () => host.readFile(system)));
            });
        }
        return this._tree;
    }
    get(path) {
        const normalizedPath = this._normalizePath(path);
        let entry = this._cacheMap.get(normalizedPath) || this._tree.get(normalizedPath) || null;
        if (entry == null && !this._initialized) {
            const systemPath = normalizedPath;
            const fileExists = this._host.exists(systemPath) && !this._host.isDirectory(systemPath);
            if (fileExists) {
                const host = this._host;
                entry = new entry_1.LazyFileEntry(normalizedPath, () => host.readFile(systemPath));
                this._tree.set(normalizedPath, entry);
            }
        }
        return entry;
    }
    branch() {
        const newTree = new FileSystemTree(this._host);
        this._copyTo(newTree);
        return newTree;
    }
    _copyTo(tree) {
        if (tree instanceof FileSystemTree) {
            const x = tree;
            x._tree = this._tree;
            x._initialized = this._initialized;
            this._actions.forEach(action => x._actions.push(action));
            [...this._cacheMap.entries()].forEach(([path, entry]) => {
                x._cacheMap.set(path, entry);
            });
        }
        else {
            super._copyTo(tree);
        }
    }
    _recursiveFileList() {
        const host = this._host;
        const list = [];
        function recurse(systemPath, schematicPath) {
            for (const name of host.listDirectory(systemPath)) {
                const systemName = host.join(systemPath, name);
                const normalizedPath = core_1.normalize(schematicPath + '/' + name);
                if (host.isDirectory(normalizedPath)) {
                    recurse(systemName, normalizedPath);
                }
                else {
                    list.push([systemName, normalizedPath]);
                }
            }
        }
        recurse('', '/');
        return list;
    }
}
exports.FileSystemTree = FileSystemTree;
class FileSystemCreateTree extends FileSystemTree {
    constructor(host) {
        super(host);
        this._recursiveFileList().forEach(([system, schematic]) => {
            this.create(schematic, host.readFile(system));
        });
        this._initialized = true;
    }
}
exports.FileSystemCreateTree = FileSystemCreateTree;
//# sourceMappingURL=data:application/json;base64,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