"""Test suite for clustering algorithms."""

import numpy as np
import pytest

from sktime.datatypes import check_is_scitype
from sktime.tests.test_all_estimators import BaseFixtureGenerator, QuickTester
from sktime.utils._testing.scenarios_clustering import ClustererFitPredictMultivariate


class ClustererFixtureGenerator(BaseFixtureGenerator):
    """Fixture generator for clusterer tests.

    Fixtures parameterized
    ----------------------
    estimator_class: estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
    estimator_instance: instance of estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
        instances are generated by create_test_instance class method
    scenario: instance of TestScenario
        ranges over all scenarios returned by retrieve_scenarios
    """

    # note: this should be separate from TestAllClusterers
    #   additional fixtures, parameters, etc should be added here
    #   Custerer should contain the tests only

    estimator_type_filter = "clusterer"


class TestAllClusterers(ClustererFixtureGenerator, QuickTester):
    """Module level tests for all sktime clusterers."""

    def test_multivariate_input_exception(self, estimator_instance):
        """Test univariate clusterers raise exception on multivariate X."""
        # check if multivariate input raises error for univariate clusterers

        # if handles multivariate, no error is to be raised
        #   that clusterer works on multivariate data is tested in test_all_estimators
        if estimator_instance.get_tag("capability:multivariate"):
            return None

        error_msg = "multivariate"

        scenario = ClustererFitPredictMultivariate()

        # check if estimator raises appropriate error message
        #   composites will raise a warning, non-composites an exception
        if estimator_instance.is_composite():
            with pytest.warns(UserWarning, match=error_msg):
                scenario.run(estimator_instance, method_sequence=["fit"])
        else:
            with pytest.raises(ValueError, match=error_msg):
                scenario.run(estimator_instance, method_sequence=["fit"])

    def test_clusterer_output(self, estimator_instance, scenario):
        """Test clusterer outputs the correct data types and values.

        Test predict produces a np.array or pd.Series with only values seen in the train
        data, and that predict_proba probability estimates add up to one.
        """
        X_new = scenario.args["predict"]["X"]
        # we use check_is_scitype to get the number instances in X_new
        #   this is more robust against different scitypes in X_new
        _, _, X_new_metadata = check_is_scitype(X_new, "Panel", return_metadata=True)
        X_new_instances = X_new_metadata["n_instances"]

        # run fit
        scenario.run(estimator_instance, method_sequence=["fit"])

        # only clusterers with capability:predict have predict
        if not estimator_instance.get_tag("capability:predict"):
            return None

        # run predict
        y_pred = scenario.run(estimator_instance, method_sequence=["predict"])

        shape_msg = "Incorrect shape returned by predict"

        # check predict
        msg = f"Incorrect type returned by predict ({type(y_pred)})"
        assert isinstance(y_pred, np.ndarray), msg
        assert y_pred.shape == (X_new_instances,), shape_msg

        # check predict proba (all clusterers have predict_proba by default)
        y_proba = scenario.run(estimator_instance, method_sequence=["predict_proba"])
        msg = f"Incorrect type returned by predict_proba ({type(y_proba)})"
        assert isinstance(y_proba, np.ndarray), msg
        assert y_proba.shape[0] == X_new_instances, shape_msg

        if hasattr(estimator_instance, "n_clusters") and estimator_instance.n_clusters:
            assert y_proba.shape[1] == estimator_instance.n_clusters

        # If a sample doesn't belong to a cluster, it may not contribute to the total
        # probability. Check that total probability in range [0,1] + rounding error
        msg = "Total assigned probability exceeds 1"
        assert np.all(y_proba.sum(axis=1) < 1.0001), msg
        msg = "Total assigned probability negative"
        assert np.all(y_proba.sum(axis=1) >= 0), msg
