/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.Authorizator;
import io.moquette.broker.ClientDescriptor;
import io.moquette.broker.IQueueRepository;
import io.moquette.broker.ISessionsRepository;
import io.moquette.broker.InMemoryQueue;
import io.moquette.broker.MQTTConnection;
import io.moquette.broker.Session;
import io.moquette.broker.SessionCorruptedException;
import io.moquette.broker.SessionEventLoopGroup;
import io.moquette.broker.SessionMessageQueue;
import io.moquette.broker.Utils;
import io.moquette.broker.scheduler.ScheduledExpirationService;
import io.moquette.broker.subscriptions.ISubscriptionsDirectory;
import io.moquette.broker.subscriptions.Subscription;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttProperties;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.codec.mqtt.MqttVersion;
import java.net.InetSocketAddress;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionRegistry {
    private int globalExpirySeconds;
    private final SessionEventLoopGroup loopsGroup;
    static final Duration EXPIRED_SESSION_CLEANER_TASK_INTERVAL = Duration.ofSeconds(1L);
    private ScheduledExpirationService<ISessionsRepository.SessionData> sessionExpirationService;
    private static final Logger LOG = LoggerFactory.getLogger(SessionRegistry.class);
    private final ConcurrentMap<String, Session> pool = new ConcurrentHashMap<String, Session>();
    private final ISubscriptionsDirectory subscriptionsDirectory;
    private final ISessionsRepository sessionsRepository;
    private final IQueueRepository queueRepository;
    private final Authorizator authorizator;
    private final Clock clock;

    SessionRegistry(ISubscriptionsDirectory subscriptionsDirectory, ISessionsRepository sessionsRepository, IQueueRepository queueRepository, Authorizator authorizator, ScheduledExecutorService scheduler, SessionEventLoopGroup loopsGroup) {
        this(subscriptionsDirectory, sessionsRepository, queueRepository, authorizator, scheduler, Clock.systemDefaultZone(), Integer.MAX_VALUE, loopsGroup);
    }

    SessionRegistry(ISubscriptionsDirectory subscriptionsDirectory, ISessionsRepository sessionsRepository, IQueueRepository queueRepository, Authorizator authorizator, ScheduledExecutorService scheduler, Clock clock, int globalExpirySeconds, SessionEventLoopGroup loopsGroup) {
        this.subscriptionsDirectory = subscriptionsDirectory;
        this.sessionsRepository = sessionsRepository;
        this.queueRepository = queueRepository;
        this.authorizator = authorizator;
        this.sessionExpirationService = new ScheduledExpirationService<ISessionsRepository.SessionData>(clock, this::removeExpiredSession);
        this.clock = clock;
        this.globalExpirySeconds = globalExpirySeconds;
        this.loopsGroup = loopsGroup;
        this.recreateSessionPool();
    }

    private void removeExpiredSession(ISessionsRepository.SessionData expiredSession) {
        String expiredAt = expiredSession.expireAt().map(Instant::toString).orElse("UNDEFINED");
        LOG.debug("Removing session {}, expired on {}", (Object)expiredSession.clientId(), (Object)expiredAt);
        this.remove(expiredSession.clientId());
        this.sessionsRepository.delete(expiredSession);
        this.subscriptionsDirectory.removeSharedSubscriptionsForClient(expiredSession.clientId());
    }

    private void trackForRemovalOnExpiration(ISessionsRepository.SessionData session) {
        LOG.debug("start tracking the session {} for removal", (Object)session.clientId());
        this.sessionExpirationService.track(session.clientId(), session);
    }

    private void untrackFromRemovalOnExpiration(ISessionsRepository.SessionData session) {
        this.sessionExpirationService.untrack(session.clientId());
    }

    private void recreateSessionPool() {
        Set<String> queues = this.queueRepository.listQueueNames();
        for (ISessionsRepository.SessionData session : this.sessionsRepository.list()) {
            if (!this.queueRepository.containsQueue(session.clientId())) continue;
            SessionMessageQueue<EnqueuedMessage> persistentQueue = this.queueRepository.getOrCreateQueue(session.clientId());
            queues.remove(session.clientId());
            Session rehydrated = new Session(session, false, persistentQueue);
            this.pool.put(session.clientId(), rehydrated);
            this.trackForRemovalOnExpiration(session);
        }
        if (!queues.isEmpty()) {
            LOG.error("Recreating sessions left {} unused queues. This is probably a bug. Session IDs: {}", (Object)queues.size(), (Object)Arrays.toString(queues.toArray()));
        }
    }

    SessionCreationResult createOrReopenSession(MqttConnectMessage msg, String clientId, String username) {
        SessionCreationResult postConnectAction;
        Session oldSession = this.retrieve(clientId);
        if (oldSession == null) {
            Session newSession = this.createNewSession(msg, clientId);
            postConnectAction = new SessionCreationResult(newSession, CreationModeEnum.CREATED_CLEAN_NEW, false);
            Session previous = this.pool.put(clientId, newSession);
            if (previous != null) {
                LOG.error("Another thread added a Session for our clientId {}, this is a bug!", (Object)clientId);
            }
            LOG.trace("case 1, not existing session with CId {}", (Object)clientId);
        } else {
            postConnectAction = this.reopenExistingSession(msg, clientId, oldSession, username);
        }
        return postConnectAction;
    }

    private SessionCreationResult reopenExistingSession(MqttConnectMessage msg, String clientId, Session oldSession, String username) {
        SessionCreationResult creationResult;
        boolean newIsClean = msg.variableHeader().isCleanSession();
        if (!oldSession.disconnected()) {
            oldSession.closeImmediately();
        }
        if (newIsClean) {
            this.purgeSessionState(oldSession);
            Session newSession = this.createNewSession(msg, clientId);
            this.pool.put(clientId, newSession);
            LOG.trace("case 2, oldSession with same CId {} disconnected", (Object)clientId);
            creationResult = new SessionCreationResult(newSession, CreationModeEnum.CREATED_CLEAN_NEW, true);
        } else {
            boolean connecting = oldSession.assignState(Session.SessionStatus.DISCONNECTED, Session.SessionStatus.CONNECTING);
            if (!connecting) {
                throw new SessionCorruptedException("old session moved in connected state by other thread");
            }
            boolean hasWillFlag = msg.variableHeader().isWillFlag();
            ISessionsRepository.SessionData newSessionData = oldSession.getSessionData();
            newSessionData = hasWillFlag ? newSessionData.withWill(this.createNewWill(msg)) : newSessionData.withoutWill();
            oldSession.updateSessionData(newSessionData);
            oldSession.markAsNotClean();
            this.reactivateSubscriptions(oldSession, username);
            LOG.trace("case 3, oldSession with same CId {} disconnected", (Object)clientId);
            creationResult = new SessionCreationResult(oldSession, CreationModeEnum.REOPEN_EXISTING, true);
        }
        this.untrackFromRemovalOnExpiration(creationResult.session.getSessionData());
        return creationResult;
    }

    private void reactivateSubscriptions(Session session, String username) {
        for (Subscription existingSub : session.getSubscriptions()) {
            boolean topicReadable = this.authorizator.canRead(existingSub.getTopicFilter(), username, session.getClientID());
            if (topicReadable) continue;
            this.subscriptionsDirectory.removeSubscription(existingSub.getTopicFilter(), session.getClientID());
        }
    }

    private void unsubscribe(Session session) {
        for (Subscription existingSub : session.getSubscriptions()) {
            this.subscriptionsDirectory.removeSubscription(existingSub.getTopicFilter(), session.getClientID());
        }
    }

    private Session createNewSession(MqttConnectMessage msg, String clientId) {
        ISessionsRepository.SessionData sessionData;
        int expiryInterval;
        boolean clean = msg.variableHeader().isCleanSession();
        SessionMessageQueue<EnqueuedMessage> queue = !clean ? this.queueRepository.getOrCreateQueue(clientId) : new InMemoryQueue();
        MqttVersion mqttVersion = Utils.versionFromConnect(msg);
        if (mqttVersion != MqttVersion.MQTT_5) {
            expiryInterval = clean ? 0 : this.globalExpirySeconds;
        } else {
            MqttProperties.MqttProperty expiryIntervalProperty = msg.variableHeader().properties().getProperty(MqttProperties.MqttPropertyType.SESSION_EXPIRY_INTERVAL.value());
            if (expiryIntervalProperty != null) {
                int preferredExpiryInterval = (Integer)expiryIntervalProperty.value();
                expiryInterval = Math.min(preferredExpiryInterval, this.globalExpirySeconds);
            } else {
                int n = expiryInterval = clean ? 0 : this.globalExpirySeconds;
            }
        }
        if (msg.variableHeader().isWillFlag()) {
            ISessionsRepository.Will will = this.createNewWill(msg);
            sessionData = new ISessionsRepository.SessionData(clientId, mqttVersion, will, expiryInterval, this.clock);
        } else {
            sessionData = new ISessionsRepository.SessionData(clientId, mqttVersion, expiryInterval, this.clock);
        }
        Session newSession = new Session(sessionData, clean, queue);
        newSession.markConnecting();
        this.sessionsRepository.saveSession(sessionData);
        if (MQTTConnection.isNeedResponseInformation(msg)) {
            this.authorizator.forceReadAccess(Topic.asTopic("/reqresp/response/" + clientId), clientId);
            this.authorizator.forceWriteToAll(Topic.asTopic("/reqresp/response/" + clientId));
        }
        return newSession;
    }

    private ISessionsRepository.Will createNewWill(MqttConnectMessage msg) {
        List userProperties;
        MqttProperties.MqttProperty correlationDataProperty;
        MqttProperties.MqttProperty responseTopicProperty;
        MqttProperties.MqttProperty contentTypeProperty;
        byte[] willPayload = msg.payload().willMessageInBytes();
        String willTopic = msg.payload().willTopic();
        boolean retained = msg.variableHeader().isWillRetain();
        MqttQoS qos = MqttQoS.valueOf((int)msg.variableHeader().willQos());
        if (Utils.versionFromConnect(msg) != MqttVersion.MQTT_5) {
            return new ISessionsRepository.Will(willTopic, willPayload, qos, retained, 0);
        }
        MqttProperties willProperties = msg.payload().willProperties();
        MqttProperties.MqttProperty willDelayIntervalProperty = willProperties.getProperty(MqttProperties.MqttPropertyType.WILL_DELAY_INTERVAL.value());
        int willDelayIntervalSeconds = willDelayIntervalProperty != null ? (Integer)willDelayIntervalProperty.value() : 0;
        ISessionsRepository.Will will = new ISessionsRepository.Will(willTopic, willPayload, qos, retained, willDelayIntervalSeconds);
        ISessionsRepository.WillOptions options = ISessionsRepository.WillOptions.empty();
        MqttProperties.MqttProperty messageExpiryIntervalProperty = willProperties.getProperty(MqttProperties.MqttPropertyType.PUBLICATION_EXPIRY_INTERVAL.value());
        if (messageExpiryIntervalProperty != null) {
            Integer messageExpiryIntervalSeconds = (Integer)messageExpiryIntervalProperty.value();
            options = options.withMessageExpiry(Duration.ofSeconds(messageExpiryIntervalSeconds.intValue()));
        }
        if ((contentTypeProperty = willProperties.getProperty(MqttProperties.MqttPropertyType.CONTENT_TYPE.value())) != null) {
            options = options.withContentType((String)contentTypeProperty.value());
        }
        if ((responseTopicProperty = willProperties.getProperty(MqttProperties.MqttPropertyType.RESPONSE_TOPIC.value())) != null) {
            options = options.withResponseTopic((String)responseTopicProperty.value());
        }
        if ((correlationDataProperty = willProperties.getProperty(MqttProperties.MqttPropertyType.CORRELATION_DATA.value())) != null) {
            options = options.withCorrelationData((byte[])correlationDataProperty.value());
        }
        if ((userProperties = willProperties.getProperties(MqttProperties.MqttPropertyType.USER_PROPERTY.value())) != null && !userProperties.isEmpty()) {
            HashMap<String, String> props = new HashMap<String, String>(userProperties.size());
            for (MqttProperties.UserProperty userProperty : userProperties) {
                props.put(((MqttProperties.StringPair)userProperty.value()).key, ((MqttProperties.StringPair)userProperty.value()).value);
            }
            options = options.withUserProperties(props);
        }
        if (options.notEmpty()) {
            return new ISessionsRepository.Will(will, options);
        }
        return will;
    }

    Session retrieve(String clientID) {
        return (Session)this.pool.get(clientID);
    }

    void connectionClosed(Session session) {
        session.disconnect();
        if (session.expireImmediately()) {
            this.purgeSessionState(session);
        } else {
            ISessionsRepository.SessionData sessionData = session.getSessionData().withExpirationComputed();
            this.trackForRemovalOnExpiration(sessionData);
        }
    }

    private void purgeSessionState(Session session) {
        LOG.debug("Remove session state for client {}", (Object)session.getClientID());
        boolean result = session.assignState(Session.SessionStatus.DISCONNECTED, Session.SessionStatus.DESTROYED);
        if (!result) {
            throw new SessionCorruptedException("Session has already changed state: " + session);
        }
        this.unsubscribe(session);
        this.remove(session.getClientID());
        this.subscriptionsDirectory.removeSharedSubscriptionsForClient(session.getClientID());
    }

    void remove(String clientID) {
        Session old = (Session)this.pool.remove(clientID);
        if (old != null) {
            this.sessionExpirationService.untrack(clientID);
            this.loopsGroup.routeCommand(clientID, "Clean up removed session", () -> {
                old.cleanUp();
                return null;
            });
        }
    }

    Collection<ClientDescriptor> listConnectedClients() {
        return this.pool.values().stream().filter(Session::connected).map(this::createClientDescriptor).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
    }

    boolean dropSession(String clientId, boolean removeSessionState) {
        LOG.debug("Disconnecting client: {}", (Object)clientId);
        if (clientId == null) {
            return false;
        }
        Session client = (Session)this.pool.get(clientId);
        if (client == null) {
            LOG.debug("Client {} not found, nothing disconnected", (Object)clientId);
            return false;
        }
        client.closeImmediately();
        if (removeSessionState) {
            this.purgeSessionState(client);
        }
        LOG.debug("Client {} successfully disconnected from broker", (Object)clientId);
        return true;
    }

    private Optional<ClientDescriptor> createClientDescriptor(Session s) {
        String clientID = s.getClientID();
        Optional<InetSocketAddress> remoteAddressOpt = s.remoteAddress();
        return remoteAddressOpt.map(r -> new ClientDescriptor(clientID, r.getHostString(), r.getPort()));
    }

    public void close() {
        this.sessionExpirationService.shutdown();
        this.updateNotCleanSessionsWithProperExpire();
        this.queueRepository.close();
        this.pool.values().forEach(Session::cleanUp);
    }

    private void updateNotCleanSessionsWithProperExpire() {
        this.pool.values().stream().filter(s -> !s.isClean()).map(Session::getSessionData).filter(s -> !s.expireAt().isPresent()).map(ISessionsRepository.SessionData::withExpirationComputed).forEach(this.sessionsRepository::saveSession);
    }

    public static class SessionCreationResult {
        final Session session;
        final CreationModeEnum mode;
        final boolean alreadyStored;

        public SessionCreationResult(Session session, CreationModeEnum mode, boolean alreadyStored) {
            this.session = session;
            this.mode = mode;
            this.alreadyStored = alreadyStored;
        }
    }

    public static enum CreationModeEnum {
        CREATED_CLEAN_NEW,
        REOPEN_EXISTING,
        DROP_EXISTING;

    }

    public static final class PubRelMarker
    extends EnqueuedMessage {
        public String toString() {
            return "PubRelMarker{}";
        }
    }

    public static class PublishedMessage
    extends EnqueuedMessage {
        final Topic topic;
        final MqttQoS publishingQos;
        final ByteBuf payload;
        final boolean retained;
        final Instant messageExpiry;
        final MqttProperties.MqttProperty[] mqttProperties;

        public PublishedMessage(Topic topic, MqttQoS publishingQos, ByteBuf payload, boolean retained, Instant messageExpiry, MqttProperties.MqttProperty ... mqttProperties) {
            this.topic = topic;
            this.publishingQos = publishingQos;
            this.payload = payload;
            this.retained = retained;
            this.messageExpiry = messageExpiry;
            this.mqttProperties = mqttProperties;
        }

        public Topic getTopic() {
            return this.topic;
        }

        public MqttQoS getPublishingQos() {
            return this.publishingQos;
        }

        public ByteBuf getPayload() {
            return this.payload;
        }

        @Override
        public void release() {
            this.payload.release();
        }

        @Override
        public void retain() {
            this.payload.retain();
        }

        public MqttProperties.MqttProperty[] getMqttProperties() {
            return this.mqttProperties;
        }

        public boolean isExpired() {
            return this.messageExpiry != Instant.MAX && Instant.now().isAfter(this.messageExpiry);
        }

        public MqttProperties.MqttProperty[] updatePublicationExpiryIfPresentOrAdd() {
            if (this.messageExpiry == Instant.MAX) {
                return this.mqttProperties;
            }
            Duration duration = Duration.between(Instant.now(), this.messageExpiry);
            long remainingSeconds = Math.round((double)duration.toMillis() / 1000.0);
            int indexOfExpiry = PublishedMessage.findPublicationExpiryProperty(this.mqttProperties);
            MqttProperties.IntegerProperty updatedProperty = new MqttProperties.IntegerProperty(MqttProperties.MqttPropertyType.PUBLICATION_EXPIRY_INTERVAL.value(), Integer.valueOf((int)remainingSeconds));
            if (indexOfExpiry != -1) {
                this.mqttProperties[indexOfExpiry] = updatedProperty;
                return this.mqttProperties;
            }
            MqttProperties.MqttProperty[] newProperties = Arrays.copyOf(this.mqttProperties, this.mqttProperties.length + 1);
            newProperties[newProperties.length - 1] = updatedProperty;
            return newProperties;
        }

        private static int findPublicationExpiryProperty(MqttProperties.MqttProperty[] properties) {
            for (int i = 0; i < properties.length; ++i) {
                if (!PublishedMessage.isPublicationExpiryProperty(properties[i])) continue;
                return i;
            }
            return -1;
        }

        private static boolean isPublicationExpiryProperty(MqttProperties.MqttProperty property) {
            return property instanceof MqttProperties.IntegerProperty && property.propertyId() == MqttProperties.MqttPropertyType.PUBLICATION_EXPIRY_INTERVAL.value();
        }

        public Instant getMessageExpiry() {
            return this.messageExpiry;
        }

        public String toString() {
            return "PublishedMessage{topic=" + this.topic + ", publishingQos=" + this.publishingQos + ", payload=" + this.payload + ", retained=" + this.retained + ", messageExpiry=" + this.messageExpiry + ", mqttProperties=" + Arrays.toString(this.mqttProperties) + '}';
        }
    }

    public static abstract class EnqueuedMessage {
        public void release() {
        }

        public void retain() {
        }
    }
}

