/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.raft.jraft.storage.logit.storage.db;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.ignite3.internal.logger.IgniteLogger;
import org.apache.ignite3.internal.logger.Loggers;
import org.apache.ignite3.raft.jraft.Lifecycle;
import org.apache.ignite3.raft.jraft.entity.LogEntry;
import org.apache.ignite3.raft.jraft.entity.codec.LogEntryDecoder;
import org.apache.ignite3.raft.jraft.entity.codec.LogEntryEncoder;
import org.apache.ignite3.raft.jraft.entity.codec.v1.V1Encoder;
import org.apache.ignite3.raft.jraft.storage.logit.option.StoreOptions;
import org.apache.ignite3.raft.jraft.storage.logit.storage.factory.LogStoreFactory;
import org.apache.ignite3.raft.jraft.storage.logit.storage.file.AbstractFile;
import org.apache.ignite3.raft.jraft.storage.logit.storage.file.FileManager;
import org.apache.ignite3.raft.jraft.storage.logit.storage.file.FileType;
import org.apache.ignite3.raft.jraft.storage.logit.storage.file.assit.AbortFile;
import org.apache.ignite3.raft.jraft.storage.logit.storage.file.assit.FlushStatusCheckpoint;
import org.apache.ignite3.raft.jraft.storage.logit.storage.file.segment.SegmentFile;
import org.apache.ignite3.raft.jraft.storage.logit.storage.service.ServiceManager;
import org.apache.ignite3.raft.jraft.storage.logit.util.Pair;

public abstract class AbstractDB
implements Lifecycle<LogStoreFactory> {
    private static final IgniteLogger LOG = Loggers.forClass(AbstractDB.class);
    private static final String FLUSH_STATUS_CHECKPOINT = "FlushStatusCheckpoint";
    private static final String ABORT_FILE = "Abort";
    protected final String storePath;
    protected FileManager fileManager;
    protected ServiceManager serviceManager;
    protected LogStoreFactory logStoreFactory;
    protected StoreOptions storeOptions;
    protected AbortFile abortFile;
    protected FlushStatusCheckpoint flushStatusCheckpoint;
    private final ScheduledExecutorService checkpointExecutor;
    private ScheduledFuture<?> checkpointScheduledFuture;

    protected AbstractDB(String storePath, ScheduledExecutorService checkpointExecutor) {
        this.storePath = storePath;
        this.checkpointExecutor = checkpointExecutor;
    }

    @Override
    public boolean init(LogStoreFactory logStoreFactory) {
        this.logStoreFactory = logStoreFactory;
        this.storeOptions = logStoreFactory.getStoreOptions();
        String flushStatusCheckpointPath = Paths.get(this.storePath, FLUSH_STATUS_CHECKPOINT).toString();
        String abortFilePath = Paths.get(this.storePath, ABORT_FILE).toString();
        this.flushStatusCheckpoint = new FlushStatusCheckpoint(flushStatusCheckpointPath, logStoreFactory.getRaftOptions());
        this.abortFile = new AbortFile(abortFilePath);
        this.serviceManager = logStoreFactory.newServiceManager(this);
        if (!this.serviceManager.init(logStoreFactory)) {
            return false;
        }
        this.fileManager = logStoreFactory.newFileManager(this.getDBFileType(), this.storePath, this.serviceManager.getAllocateService());
        int interval = this.storeOptions.getCheckpointFlushStatusInterval();
        this.checkpointScheduledFuture = this.checkpointExecutor.scheduleAtFixedRate(this::doCheckpoint, interval, interval, TimeUnit.MILLISECONDS);
        return true;
    }

    @Override
    public void shutdown() {
        this.checkpointScheduledFuture.cancel(false);
        this.doCheckpoint();
        if (this.serviceManager != null) {
            this.serviceManager.shutdown();
        }
        if (this.fileManager != null) {
            this.fileManager.shutdown();
        }
        if (this.abortFile != null) {
            this.abortFile.destroy();
        }
    }

    public String getDBName() {
        return this.getClass().getSimpleName();
    }

    public abstract FileType getDBFileType();

    public abstract int getDBFileSize();

    public LogEntryIterator iterator(LogEntryDecoder logEntryDecoder, long beginIndex, int beginPosition) {
        AbstractFile[] files = this.fileManager.findFileFromLogIndex(beginIndex);
        return new LogEntryIterator(files, logEntryDecoder, beginPosition);
    }

    public LogEntryIterator iterator(LogEntryDecoder logEntryDecoder) {
        AbstractFile[] files = this.fileManager.copyFiles();
        return new LogEntryIterator(files, logEntryDecoder, 0);
    }

    public synchronized void recover() {
        List<AbstractFile> files = this.fileManager.loadExistedFiles();
        try {
            int startRecoverIndex;
            boolean normalExit;
            if (files.isEmpty()) {
                this.fileManager.setFlushedPosition(0L);
                this.abortFile.create();
                return;
            }
            this.flushStatusCheckpoint.load();
            boolean bl = normalExit = !this.abortFile.exists();
            if (!normalExit) {
                startRecoverIndex = this.findLastCheckpointFile(files, this.flushStatusCheckpoint);
                LOG.info("{} {} did not exit normally, will try to recover files from fileIndex:{}.", this.getDBName(), this.storePath, startRecoverIndex);
            } else {
                startRecoverIndex = files.size() - 1;
            }
            long recoverOffset = (long)startRecoverIndex * (long)this.getDBFileSize();
            recoverOffset = this.recoverFiles(startRecoverIndex, files, recoverOffset);
            this.fileManager.setFlushedPosition(recoverOffset);
            if (normalExit) {
                this.abortFile.create();
            } else {
                this.abortFile.touch();
            }
        }
        catch (Exception e) {
            LOG.error("Error on recover {} files , store path: {} , {}", this.getDBName(), this.storePath, e);
            throw new RuntimeException(e);
        }
        finally {
            this.startServiceManager();
        }
    }

    protected long recoverFiles(int startRecoverIndex, List<AbstractFile> files, long processOffset) {
        AbstractFile preFile = null;
        boolean needTruncate = false;
        for (int index = 0; index < files.size(); ++index) {
            boolean isLastFile;
            AbstractFile file = files.get(index);
            boolean bl = isLastFile = index == files.size() - 1;
            if (index < startRecoverIndex) {
                file.updateAllPosition(this.getDBFileSize());
            } else {
                AbstractFile.RecoverResult result = file.recover();
                if (result.recoverSuccess()) {
                    if (result.recoverTotal()) {
                        processOffset += isLastFile ? (long)result.getLastOffset() : (long)this.getDBFileSize();
                    } else {
                        processOffset += (long)result.getLastOffset();
                        needTruncate = true;
                    }
                } else {
                    needTruncate = true;
                }
            }
            if (preFile != null) {
                preFile.setLastLogIndex(file.getFirstLogIndex() - 1L);
            }
            preFile = file;
            if (!needTruncate) continue;
            LOG.warn("Try to truncate files to processOffset:{} when recover files", processOffset);
            this.fileManager.truncateSuffixByOffset(processOffset);
            break;
        }
        return processOffset;
    }

    private int findLastCheckpointFile(List<AbstractFile> files, FlushStatusCheckpoint checkpoint) {
        if (checkpoint == null || checkpoint.fileName == null) {
            return 0;
        }
        for (int fileIndex = 0; fileIndex < files.size(); ++fileIndex) {
            AbstractFile file = files.get(fileIndex);
            if (!AbstractDB.getFileName(file).equalsIgnoreCase(checkpoint.fileName)) continue;
            return fileIndex;
        }
        return 0;
    }

    private static String getFileName(AbstractFile file) {
        return Path.of(file.getFilePath(), new String[0]).getFileName().toString();
    }

    private void doCheckpoint() {
        long flushedPosition = this.getFlushedPosition();
        if (flushedPosition % (long)this.getDBFileSize() == 0L) {
            --flushedPosition;
        }
        AbstractFile file = this.fileManager.findFileByOffset(flushedPosition, false);
        try {
            if (file != null) {
                this.flushStatusCheckpoint.setFileName(AbstractDB.getFileName(file));
                this.flushStatusCheckpoint.setFlushPosition(flushedPosition);
                this.flushStatusCheckpoint.setLastLogIndex(this.getLastLogIndex());
                this.flushStatusCheckpoint.save();
            }
        }
        catch (IOException e) {
            LOG.error("Error when do checkpoint in db:{}", e, this.getDBName());
        }
    }

    public Pair<Integer, Long> appendLogAsync(long logIndex, byte[] data) {
        int waitToWroteSize = SegmentFile.getWriteBytes(data);
        SegmentFile segmentFile = (SegmentFile)this.fileManager.getLastFile(logIndex, waitToWroteSize, true);
        if (segmentFile != null) {
            int pos = segmentFile.appendData(logIndex, data);
            long expectFlushPosition = segmentFile.getFileFromOffset() + (long)pos + (long)waitToWroteSize;
            return Pair.of(pos, expectFlushPosition);
        }
        return Pair.of(-1, -1L);
    }

    public Pair<Integer, Long> appendLogAsync(long logIndex, LogEntryEncoder encoder, LogEntry logEntry) {
        V1Encoder v1Encoder = (V1Encoder)encoder;
        int dataSize = v1Encoder.size(logEntry);
        int waitToWroteSize = SegmentFile.getWriteBytes(dataSize);
        SegmentFile segmentFile = (SegmentFile)this.fileManager.getLastFile(logIndex, waitToWroteSize, true);
        if (segmentFile != null) {
            int pos = segmentFile.appendData(logIndex, v1Encoder, logEntry, dataSize);
            long expectFlushPosition = segmentFile.getFileFromOffset() + (long)pos + (long)waitToWroteSize;
            return Pair.of(pos, expectFlushPosition);
        }
        return Pair.of(-1, -1L);
    }

    public byte[] lookupLog(long logIndex, int pos) {
        long targetFlushPosition;
        SegmentFile segmentFile = (SegmentFile)this.fileManager.findFileByLogIndex(logIndex, false);
        if (segmentFile != null && (targetFlushPosition = segmentFile.getFileFromOffset() + (long)pos) <= this.getFlushedPosition()) {
            return segmentFile.lookupData(logIndex, pos);
        }
        return null;
    }

    public boolean waitForFlush(long maxExpectedFlushPosition, int maxFlushTimes) {
        int cnt = 0;
        while (this.getFlushedPosition() < maxExpectedFlushPosition) {
            this.flush();
            if (++cnt <= maxFlushTimes) continue;
            LOG.error("Try flush db {} times, but the flushPosition {} can't exceed expectedFlushPosition {}", maxFlushTimes, this.getFlushedPosition(), maxExpectedFlushPosition);
            return false;
        }
        return true;
    }

    public void startServiceManager() {
        this.serviceManager.start();
    }

    public boolean flush() {
        return this.fileManager.flush();
    }

    public boolean truncatePrefix(long firstIndexKept) {
        return this.fileManager.truncatePrefix(firstIndexKept);
    }

    public boolean truncateSuffix(long lastIndexKept, int pos) {
        if (this.fileManager.truncateSuffix(lastIndexKept, pos)) {
            this.doCheckpoint();
            return true;
        }
        return false;
    }

    public boolean reset(long nextLogIndex) {
        this.flushStatusCheckpoint.destroy();
        this.fileManager.reset(nextLogIndex);
        this.doCheckpoint();
        return true;
    }

    public long getFlushedPosition() {
        return this.fileManager.getFlushedPosition();
    }

    public StoreOptions getStoreOptions() {
        return this.storeOptions;
    }

    public String getStorePath() {
        return this.storePath;
    }

    public long getFirstLogIndex() {
        return this.fileManager.getFirstLogIndex();
    }

    public long getLastLogIndex() {
        return this.fileManager.getLastLogIndex();
    }

    public static class LogEntryIterator
    implements Iterator<LogEntry> {
        private final AbstractFile[] files;
        private int currentReadPos;
        private int preReadPos;
        private int currentFileId;
        private final LogEntryDecoder logEntryDecoder;

        public LogEntryIterator(AbstractFile[] files, LogEntryDecoder logEntryDecoder, int currentReadPos) {
            this.files = files;
            this.logEntryDecoder = logEntryDecoder;
            if (files.length > 0) {
                this.currentFileId = 0;
                this.currentReadPos = Math.max(currentReadPos, 26);
            } else {
                this.currentFileId = -1;
                this.currentReadPos = -1;
            }
        }

        @Override
        public boolean hasNext() {
            return this.currentFileId >= 0 && this.currentFileId < this.files.length;
        }

        @Override
        public LogEntry next() {
            byte[] data;
            if (this.currentFileId == -1) {
                return null;
            }
            while (true) {
                if (this.currentFileId >= this.files.length) {
                    return null;
                }
                SegmentFile segmentFile = (SegmentFile)this.files[this.currentFileId];
                if (segmentFile == null) {
                    return null;
                }
                data = segmentFile.lookupData(this.currentReadPos);
                if (data != null) break;
                ++this.currentFileId;
                this.currentReadPos = 26;
            }
            this.preReadPos = this.currentReadPos;
            this.currentReadPos += SegmentFile.getWriteBytes(data);
            return this.logEntryDecoder.decode(data);
        }

        public int getReadPosition() {
            return this.preReadPos;
        }
    }
}

